<?php
/* ------------------------------------------------------------------------------------
*  COPYRIGHT AND TRADEMARK NOTICE
*  Copyright 2008-2026 Arnan de Gans. All Rights Reserved.
*  The software AdRotate, related services and ADROTATE the name are owned by Arnan de Gans.

*  COPYRIGHT NOTICES AND ALL THE COMMENTS SHOULD REMAIN INTACT.
*  By using this code you agree to indemnify Arnan de Gans from any
*  liability that might arise from its use.
------------------------------------------------------------------------------------ */

/*-------------------------------------------------------------
 Name:      adrotate_rand
 Purpose:   Generate a random string
-------------------------------------------------------------*/
function adrotate_rand($length = 8) {
	$available_chars = 'abcdefghijklmnopqrstuvwxyz';

	$result = '';
	for($i = 0; $i < $length; $i++) {
		$result .= $available_chars[mt_rand(0, 25)];
	}

	return $result;
}

/*-------------------------------------------------------------
 Name:      adrotate_fetch_rss_feed
 Purpose:   Load RSS feeds to show in the AdRotate dashboard.
-------------------------------------------------------------*/
function adrotate_fetch_rss_feed() {
	$feeds = array("https://ajdg.solutions/feed/", "https://www.arnan.me/feed/page:feed.xml");
	$feed_items = array();
	
	foreach($feeds as $k => $feed) {
		// Get a feed
		$rss = fetch_feed($feed);
		if(!is_wp_error($rss)) {
			if($rss->get_item_quantity()) {
				foreach($rss->get_items(0, 3) as $item) {
					$date = trim($item->get_date('U'));
					$title = esc_html(trim(strip_tags($item->get_title())));	
					$link = esc_url(trim(strip_tags($item->get_link())));

					if(empty($title)) $title = __('Untitled');
					if(!empty($link) AND stristr($link, 'http') !== $link) $link = substr($link, 1);	
			
					$feed_items[$date]["title"] = $title;
					$feed_items[$date]["link"] = $link;
					
					unset($item, $link, $title, $date);
				}
			}
		}
		unset($rss, $k, $feed);
	}

	if(count($feed_items) > 0) {
		// Sort by key (timestamp)
		krsort($feed_items);
	
		// Prepare output
		$feed_output = '<ul>';
		foreach($feed_items as $date => $item) {
			$nice_date = ' <span class="rss-date">'.date_i18n(get_option('date_format'), $date).'</span>';

			$feed_output .= (empty($item["link"])) ? "<li>".$item["title"]."<br /><em>".$nice_date."</em></li>" : "<li><a class='rsswidget' href='".$item["link"]."'>".$item["title"]."</a><br /><em>".$nice_date."</em></li>";
			
			unset($date, $item, $nice_date);
		}
		$feed_output .= '</ul>';
	} else {
		$feed_output = '<p>Feeds could not be loaded, or there was nothing in it!</p>';
	}

	// Done!
	return $feed_output;
}

/*-------------------------------------------------------------
 Name:      adrotate_select_categories
 Purpose:   Create scrolling menu of all categories.
-------------------------------------------------------------*/
function adrotate_select_categories($savedcats, $count = 2, $child_of = 0, $parent = 0) {
	if(!is_array($savedcats)) $savedcats = explode(',', $savedcats);
	$categories = get_categories(array('child_of' => $parent, 'parent' => $parent,  'orderby' => 'id', 'order' => 'asc', 'hide_empty' => 0));

	if(!empty($categories)) {
		$output = "";
		if($parent == 0) {
			$output .= "<table width=\"100%\">";
			$output .= "<thead><tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" /></td><td style=\"padding: 0px;\">Select All</td></tr></thead>";
			$output .= "<tbody>";
		}
		foreach($categories as $category) {
			if($category->parent > 0) {
				if($category->parent != $child_of) {
					$count = $count + 1;
				}
				$indent = "&nbsp;".str_repeat('-', $count * 2)."&nbsp;";
			} else {
				$indent = "";
			}
			$output .= "<tr>";

			$output .= "<td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_categories[]\" value=\"".$category->cat_ID."\"";
			$output .= (in_array($category->cat_ID, $savedcats)) ? " checked" : "";
			$output .= "></td><td style=\"padding:0px;\">".$indent.$category->name." (".$category->category_count.")</td>";

			$output .= "</tr>";
			$output .= adrotate_select_categories($savedcats, $count, $category->parent, $category->cat_ID);
			$child_of = $parent;
		}
		if($parent == 0) {
			$output .= "</tbody></table>";
		}
		return $output;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_select_pages
 Purpose:   Create scrolling menu of all pages.
-------------------------------------------------------------*/
function adrotate_select_pages($savedpages, $count = 2, $child_of = 0, $parent = 0) {
	if(!is_array($savedpages)) $savedpages = explode(',', $savedpages);
	$pages = get_pages(array('child_of' => $parent, 'parent' => $parent, 'sort_column' => 'ID', 'sort_order' => 'asc'));

	if(!empty($pages)) {
		$output = "";
		if($parent == 0) {
			$output .= "<table width=\"100%\">";
			$output .= "<thead><tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" /></td><td style=\"padding:0px;\">Select All</td></tr></thead>";
			$output .= "<tbody>";
		}
		foreach($pages as $page) {
			if($page->post_parent > 0) {
				if($page->post_parent != $child_of) {
					$count = $count + 1;
				}
				$indent = "&nbsp;".str_repeat('-', $count * 2)."&nbsp;";
			} else {
				$indent = "";
			}
			$output .= "<tr>";
			$output .= "<td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_pages[]\" value=\"".$page->ID."\"";
			if(in_array($page->ID, $savedpages)) {
				$output .= " checked";
			}
			$output .= "></td><td style=\"padding: 0px;\">".$indent.$page->post_title."</td>";
			$output .= "</tr>";
			$output .= adrotate_select_pages($savedpages, $count, $page->post_parent, $page->ID);
			$child_of = $parent;
		}
		if($parent == 0) {
			$output .= "</tbody></table>";
		}
		return $output;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_select_woo_categories
 Purpose:   Create scrolling menu of all categories.
-------------------------------------------------------------*/
function adrotate_select_woo_categories($savedcats, $count = 2, $child_of = 0, $parent = 0) {
	if(!is_array($savedcats)) $savedcats = explode(',', $savedcats);
	$categories = get_categories(array('taxonomy' => 'product_cat', 'child_of' => $parent, 'parent' => $parent,  'orderby' => 'id', 'order' => 'asc', 'hide_empty' => 0));

	if(!empty($categories)) {
		$output = "";
		if($parent == 0) {
			$output .= "<table width=\"100%\">";
			$output .= "<thead><tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" /></td><td style=\"padding:0px;\">Select All</td></tr></thead>";
			$output .= "<tbody>";
		}
		foreach($categories as $category) {
			if($category->parent > 0) {
				if($category->parent != $child_of) {
					$count = $count + 1;
				}
				$indent = "&nbsp;".str_repeat('-', $count * 2)."&nbsp;";
			} else {
				$indent = "";
			}
			$output .= "<tr>";

			$output .= "<td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_woo_categories[]\" value=\"".$category->cat_ID."\"";
			$output .= (in_array($category->cat_ID, $savedcats)) ? " checked" : "";
			$output .= "></td><td style=\"padding: 0px;\">".$indent.$category->name." (".$category->category_count.")</td>";

			$output .= "</tr>";
			$output .= adrotate_select_woo_categories($savedcats, $count, $category->parent, $category->cat_ID);
			$child_of = $parent;
		}
		if($parent == 0) {
			$output .= "</tbody></table>";
		}
		return $output;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_select_bbpress_forums
 Purpose:   Create scrolling menu of all categories.
-------------------------------------------------------------*/
function adrotate_select_bbpress_forums($savedforums, $count = 2, $child_of = 0, $parent = 0) {
	if(!is_array($savedforums)) $savedforums = explode(',', $savedforums);
	$forums = get_pages(array('post_type' => 'forum', 'child_of' => $parent, 'parent' => $parent, 'sort_column' => 'ID', 'sort_order' => 'asc'));

	if(!empty($forums)) {
		$output = "";
		if($parent == 0) {
			$output .= "<table width=\"100%\">";
			$output .= "<thead><tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" /></td><td style=\"padding:0px;\">Select All</td></tr></thead>";
			$output .= "<tbody>";
		}
		foreach($forums as $forum) {
			if($forum->post_parent > 0) {
				if($forum->post_parent != $child_of) {
					$count = $count + 1;
				}
				$indent = "&nbsp;".str_repeat('-', $count * 2)."&nbsp;";
			} else {
				$indent = "";
			}
			$output .= "<tr>";
			$output .= "<td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_bbpress_forums[]\" value=\"".$forum->ID."\"";
			if(in_array($forum->ID, $savedforums)) {
				$output .= " checked";
			}
			$output .= "></td><td style=\"padding:0px;\">".$indent.$forum->post_title."</td>";
			$output .= "</tr>";
			$output .= adrotate_select_bbpress_forums($savedforums, $count, $forum->post_parent, $forum->ID);
			$child_of = $parent;
		}
		if($parent == 0) {
			$output .= "</tbody></table>";
		}
		return $output;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_countries
 Purpose:   List of countries
-------------------------------------------------------------*/
function adrotate_countries() {
	return array(
		// Europe
		'EUROPE' => "Europe",
		'AL' => "Albania",
		'AM' => "Armenia",
		'AD' => "Andorra",
		'AT' => "Austria",
		'AZ' => "Azerbaijan",
		'BY' => "Belarus",
		'BE' => "Belgium",
		'BA' => "Bosnia and Herzegovina",
		'BG' => "Bulgaria",
		'HR' => "Croatia",
		'CY' => "Cyprus",
		'CZ' => "Czech Republic",
		'DK' => "Denmark",
		'EE' => "Estonia",
		'FI' => "Finland",
		'FR' => "France",
		'GE' => "Georgia",
		'DE' => "Germany",
		'GR' => "Greece",
		'HU' => "Hungary",
		'IS' => "Iceland",
		'IE' => "Ireland",
		'IT' => "Italy",
		'LV' => "Latvia",
		'LI' => "Liechtenstein",
		'LT' => "Lithuania",
		'LU' => "Luxembourg",
		'MK' => "Macedonia",
		'MT' => "Malta",
		'MD' => "Moldova",
		'MC' => "Monaco",
		'ME' => "Montenegro",
		'NL' => "the Netherlands",
		'NO' => "Norway",
		'PL' => "Poland",
		'PT' => "Portugal",
		'RO' => "Romania",
		'SM' => "San Marino",
		'RS' => "Serbia",
		'ES' => "Spain",
		'SK' => "Slovakia",
		'SI' => "Slovenia",
		'SE' => "Sweden",
		'CH' => "Switzerland",
		'VA' => "Vatican City",
		'TR' => "Turkey",
		'UA' => "Ukraine",
		'GB' => "United Kingdom",

		// North America
		'NORTHAMERICA' => "North America",
		'AG' => "Antigua and Barbuda",
		'BS' => "Bahamas",
		'BB' => "Barbados",
		'BZ' => "Belize",
		'CA' => "Canada",
		'CR' => "Costa Rica",
		'CU' => "Cuba",
		'DM' => "Dominica",
		'DO' => "Dominican Republic",
		'SV' => "El Salvador",
		'GD' => "Grenada",
		'GT' => "Guatemala",
		'HT' => "Haiti",
		'HN' => "Honduras",
		'JM' => "Jamaica",
		'MX' => "Mexico",
		'NI' => "Nicaragua",
		'PA' => "Panama",
		'KN' => "Saint Kitts and Nevis",
		'LC' => "Saint Lucia",
		'VC' => "Saint Vincent",
		'TT' => "Trinidad and Tobago",
		'US' => "United States",

		// South America
		'SOUTHAMERICA' => "South America",
		'AR' => "Argentina",
		'BO' => "Bolivia",
		'BR' => "Brazil",
		'CL' => "Chile",
		'CO' => "Colombia",
		'EC' => "Ecuador",
		'GY' => "Guyana",
		'PY' => "Paraguay",
		'PE' => "Peru",
		'SR' => "Suriname",
		'UY' => "Uruguay",
		'VE' => "Venezuela",

		// South East Asia + Australia + New Zealand
		'SOUTHEASTASIA' => "Southeast Asia, Australia and New Zealand",
		'AU' => "Australia",
		'BN' => "Brunei",
		'KH' => "Cambodia",
		'TL' => "East Timor (Timor Timur)",
		'ID' => "Indonesia",
		'LA' => "Laos",
		'MY' => "Malaysia",
		'MM' => "Myanmar",
		'NZ' => "New Zealand",
		'PH' => "Philippines",
		'SG' => "Singapore",
		'TH' => "Thailand",
		'VN' => "Vietnam",

		// Misc
		'MISC' => "Rest of the world",
		'AF' => "Afghanistan",
		'DZ' => "Algeria",
		'AO' => "Angola",
		'BH' => "Bahrain",
		'BD' => "Bangladesh",
		'BJ' => "Benin",
		'BT' => "Bhutan",
		'BF' => "Burkina Faso",
		'BI' => "Burundi",
		'CM' => "Cameroon",
		'CV' => "Cape Verde",
		'CF' => "Central African Republic",
		'TD' => "Chad",
		'CN' => "China",
		'KM' => "Comoros",
		'CG' => "Congo (Brazzaville)",
		'CD' => "Congo",
		'CI' => "Cote d'Ivoire",
		'DJ' => "Djibouti",
		'EG' => "Egypt",
		'GQ' => "Equatorial Guinea",
		'ER' => "Eritrea",
		'ET' => "Ethiopia",
		'FJ' => "Fiji",
		'GA' => "Gabon",
		'GM' => "Gambia",
		'GH' => "Ghana",
		'GN' => "Guinea",
		'GW' => "Guinea-Bissau",
		'IN' => "India",
		'IR' => "Iran",
		'IQ' => "Iraq",
		'IL' => "Israel",
		'JP' => "Japan",
		'JO' => "Jordan",
		'KZ' => "Kazakhstan",
		'KE' => "Kenya",
		'KI' => "Kiribati",
		'KP' => "north Korea",
		'KR' => "south Korea",
		'KW' => "Kuwait",
		'KG' => "Kyrgyzstan",
		'LV' => "Latvia",
		'LB' => "Lebanon",
		'LS' => "Lesotho",
		'LR' => "Liberia",
		'LY' => "Libya",
		'MG' => "Madagascar",
		'MW' => "Malawi",
		'MV' => "Maldives",
		'MN' => "Mongolia",
		'ML' => "Mali",
		'MH' => "Marshall Islands",
		'MR' => "Mauritania",
		'MU' => "Mauritius",
		'FM' => "Micronesia",
		'MA' => "Morocco",
		'MZ' => "Mozambique",
		'NA' => "Namibia",
		'NR' => "Nauru",
		'NP' => "Nepal",
		'NE' => "Niger",
		'NG' => "Nigeria",
		'OM' => "Oman",
		'PK' => "Pakistan",
		'PW' => "Palau",
		'PG' => "Papua New Guinea",
		'QA' => "Qatar",
		'RU' => "Russia",
		'RW' => "Rwanda",
		'WS' => "Samoa",
		'ST' => "Sao Tome and Principe",
		'SA' => "Saudi Arabia",
		'SN' => "Senegal",
		'SC' => "Seychelles",
		'SL' => "Sierra Leone",
		'SB' => "Solomon Islands",
		'SO' => "Somalia",
		'ZA' => "South Africa",
		'LK' => "Sri Lanka",
		'SY' => "Syria",
		'SD' => "Sudan",
		'SZ' => "Swaziland",
		'TW' => "Taiwan",
		'TJ' => "Tajikistan",
		'TO' => "Tonga",
		'TM' => "Turkmenistan",
		'TV' => "Tuvalu",
		'TZ' => "Tanzania",
		'TG' => "Togo",
		'TN' => "Tunisia",
		'UG' => "Uganda",
		'AE' => "United Arab Emirates",
		'UZ' => "Uzbekistan",
		'VU' => "Vanuatu",
		'YE' => "Yemen",
		'ZM' => "Zambia",
		'ZW' => "Zimbabwe",

		// Misc
		'SPEC' => "Special Territories and Regions",
		'AW' => "Aruba",
		'KY' => "Cayman Islands",
		'CW' => "Curacao",
		'AW' => "Guam",
		'HK' => "Hong Kong",
		'MO' => "Macao",
		'MQ' => "Martinique",
		'PR' => "Puerto Rico",
		'AW' => "Sint Maarten",
		'VA' => "Holy See (Vatican City)"
	);
}

/*-------------------------------------------------------------
 Name:      adrotate_select_countries
 Purpose:   Create scrolling menu of all countries.
-------------------------------------------------------------*/
function adrotate_select_countries($savedcountries) {
	if(!is_array($savedcountries)) $savedcountries = array();
	$countries = adrotate_countries();

	$output = "<table width=\"100%\">";
	$output .= "<thead>";
	$output .= "<tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" /></td><td style=\"padding:0px;\">(De)select all</td></tr>";
	$output .= "</thead>";

	$output .= "<tbody>";
	$output .= "<tr><td colspan=\"2\" style=\"padding:0px;\"><em>--- Regions ---</em></td></tr>";
	$output .= "<tr><td class=\"check-column\" style=\"padding: 0px;\"><input type=\"checkbox\" name=\"adrotate_geo_westeurope\" value=\"1\" /></td><td style=\"padding:0px;\">West/Central Europe</td></tr>";
	$output .= "<tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_geo_easteurope\" value=\"1\" /></td><td style=\"padding:0px;\">East/Central Europe</td></tr>";
	$output .= "<tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_geo_northamerica\" value=\"1\" /></td><td style=\"padding:0px;\">North America</td></tr>";
	$output .= "<tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_geo_southamerica\" value=\"1\" /></td><td style=\"padding:0px;\">South America</td></tr>";
	$output .= "<tr><td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_geo_southeastasia\" value=\"1\" /></td><td style=\"padding:0px;\">Southeast Asia, Australia and New Zealand</td></tr>";
	foreach($countries as $k => $v) {
		$output .= "<tr>";
		if(strlen($k) > 2) {
			$output .= "<td colspan=\"2\" style=\"padding:0px;\"><em>--- ".$v." ---</em></td>";
		} else {
			$output .= "<td class=\"check-column\" style=\"padding:0px;\"><input type=\"checkbox\" name=\"adrotate_geo_countries[]\"  value=\"".$k."\"";
			$output .= (in_array($k, $savedcountries)) ? " checked" : "";
			$output .= "></td><td style=\"padding:0px;\">".$v."</td>";
		}
		$output .= "</tr>";
	}
	$output .= "</tbody></table>";
	return $output;
}

/*-------------------------------------------------------------
 Name:      adrotate_evaluate_ads
 Purpose:   Initiate evaluations for errors and determine the ad status
-------------------------------------------------------------*/
function adrotate_evaluate_ads() {
	global $wpdb;

	// Fetch ads
	$ads = $wpdb->get_results("SELECT `id` FROM `{$wpdb->prefix}adrotate` WHERE `type` != 'disabled' AND `type` != 'generator' AND `type` != 'a_empty' AND `type` != 'a_error' AND `type` != 'queue' AND `type` != 'reject' AND `type` != 'archived' AND `type` != 'trash' AND `type` != 'empty' ORDER BY `id` ASC;");

	// Determine error states
	$error = $limit = $expired = $expiressoon = $expiresweek = $normal = $unknown = 0;
	foreach($ads as $ad) {
		$result = adrotate_evaluate_ad($ad->id);
		if($result == 'error') {
			$error++;
			$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = 'error' WHERE `id` = {$ad->id};");
		}

		if($result == 'limit') {
			$limit++;
			$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = 'limit' WHERE `id` = {$ad->id};");
		}

		if($result == 'expired') {
			$expired++;
			$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = 'expired' WHERE `id` = {$ad->id};");
		}

		if($result == '2days') {
			$expiressoon++;
			$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = '2days' WHERE `id` = {$ad->id};");
		}

		if($result == '7days') {
			$expiresweek++;
			$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = '7days' WHERE `id` = {$ad->id};");
		}

		if($result == 'active') {
			$normal++;
			$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = 'active' WHERE `id` = {$ad->id};");
		}

		if($result == 'unknown') {
			$unknown++;
		}
		unset($ad);
	}

	// Delete unfinished ads
	$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate` WHERE `type` = 'empty' OR `type` = 'a_empty' OR `type` = 'generator';");

	$result = array('error' => $error, 'limit' => $limit, 'expired' => $expired, 'expiressoon' => $expiressoon, 'expiresweek' => $expiresweek, 'normal' => $normal, 'unknown' => $unknown);
	update_option('adrotate_advert_status', $result);
	unset($ads, $result);
}

/*-------------------------------------------------------------
 Name:      adrotate_evaluate_ad
 Purpose:   Evaluates ads for errors
-------------------------------------------------------------*/
function adrotate_evaluate_ad($ad_id) {
	global $wpdb, $adrotate_config;

	$now = current_time('timestamp');
	$in2days = $now + 172800;
	$in7days = $now + 604800;

	// Fetch ad and its data
	$ad = $wpdb->get_row($wpdb->prepare("SELECT `id`, `bannercode`, `imagetype`, `image`, `tracker`, `desktop`, `mobile`, `tablet`, `budget`, `os_ios`, `os_android`,`crate`, `irate`, `state_req`, `cities`, `states` FROM `{$wpdb->prefix}adrotate` WHERE `id` = %d;", $ad_id));

	$stoptime = $wpdb->get_var("SELECT `stoptime` FROM `{$wpdb->prefix}adrotate_schedule`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = {$ad->id} AND `schedule` = `{$wpdb->prefix}adrotate_schedule`.`id` ORDER BY `stoptime` DESC LIMIT 1;");
	$has_groups = $wpdb->get_var("SELECT COUNT(`group`) FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = {$ad->id} AND `schedule` = 0 AND `user` = 0;");
	$has_schedules = $wpdb->get_var("SELECT COUNT(`schedule`) FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = {$ad->id} AND `group` = 0 AND `user` = 0;");
	$has_advertiser = $wpdb->get_var("SELECT `user` FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = {$ad->id} AND `group` = 0 AND `user` > 0 AND `schedule` = 0;");

	$bannercode = stripslashes(htmlspecialchars_decode($ad->bannercode, ENT_QUOTES));
	$cities = unserialize($ad->cities);
	$states = unserialize($ad->states);

	// Limits exceeded?
	$temp_clicks = $temp_impressions = $temp_max_clicks = $temp_max_impressions = 0;
	$cachekey = 'adrotate_schedule_'.$ad_id;
	$schedules = wp_cache_get($cachekey);
	if(false === $schedules) {
		$schedules = $wpdb->get_results("SELECT `{$wpdb->prefix}adrotate_schedule`.* FROM `{$wpdb->prefix}adrotate_schedule`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `schedule` = `{$wpdb->prefix}adrotate_schedule`.`id` AND `ad` = {$ad->id} ORDER BY `starttime` ASC;");
		wp_cache_set($cachekey, $schedules, '', 300);
	}

	foreach($schedules as $schedule) {
		if($now >= $schedule->starttime AND $now <= $schedule->stoptime) {
			$stat = adrotate_stats($ad_id, false, $schedule->starttime, $schedule->stoptime);
			if(!is_array($stat)) $stat = array('clicks' => 0, 'impressions' => 0);

			if($stat['clicks'] > $temp_clicks) $temp_clicks = $stat['clicks'];
			if($stat['impressions'] > $temp_impressions) $temp_impressions = $stat['impressions'];
			$temp_max_clicks = $schedule->maxclicks;
			$temp_max_impressions = $schedule->maximpressions;

			unset($stat);
		}
		unset($schedule);
	}

	// Determine error states
	if(
		strlen($bannercode) < 1 // AdCode empty
		OR ($ad->tracker == 'N' AND $has_advertiser > 0) // Didn't enable stats, DID set a advertiser
		OR (preg_match_all('/(%asset%)/i', $bannercode, $things) AND $ad->image == '' AND $ad->imagetype == '') // Did use %asset% but didn't select an image
		OR (!preg_match_all('/(%asset%)/i', $bannercode, $things) AND $ad->image != '' AND $ad->imagetype != '') // Didn't use %asset% but selected an image
		OR (($ad->image == '' AND $ad->imagetype != '') OR ($ad->image != '' AND $ad->imagetype == '')) // Image and Imagetype mismatch
		OR ($has_advertiser > 0 AND ($ad->crate > 0 OR $ad->irate > 0) AND $ad->budget < 1) // Has advertiser but ran out of budget
		OR $has_schedules == 0 // No Schedules for this ad
		OR ((!preg_match_all('/<(a)[^>](.*?)>/i', $bannercode, $things) OR preg_match_all('/<(ins|script|embed|iframe)[^>](.*?)>/i', $bannercode, $things)) AND ($ad->tracker == 'Y' OR $ad->tracker == 'C')) // Clicks active but no valid link/tag present
		OR ($ad->tracker == 'N' AND ($ad->crate > 0 OR $ad->irate > 0))	// Stats inactive but set a Click|Impression rate
		OR ($has_groups > 0 AND ($ad->desktop == 'N' AND $ad->mobile == 'N' AND $ad->tablet == 'N')) // Has group but no devices selected
		OR ($has_groups > 0 AND ($ad->mobile == 'Y' OR $ad->tablet == 'Y') AND $ad->os_ios == 'N' AND $ad->os_android == 'N') // Wants mobile device, but has no OS selected
		OR ($ad->state_req == 'Y' AND count($cities) == 0 AND count($states) == 0) // Geo Targeting, state requried but no cities and states set
		OR ($ad->state_req == 'Y' AND count($cities) > 0 AND count($states) == 0) // Geo Targeting, state requried but has no states
		OR ($ad->state_req == 'Y' AND count($cities) == 0 AND count($states) > 0) // Geo Targeting, state requried but has no cities
	) {
		return 'error';
	} else if(
		($temp_clicks > $temp_max_clicks AND $temp_max_clicks > 0) // Click limit reached?
		OR ($temp_impressions > $temp_max_impressions AND $temp_max_impressions > 0) // Impression limit reached?
		OR (($ad->crate > 0 OR $ad->irate > 0) AND $ad->budget <= 0) // Ad ran out of money
	){
		unset($temp_clicks, $temp_impressions, $temp_max_clicks, $temp_max_impressions);

		return 'limit';
	} else if(
		$stoptime <= $now // Past the enddate
	){
		return 'expired';
	} else if(
		$stoptime <= $in2days AND $stoptime >= $now	// Expires in 2 days
	){
		return '2days';
	} else if(
		$stoptime <= $in7days AND $stoptime >= $now	// Expires in 7 days
	){
		return '7days';
	} else {
		return 'active';
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_prepare_color
 Purpose:   Check if ads are expired and set a color for its end date
-------------------------------------------------------------*/
function adrotate_prepare_color($enddate) {
	$now = current_time('timestamp');
	$in2days = $now + 172800;
	$in7days = $now + 604800;

	if($enddate <= $now) {
		return '#CC2900'; // red
	} else if($enddate <= $in2days AND $enddate >= $now) {
		return '#F90'; // orange
	} else if($enddate <= $in7days AND $enddate >= $now) {
		return '#E6B800'; // yellow
	} else {
		return '#009900'; // green
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_ad_is_in_groups
 Purpose:   Build list of groups the ad is in (overview)
-------------------------------------------------------------*/
function adrotate_ad_is_in_groups($id) {
	global $wpdb;

	$output = '';
	$groups	= $wpdb->get_results("
		SELECT
			`{$wpdb->prefix}adrotate_groups`.`name`
		FROM
			`{$wpdb->prefix}adrotate_groups`,
			`{$wpdb->prefix}adrotate_linkmeta`
		WHERE
			`{$wpdb->prefix}adrotate_linkmeta`.`ad` = {$id}
			AND `{$wpdb->prefix}adrotate_linkmeta`.`group` = `{$wpdb->prefix}adrotate_groups`.`id`
			AND `{$wpdb->prefix}adrotate_linkmeta`.`user` = 0
		;");
	if($groups) {
		foreach($groups as $group) {
			$output .= $group->name.', ';
		}
	}
	$output = rtrim($output, ', ');

	return $output;
}

/*-------------------------------------------------------------
 Name:      adrotate_get_sorted_roles
 Purpose:   Returns all roles and capabilities, sorted by user level. Lowest to highest.
-------------------------------------------------------------*/
function adrotate_get_sorted_roles() {
	global $wp_roles;

	$editable_roles = apply_filters('editable_roles', $wp_roles->roles);
	$sorted = array();

	foreach($editable_roles as $role => $details) {
		$sorted[$details['name']] = get_role($role);
	}

	$sorted = array_reverse($sorted);

	return $sorted;
}

/*-------------------------------------------------------------
 Name:      adrotate_set_capability
 Purpose:   Grant or revoke capabilities to a role and all higher roles
-------------------------------------------------------------*/
function adrotate_set_capability($lowest_role, $capability){
	$check_order = adrotate_get_sorted_roles();
	$add_capability = false;

	foreach($check_order as $role) {
		if($lowest_role == $role->name) $add_capability = true;
		if(empty($role)) continue;

		$add_capability ? $role->add_cap($capability) : $role->remove_cap($capability) ;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_remove_capability
 Purpose:   Remove the $capability from the all roles
-------------------------------------------------------------*/
function adrotate_remove_capability($capability){
	$check_order = adrotate_get_sorted_roles();

	foreach($check_order as $role) {
		$role = get_role($role->name);
		$role->remove_cap($capability);
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_mail_advertiser
 Purpose:   Email a selected advertiser about his account/adverts/whatever
-------------------------------------------------------------*/
function adrotate_mail_advertiser() {
	global $wpdb;

	if(wp_verify_nonce($_POST['adrotate_nonce'], 'adrotate_email_advertiser')) {
		$author = $_POST['adrotate_username'];
		$useremail = $_POST['adrotate_email'];
		$subject = strip_tags(stripslashes(trim($_POST['adrotate_subject'], '\t\n ')));
		$advert_id	= trim($_POST['adrotate_advert'], '\t\n ');
		$text = strip_tags(stripslashes(trim($_POST['adrotate_message'], '\t\n ')));

		$advert = $wpdb->get_row("SELECT `id`, `title`, `type` FROM `{$wpdb->prefix}adrotate` WHERE `id` = {$advert_id};");

		if(strlen($subject) < 1) $subject = "Publisher notification";
		if(strlen($text) < 1) $text = "No message given";

		$sitename = strtolower($_SERVER['SERVER_NAME']);
        if(substr($sitename, 0, 4) == 'www.') $sitename = substr($sitename, 4);

		$siteurl = get_option('siteurl');
		$adurl = $siteurl.'/wp-admin/admin.php?page=adrotate-advertiser&view=edit&ad='.$advert->id;

	    $headers = "Content-Type: text/html; charset=UTF-8\r\n"."From: AdRotate Pro <wordpress@$sitename>\r\n";

		$subject = esc_html__("[AdRotate]", 'adrotate-pro').' '.$subject;

		$message = "<p>Hello $author,</p>";
		if($advert->id > 0) $message .= "<p>About: ".$advert->id." - ".$advert->title." (".$advert->type.")</p>";
		$message .= "<p>$text</p>";
		$message .= "<p>".esc_html__("You can reply to this message by clicking reply in your email client.", 'adrotate-pro')."</p>";
		$message .= "<p>".esc_html__("Have a nice day!", 'adrotate-pro')."<br />";
		$message .= esc_html__("Your AdRotate Notifier", 'adrotate-pro')."</p>";

		wp_mail($useremail, $subject, $message, $headers);

		wp_redirect('admin.php?page=adrotate-advertisers&status=223');
		exit;
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_icon
 Purpose:   Return SVG images
-------------------------------------------------------------*/
function adrotate_icon($icon) {
	$icons = array(
		'tick' => 'data:image/svg+xml;base64,'.base64_encode('<svg height="10" width="10" xmlns="http://www.w3.org/2000/svg"><circle r="4" cx="5" cy="5" fill="#94ebb9" stroke="#3eda82" stroke-width="1" /></svg>'),
		'cross' => 'data:image/svg+xml;base64,'.base64_encode('<svg height="10" width="10" xmlns="http://www.w3.org/2000/svg"><circle r="4" cx="5" cy="5" fill="#ffbdb3" stroke="#ff644d" stroke-width="1" /></svg>'),
		'desktop' => 'data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiA/PjwhRE9DVFlQRSBzdmcgIFBVQkxJQyAnLS8vVzNDLy9EVEQgU1ZHIDEuMS8vRU4nICAnaHR0cDovL3d3dy53My5vcmcvR3JhcGhpY3MvU1ZHLzEuMS9EVEQvc3ZnMTEuZHRkJz48c3ZnIGVuYWJsZS1iYWNrZ3JvdW5kPSJuZXcgMCAwIDQ4IDQ4IiBoZWlnaHQ9IjQ4cHgiIHZlcnNpb249IjEuMSIgdmlld0JveD0iMCAwIDQ4IDQ4IiB3aWR0aD0iNDhweCIgeG1sOnNwYWNlPSJwcmVzZXJ2ZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayI+PGcgaWQ9IkV4cGFuZGVkIj48Zz48Zz48cGF0aCBkPSJNNDQsNDBINGMtMi4yMDYsMC00LTEuNzk0LTQtNFY2YzAtMi4yMDYsMS43OTQtNCw0LTRoNDBjMi4yMDYsMCw0LDEuNzk0LDQsNHYzMEM0OCwzOC4yMDYsNDYuMjA2LDQwLDQ0LDQweiBNNCw0ICAgICBDMi44OTcsNCwyLDQuODk3LDIsNnYzMGMwLDEuMTAzLDAuODk3LDIsMiwyaDQwYzEuMTAzLDAsMi0wLjg5NywyLTJWNmMwLTEuMTAzLTAuODk3LTItMi0ySDR6Ii8+PC9nPjxnPjxwYXRoIGQ9Ik0zOCw0NkgxMGMtMC41NTIsMC0xLTAuNDQ4LTEtMXMwLjQ0OC0xLDEtMWgyOGMwLjU1MiwwLDEsMC40NDgsMSwxUzM4LjU1Miw0NiwzOCw0NnoiLz48L2c+PGc+PHBhdGggZD0iTTMxLjAyLDQ2Yy0wLjAwNywwLTAuMDEzLDAtMC4wMiwwSDE3Yy0wLjMyMSwwLTAuNjIzLTAuMTU0LTAuODExLTAuNDE2Yy0wLjE4OC0wLjI2MS0wLjIzOS0wLjU5Ni0wLjEzOC0wLjkwMWwyLTYgICAgIEMxOC4xODgsMzguMjc1LDE4LjU2OSwzOCwxOSwzOGgxMGMwLjQzMSwwLDAuODEyLDAuMjc1LDAuOTQ5LDAuNjg0bDEuOTQsNS44MjFDMzEuOTcyLDQ0LjY1MSwzMi4wMiw0NC44MiwzMi4wMiw0NSAgICAgQzMyLjAyLDQ1LjU1MiwzMS41NzIsNDYsMzEuMDIsNDZ6IE0xOC4zODcsNDRoMTEuMjI2bC0xLjMzMy00aC04LjU1OUwxOC4zODcsNDR6Ii8+PC9nPjxnPjxwYXRoIGQ9Ik00NCwzMkg0VjZoNDBWMzJ6IE02LDMwaDM2VjhINlYzMHoiLz48L2c+PGc+PGNpcmNsZSBjeD0iMjQiIGN5PSIzNSIgcj0iMiIvPjwvZz48L2c+PC9nPjwvc3ZnPg==',
		'mobile' => 'data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiA/PjwhRE9DVFlQRSBzdmcgIFBVQkxJQyAnLS8vVzNDLy9EVEQgU1ZHIDEuMS8vRU4nICAnaHR0cDovL3d3dy53My5vcmcvR3JhcGhpY3MvU1ZHLzEuMS9EVEQvc3ZnMTEuZHRkJz48c3ZnIGVuYWJsZS1iYWNrZ3JvdW5kPSJuZXcgMCAwIDQ4IDQ4IiBoZWlnaHQ9IjQ4cHgiIHZlcnNpb249IjEuMSIgdmlld0JveD0iMCAwIDQ4IDQ4IiB3aWR0aD0iNDhweCIgeG1sOnNwYWNlPSJwcmVzZXJ2ZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayI+PGcgaWQ9IkV4cGFuZGVkIj48Zz48Zz48cGF0aCBkPSJNMzUsNDhIMTNjLTIuNzU3LDAtNS0yLjI0My01LTVWNWMwLTIuNzU3LDIuMjQzLTUsNS01aDIyYzIuNzU3LDAsNSwyLjI0Myw1LDV2MzhDNDAsNDUuNzU3LDM3Ljc1Nyw0OCwzNSw0OHogTTEzLDIgICAgIGMtMS42NTQsMC0zLDEuMzQ2LTMsM3YzOGMwLDEuNjU0LDEuMzQ2LDMsMywzaDIyYzEuNjU0LDAsMy0xLjM0NiwzLTNWNWMwLTEuNjU0LTEuMzQ2LTMtMy0zSDEzeiIvPjwvZz48Zz48cGF0aCBkPSJNMzksMTBIOWMtMC41NTMsMC0xLTAuNDQ4LTEtMXMwLjQ0Ny0xLDEtMWgzMGMwLjU1MywwLDEsMC40NDgsMSwxUzM5LjU1MywxMCwzOSwxMHoiLz48L2c+PGc+PHBhdGggZD0iTTM5LDQwSDljLTAuNTUzLDAtMS0wLjQ0OC0xLTFzMC40NDctMSwxLTFoMzBjMC41NTMsMCwxLDAuNDQ4LDEsMVMzOS41NTMsNDAsMzksNDB6Ii8+PC9nPjxnPjxwYXRoIGQ9Ik0yNCw0MWMtMS4xMDQsMC0yLDAuODk2LTIsMnMwLjg5NiwyLDIsMnMyLTAuODk2LDItMlMyNS4xMDQsNDEsMjQsNDFMMjQsNDF6Ii8+PC9nPjxnPjxwYXRoIGQ9Ik0yOSw2SDE5Yy0wLjU1MywwLTEtMC40NDgtMS0xczAuNDQ3LTEsMS0xaDEwYzAuNTUzLDAsMSwwLjQ0OCwxLDFTMjkuNTUzLDYsMjksNnoiLz48L2c+PC9nPjwvZz48L3N2Zz4=',
		'tablet' => 'data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiA/PjwhRE9DVFlQRSBzdmcgIFBVQkxJQyAnLS8vVzNDLy9EVEQgU1ZHIDEuMS8vRU4nICAnaHR0cDovL3d3dy53My5vcmcvR3JhcGhpY3MvU1ZHLzEuMS9EVEQvc3ZnMTEuZHRkJz48c3ZnIGVuYWJsZS1iYWNrZ3JvdW5kPSJuZXcgMCAwIDQ4IDQ4IiBoZWlnaHQ9IjQ4cHgiIHZlcnNpb249IjEuMSIgdmlld0JveD0iMCAwIDQ4IDQ4IiB3aWR0aD0iNDhweCIgeG1sOnNwYWNlPSJwcmVzZXJ2ZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayI+PGcgaWQ9IkV4cGFuZGVkIj48Zz48Zz48cGF0aCBkPSJNMzksNDhIOWMtMi43NTcsMC01LTIuMjQzLTUtNVY1YzAtMi43NTcsMi4yNDMtNSw1LTVoMzBjMi43NTcsMCw1LDIuMjQzLDUsNXYzOEM0NCw0NS43NTcsNDEuNzU3LDQ4LDM5LDQ4eiBNOSwyICAgICBDNy4zNDYsMiw2LDMuMzQ2LDYsNXYzOGMwLDEuNjU0LDEuMzQ2LDMsMywzaDMwYzEuNjU0LDAsMy0xLjM0NiwzLTNWNWMwLTEuNjU0LTEuMzQ2LTMtMy0zSDl6Ii8+PC9nPjxnPjxwYXRoIGQ9Ik0yNC4xNjcsNDFjLTEuMTA1LDAtMiwwLjg5NS0yLDJzMC44OTUsMiwyLDJjMS4xMDQsMCwyLTAuODk1LDItMlMyNS4yNzEsNDEsMjQuMTY3LDQxTDI0LjE2Nyw0MXoiLz48L2c+PGc+PHBhdGggZD0iTTM5LDQwSDljLTAuNTUyLDAtMS0wLjQ0OC0xLTFWN2MwLTAuNTUyLDAuNDQ4LTEsMS0xaDMwYzAuNTUyLDAsMSwwLjQ0OCwxLDF2MzJDNDAsMzkuNTUyLDM5LjU1Miw0MCwzOSw0MHogTTEwLDM4aDI4VjggICAgIEgxMFYzOHoiLz48L2c+PGc+PHBhdGggZD0iTTI0LjE2NywzYy0wLjU1MywwLTEsMC40NDctMSwxczAuNDQ3LDEsMSwxYzAuNTUzLDAsMS0wLjQ0NywxLTFTMjQuNzE5LDMsMjQuMTY3LDNMMjQuMTY3LDN6Ii8+PC9nPjwvZz48L2c+PC9zdmc+',
	);

	return $icons[$icon];
}

/*-------------------------------------------------------------
 Name:      adrotate_dashboard_scripts
 Purpose:   Load all dashboard stuff
-------------------------------------------------------------*/
function adrotate_dashboard_scripts() {
	$page = (isset($_GET['page'])) ? $_GET['page'] : '';

    if(strpos($page, 'adrotate') !== false) {
		wp_enqueue_style('jquery-ui', plugins_url('library/datepicker.css', __FILE__));

		wp_enqueue_script('jquery');
		wp_enqueue_script('jquery-ui-datepicker');
		wp_enqueue_script('raphael', plugins_url('/library/raphael-min.js', __FILE__), array('jquery'), ADROTATE_VERSION);
		wp_enqueue_script('elycharts', plugins_url('/library/elycharts.min.js', __FILE__), array('jquery', 'raphael'), ADROTATE_VERSION);
		wp_enqueue_script('textatcursor', plugins_url('/library/textatcursor.js', __FILE__), ADROTATE_VERSION);
		wp_enqueue_script('tablesorter', plugins_url('/library/jquery.tablesorter.min.js', __FILE__), array('jquery'), ADROTATE_VERSION);
		wp_enqueue_script('adrotate-tablesorter', plugins_url('/library/jquery.tablesorter-config.js', __FILE__), array('tablesorter'), ADROTATE_VERSION);
		wp_enqueue_script('adrotate-datepicker', plugins_url('/library/jquery.datepicker.js', __FILE__), array('jquery'), ADROTATE_VERSION);
	}

	// Keep global for notifications
	wp_enqueue_style('adrotate-admin-stylesheet', plugins_url('library/dashboard.css', __FILE__));
}

/*-------------------------------------------------------------
 Name:      adrotate_dashboard_error
 Purpose:   Show errors for problems in using AdRotate
-------------------------------------------------------------*/
function adrotate_dashboard_error() {
	global $wpdb, $adrotate_config;

	$oneyear = current_time('timestamp') - (DAY_IN_SECONDS * 365);

	// License
	$a = get_option('ajdg_activate_adrotate-pro', array());
	if($a AND $a['status'] == 0) {
		$error['adrotate_license'] = esc_html__("You did not yet activate your AdRotate Professional license!", 'adrotate-pro'). " <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=license')."\">".esc_html__("Activate license", 'adrotate-pro')."</a>.";
	}

	// Adverts
	$status = get_option('adrotate_advert_status');
	$adrotate_notifications	= get_option('adrotate_notifications');
	if($adrotate_notifications['notification_dash'] == 'Y') {
		if($status['expired'] > 0 AND $adrotate_notifications['notification_dash_expired'] == 'Y') {
			$error['advert_expired'] = sprintf(_n("%s advert is expired.", "%s adverts expired!", $status['expired'], 'adrotate-pro'), $status['expired'])." <a href=\"".admin_url('admin.php?page=adrotate')."\">".esc_html__("Check adverts", 'adrotate-pro')."</a>.";
		}
		if($status['expiressoon'] > 0 AND $adrotate_notifications['notification_dash_soon'] == 'Y') {
			$error['advert_soon'] = sprintf(_n("%s advert expires in less than 2 days.", "%s adverts are expiring in less than 2 days!", $status['expiressoon'], 'adrotate-pro'), $status['expiressoon'])." <a href=\"".admin_url('admin.php?page=adrotate')."\">".esc_html__("Check adverts", 'adrotate-pro')."</a>.";
		}
		if($status['expiresweek'] > 0 AND $adrotate_notifications['notification_dash_week'] == 'Y') {
			$error['advert_week'] = sprintf(_n("%s advert expires in less than a week.", "%s adverts are expiring in less than a week!", $status['expiresweek'], 'adrotate-pro'), $status['expiresweek'])." <a href=\"".admin_url('admin.php?page=adrotate')."\">".esc_html__("Check adverts", 'adrotate-pro')."</a>.";
		}
	}
	if($status['error'] > 0) {
		$error['advert_config'] = sprintf(_n("%s advert with configuration errors.", "%s adverts have configuration errors!", $status['error'], 'adrotate-pro'), $status['error'])." <a href=\"".admin_url('admin.php?page=adrotate')."\">".esc_html__("Check adverts", 'adrotate-pro')."</a>.";
	}

	// Schedules
	if($adrotate_notifications['notification_dash'] == 'Y') {
		if($adrotate_notifications['notification_schedules'] == 'Y') {
			$schedules = $wpdb->get_results("SELECT * FROM `{$wpdb->prefix}adrotate_schedule` WHERE `name` != '' ORDER BY `id` ASC;");
			if($schedules) {
				$now = current_time('timestamp');
				$in2days = $now + 172800;
				$schedule_warning = array();

				foreach($schedules as $schedule) {
					if(($schedule->spread == 'Y' OR $schedule->spread_all == 'Y') AND $schedule->maximpressions == 0) $schedule_warning[] = $schedule->id;
					if(($schedule->spread == 'Y' OR $schedule->spread_all == 'Y') AND $schedule->maximpressions < 2000) $schedule_warning[] = $schedule->id;
					if($schedule->day_mon == 'N' AND $schedule->day_tue == 'N' AND $schedule->day_wed == 'N' AND $schedule->day_thu == 'N' AND $schedule->day_fri == 'N' AND $schedule->day_sat == 'N' AND $schedule->day_sun == 'N') $schedule_warning[] = $schedule->id;
//					if($schedule->stoptime < $in2days) $schedule_warning[] = $schedule->id;
//					if($schedule->stoptime < $now) $schedule_warning[] = $schedule->id;
				}

				$schedule_warning = count(array_unique($schedule_warning));

				if($schedule_warning > 0) {
					$error['schedule_warning'] = sprintf(_n("%s schedule has a warning.", "%s schedules have warnings!", $schedule_warning, 'adrotate-pro'), $schedule_warning)." <a href=\"".admin_url('admin.php?page=adrotate-schedules')."\">".esc_html__("Check schedules", 'adrotate-pro')."</a>.";
				}
			}
			unset($schedule_warning, $schedules);
		}
	}

	// Caching
	if($adrotate_config['w3caching'] == 'Y' AND !is_plugin_active('w3-total-cache/w3-total-cache.php')) {
		$error['w3tc_not_active'] = esc_html__("You have enabled caching support but W3 Total Cache is not active on your site!", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=misc')."\">".esc_html__("Disable W3 Total Cache Support", 'adrotate-pro')."</a>.";
	}
	if($adrotate_config['w3caching'] == "Y" AND !defined('W3TC_DYNAMIC_SECURITY')) {
		$error['w3tc_no_hash'] = esc_html__("You have enable caching support but the W3TC_DYNAMIC_SECURITY definition is not set.", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=misc')."\">".esc_html__("How to configure W3 Total Cache", 'adrotate-pro')."</a>.";
	}

	if($adrotate_config['borlabscache'] == 'Y' AND !is_plugin_active('borlabs-cache/borlabs-cache.php')) {
		$error['borlabs_not_active'] = esc_html__("You have enable caching support but Borlabs Cache is not active on your site!", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=misc')."\">".esc_html__("Disable Borlabs Cache Support", 'adrotate-pro')."</a>.";
	}
	if($adrotate_config['borlabscache'] == 'Y' AND is_plugin_active('borlabs-cache/borlabs-cache.php')) {
		$borlabs_config = get_option('BorlabsCacheConfigInactive');
		if($borlabs_config['cacheActivated'] == 'yes' AND strlen($borlabs_config['fragmentCaching']) < 1) {
			$error['borlabs_fragment_error'] = esc_html__("You have enabled Borlabs Cache support but Fragment caching is not enabled!", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=borlabs-cache-fragments')."\">".esc_html__("Enable Fragment Caching", 'adrotate-pro')."</a>.";
		}
		unset($borlabs_config);
	}

	// Notifications
	if($adrotate_notifications['notification_email'] == 'Y' AND $adrotate_notifications['notification_mail_geo'] == 'N' AND $adrotate_notifications['notification_mail_status'] == 'N' AND $adrotate_notifications['notification_mail_queue'] == 'N' AND $adrotate_notifications['notification_mail_approved'] == 'N' AND $adrotate_notifications['notification_mail_rejected'] == 'N') {
		$error['mail_not_configured'] = esc_html__("You have enabled email notifications but did not select anything to be notified about.", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=notifications')."\">".esc_html__("Set up notifications", 'adrotate-pro')."</a>.";
	}

	// Geo Related
	$lookups = get_option('adrotate_geo_requests');

	if($a AND $a['status'] == 0 AND $adrotate_config['enable_geo'] == 5) {
		$error['geo_license'] = esc_html__("The AdRotate Geo service can only be used after you activate your license for this website.", 'adrotate-pro'). " <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=license')."\">".esc_html__("Activate license", 'adrotate-pro')."</a>!";
	}
	if(($adrotate_config['enable_geo'] == 3 OR $adrotate_config['enable_geo'] == 4 OR $adrotate_config['enable_geo'] == 5) AND $lookups > 0 AND $lookups < 1000) {
		$error['geo_almostnolookups'] = sprintf(esc_html__("You are running out of Geo Targeting Lookups. You have less than %d remaining lookups.", 'adrotate-pro'), $lookups);
	}
	if(($adrotate_config['enable_geo'] == 3 OR $adrotate_config['enable_geo'] == 4) AND $lookups < 1) {
		$error['geo_nolookups'] = esc_html__("Geo Targeting is no longer working because you have no more lookups.", 'adrotate-pro');
	}
	if($adrotate_config['enable_geo'] == 5 AND $lookups < 1) {
		$error['geo_nolookups'] = esc_html__("AdRotate Geo is no longer working because you have no more lookups for today. This resets at midnight UTC/GMT.", 'adrotate-pro');
	}
	if(($adrotate_config['enable_geo'] == 3 OR $adrotate_config['enable_geo'] == 4) AND (strlen($adrotate_config['geo_email']) < 1 OR strlen($adrotate_config['geo_pass']) < 1)) {
		$error['geo_maxmind_details'] = esc_html__("Geo Targeting is not working because your MaxMind account details are incomplete.", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=geo')."\">".esc_html__("Enter MaxMind account details", 'adrotate-pro')."</a>.";
	}
	if($adrotate_config['enable_geo'] == 6 AND !isset($_SERVER['HTTP_CF_IPCOUNTRY'])) {
		$error['geo_cloudflare_header'] = esc_html__("Geo Targeting is not working. Check if IP Geolocation is enabled in your CloudFlare account.", 'adrotate-pro');
	}
	if($adrotate_config['enable_geo'] == 7 AND strlen($adrotate_config['geo_pass']) < 1) {
		$error['geo_ipstack_details'] = esc_html__("Geo Targeting is not working because your ipstack account API key is missing.", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=geo')."\">".esc_html__("Enter API key", 'adrotate-pro')."</a>.";
	}

	// Advertiser Related
	if($adrotate_config['enable_advertisers'] == 'Y' ) {
		$queue_count = $wpdb->get_var("SELECT count(`type`) FROM `{$wpdb->prefix}adrotate` WHERE `type` = 'queue';");
		if($queue_count > 0) {
			$error['advertiser_queue'] = esc_html__("One or more adverts are awaiting moderation.", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate&view=queue')."\">".esc_html__("Check queue", 'adrotate-pro')."</a>.";
		}
	}

	// Misc
	if(!is_writable(WP_CONTENT_DIR.'/'.$adrotate_config['banner_folder'])) {
		$error['banners_folder'] = esc_html__("Your AdRotate Banner folder is not writable or does not exist.", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=maintenance')."\">".esc_html__("Set up your banners folder", 'adrotate-pro')."</a>.";
	}
	if(!is_writable(WP_CONTENT_DIR.'/'.$adrotate_config['report_folder'])) {
		$error['reports_folder'] = esc_html__("Your AdRotate Reports folder is not writable or does not exist.", 'adrotate-pro')." <a href=\"".admin_url('/admin.php?page=adrotate-settings&tab=maintenance')."\">".esc_html__("Set up your reports folder", 'adrotate-pro')."</a>.";
	}
	if(is_dir(WP_PLUGIN_DIR.'/adrotate/')) {
		$error['adrotate_free_version_exists'] = esc_html__("You still have the free version of AdRotate installed. Please remove it!", 'adrotate-pro')." <a href=\"".admin_url('/plugins.php?s=adrotate&plugin_status=all')."\">".esc_html__("Delete AdRotate plugin", 'adrotate-pro')."</a>.";
	}
	if(basename(__DIR__) != 'adrotate' AND basename(__DIR__) != 'adrotate-pro') {
		$error['adrotate_folder_names'] = esc_html__("Something is wrong with your installation of AdRotate Pro. Either the plugin is installed twice or your current installation has the wrong folder name. Please install the plugin properly!", 'adrotate-pro')." <a href=\"https://support.ajdg.net/knowledgebase.php?article=8\" target=\"_blank\">".esc_html__("Installation instructions", 'adrotate-pro')."</a>.";
	}

	$error = (isset($error) AND is_array($error)) ? $error : false;

	return $error;
}

/*-------------------------------------------------------------
 Name:      adrotate_notifications_dashboard
 Purpose:   Show dashboard notifications
-------------------------------------------------------------*/
function adrotate_notifications_dashboard() {
	global $current_user, $wpdb;

	$page = (isset($_GET['page'])) ? $_GET['page'] : '';
	$displayname = (strlen($current_user->user_firstname) > 0) ? $current_user->user_firstname : $current_user->display_name;

	if(current_user_can('adrotate_ad_manage')) {
		// Advert notifications, errors, important stuff
		$adrotate_has_error = adrotate_dashboard_error();
		if($adrotate_has_error) {
			echo "<div class=\"ajdg-notification notice\">";
			echo "	<div class=\"ajdg-notification-logo\" style=\"background-image:url('".plugins_url('/images/notification.png', __FILE__)."');\"><span></span></div>";
			echo "	<div class=\"ajdg-notification-message\"><strong>AdRotate Professional</strong> has detected "._n("one issue that requires", "several issues that require", count($adrotate_has_error), 'adrotate-pro')." ".esc_html__("your attention:", 'adrotate-pro')."<br />";
			foreach($adrotate_has_error as $error => $message) {
				echo "&raquo; ".$message."<br />";
			}
			echo "	</div>";
			echo "</div>";
		}
	}

	if(current_user_can('update_plugins')) {
		// Finish update
		$adrotate_db_version = get_option('adrotate_db_version');
		$adrotate_version = get_option('adrotate_version');
		$recommend_gooseup = get_option('adrotate_hide_gooseup');

		if($adrotate_db_version['current'] < ADROTATE_DB_VERSION OR $adrotate_version['current'] < ADROTATE_VERSION) {
			$plugin_version = get_plugins();
			$plugin_version = $plugin_version['adrotate-pro/adrotate-pro.php']['Version'];

			// Do the update
			adrotate_finish_upgrade();

			// Thank user for updating
			echo "<div class=\"ajdg-notification notice\">";
			echo "	<div class=\"ajdg-notification-logo\" style=\"background-image:url('".plugins_url('/images/notification.png', __FILE__)."');\"><span></span></div>";
			echo "	<div class=\"ajdg-notification-message\">Hello <strong>".$displayname."</strong>! Thanks for updating <strong>AdRotate Professional</strong> to version <strong>".$plugin_version."</strong>!<br /> Your <strong>Database and settings</strong> have been updated to the latest version.<br />For an overview of what has changed you can take a look at the <a href=\"https://ajdg.solutions/blog/\" target=\"_blank\">recent blog post</a> on my website.</div>";
			echo "</div>";
		}
		
		if(isset($_GET['hide']) AND $_GET['hide'] == 'gooseup') update_option('adrotate_hide_gooseup', current_time('timestamp') + (8 * MONTH_IN_SECONDS));
		if(strpos($page, 'adrotate') !== false AND !is_plugin_active('gooseup/gooseup.php') AND $recommend_gooseup < time()) {
			// Recommend GooseUp
			echo "<div class=\"ajdg-notification notice\">";
			echo "	<div class=\"ajdg-notification-logo\" style=\"background-image: url('".plugins_url('/images/notification.png', __FILE__)."');\"><span></span></div>";
			echo "	<div class=\"ajdg-notification-message\">In order to provide reliable and fast updates I have created <b>GooseUp</b>. This is a system dedicated to providing updates for all my plugins. <b>GooseUp</b> also supports Auto-Updates, just like plugins that are downloaded from the WordPress website.<br /><b>The GooseUp plugin is available for free on my website - <a href=\"https://ajdg.solutions/product/gooseup/\" target=\"_blank\">Learn more about GooseUp &raquo;</a></b></div>";
			echo "	<div class=\"ajdg-notification-cta\">";
			echo "		<a href=\"https://ajdg.solutions/wp-content/free-downloads/gooseup.zip\" class=\"ajdg-notification-act button-primary\" target=\"_blank\">DOWNLOAD GOOSEUP</a>";
			echo "		<a href=\"".admin_url('admin.php?page=adrotate')."&hide=gooseup\" class=\"ajdg-notification-dismiss\">Maybe later</a>";
			echo "	</div>";
			echo "</div>";
		}
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_meta_links
 Purpose:	Extra links on the plugins dashboard page
-------------------------------------------------------------*/
function adrotate_meta_links($links, $file) {
	if($file !== 'adrotate-pro/adrotate-pro.php' ) return $links;

	$links['ajdg-help'] = sprintf('<a href="%s" target="_blank">%s</a>', admin_url('admin.php?page=adrotate-support'), 'Plugin support');
	$links['ajdg-more'] = sprintf('<a href="%s" target="_blank">%s</a>', 'https://ajdg.solutions/plugins/', 'More plugins');
	if(!is_plugin_active('gooseup/gooseup.php')) {
		$links['ajdg-gooseup'] = sprintf('<a href="%s" target="_blank"><strong>%s</strong></a>', 'https://ajdg.solutions/product/gooseup/', 'Get GooseUp to enable updates');
	}

	return $links;
}

/*-------------------------------------------------------------
 Name:      adrotate_action_links
 Purpose:	Extra links on the plugins dashboard page
-------------------------------------------------------------*/
function adrotate_action_links($links) {
	$links['ajdg-dashboard'] = sprintf('<a href="%s">%s</a>', admin_url('admin.php?page=adrotate'), 'Dashboard');

	return $links;
}

/*-------------------------------------------------------------
 Name:      adrotate_user_notice
 Purpose:   Credits shown on user statistics
-------------------------------------------------------------*/
function adrotate_user_notice() {

	echo "<table class=\"widefat\" style=\"margin-top:.5em\">";

	echo "<thead>";
	echo "<tr valign=\"top\">";
	echo "	<th colspan=\"2\">".esc_html__("Your adverts", 'adrotate-pro')."</th>";
	echo "</tr>";
	echo "</thead>";

	echo "<tbody>";
	echo "<tr>";
	echo "	<td><center><a href=\"https://ajdg.solutions/product-category/adrotate-pro/?mtm_campaign=adrotatepro&mtm_keyword=credits\" title=\"AdRotate plugin for WordPress\"><img src=\"".plugins_url('/images/logo-60x60.png', __FILE__)."\" alt=\"logo-60x60\" width=\"60\" height=\"60\" /></a></center></td>";
	echo "	<td>".esc_html__("The overall stats do not take adverts from other advertisers into account.", 'adrotate-pro')."<br />".esc_html__("All statistics are indicative. They do not nessesarily reflect results counted by other parties.", 'adrotate-pro')."<br />".esc_html__("Your ads are published with", 'adrotate-pro')." <a href=\"https://ajdg.solutions/product-category/adrotate-pro/?mtm_campaign=adrotatepro&mtm_keyword=credits\" target=\"_blank\">AdRotate Professional for WordPress</a>.</td>";

	echo "</tr>";
	echo "</tbody>";

	echo "</table>";
}

/*-------------------------------------------------------------
 Name:      adrotate_dropdown_folder_contents
 Purpose:   List folder contents for dropdown menu
-------------------------------------------------------------*/
function adrotate_dropdown_folder_contents($base_dir, $extensions = array('jpg', 'jpeg', 'gif', 'png', 'webp', 'svg', 'html', 'htm', 'js', 'mp4'), $max_level = 1, $level = 0, $parent = '') {
	$index = array();

	// List the folders and files
	foreach(scandir($base_dir) as $file) {
		if($file == '.' || $file == '..' || $file == '.DS_Store' || $file == 'index.php') continue;

		$dir = $base_dir.'/'.$file;
		if(is_dir($dir)) {
			if($level >= $max_level) continue;
			$index[] = adrotate_dropdown_folder_contents($dir, $extensions, $max_level, $level+1, $file);
		} else {
			$fileinfo = pathinfo($file);
			if(array_key_exists('extension', $fileinfo)) {
				if(in_array($fileinfo['extension'], $extensions)) {
					if($level > 0) $file = $parent.'/'.$file;
					$index[]= $file;
				}
			}
		}
	}
	unset($file);

	// Clean up and sort ascending
	$items = array();
	foreach($index as $key => $item) {
		if(is_array($item)) {
			unset($index[$key]);
			if(count($item) > 0) {
				foreach($item as $k => $v) {
					$index[] = $v;
				}
				unset($k, $v);
			}
		}
	}
	unset($key, $item);
	sort($index);

	return $index;
}

/*-------------------------------------------------------------
 Name:      adrotate_mediapage_folder_contents
 Purpose:   List sub-folder contents for media manager
-------------------------------------------------------------*/
function adrotate_mediapage_folder_contents($asset_folder, $extensions = array('jpg', 'jpeg', 'gif', 'png', 'webp', 'svg', 'html', 'htm', 'js', 'mp4'), $level = 1) {
	$index = $assets = array();

	// Read Banner folder
	if($handle = opendir($asset_folder)) {
	    while(false !== ($file = readdir($handle))) {
	        if($file != '.' AND $file != '..' AND $file != 'index.php' AND $file != '.DS_Store') {
	            $assets[] = $file;
	        }
	    }
	    closedir($handle);

	    if(count($assets) > 0) {
			$new_level = $level + 1;

			foreach($assets as $key => $asset) {
				$fileinfo = pathinfo($asset);
				unset($fileinfo['dirname']);
				if(is_dir($asset_folder.'/'.$asset)) { // Read subfolder
					if($level <= 2) { // Not to deep
						$fileinfo['contents'] = adrotate_mediapage_folder_contents($asset_folder.'/'.$asset, $extensions, $new_level);
						$index[] = $fileinfo;
					}
				} else { // It's a file
					if(array_key_exists('extension', $fileinfo)) {
						if(in_array($fileinfo['extension'], $extensions)) {
							$index[] = $fileinfo;
						}
					}
				}
				unset($fileinfo);
			}
			unset($level, $new_level);
		}
	}

	return $index;
}

/*-------------------------------------------------------------
 Name:      adrotate_clean_folder_contents
 Purpose:   Delete unwanted advert assets after uploading a zip file
-------------------------------------------------------------*/
function adrotate_clean_folder_contents($asset_folder) {
	$index = $assets = array();

	// Read asset folder
	if($handle = opendir($asset_folder)) {
		$extensions = array('jpg', 'jpeg', 'gif', 'png', 'webp', 'svg', 'html', 'htm', 'js', 'mp4');

	    while(false !== ($asset = readdir($handle))) {
	        if($asset != '.' AND $asset != '..') {
				$fileinfo = pathinfo($asset);
				unset($fileinfo['dirname']);
				if(is_dir($asset_folder.'/'.$asset)) { // Read subfolder
					adrotate_clean_folder_contents($asset_folder.'/'.$asset);
					if(count(scandir($asset_folder.'/'.$asset)) == 2) { // Remove empty folder
						adrotate_unlink($asset, $asset_folder);
					}
				} else { // It's a file
					if(array_key_exists('extension', $fileinfo)) {
						if(!in_array($fileinfo['extension'], $extensions)) {
							adrotate_unlink($asset, $asset_folder);
						}
					}
				}
				unset($fileinfo);
	        }
	    }
	    closedir($handle);
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_unlink
 Purpose:   Delete a file or folder from the banners folder
-------------------------------------------------------------*/
function adrotate_unlink($asset, $path = '') {
	global $adrotate_config;

	if(!empty($asset)) {
		$access_type = get_filesystem_method();
		if($access_type === 'direct') {
			if($path == 'banners') {
				$path = WP_CONTENT_DIR.'/'.$adrotate_config['banner_folder'].'/'.$asset;
			} else if($path == 'reports') {
				$path = WP_CONTENT_DIR.'/'.$adrotate_config['report_folder'].'/'.$asset;
			} else {
				$path = $path.'/'.$asset;
			}

			$credentials = request_filesystem_credentials(site_url().'/wp-admin/', '', false, false, array());

			if(!WP_Filesystem($credentials)) {
				return false;
			}

			global $wp_filesystem;

			if(!is_dir($path)) { // It's a file
				if(@unlink($path)) {
					return true;
				} else {
					return false;
				}
			} else { // It's a folder
				if($wp_filesystem->rmdir($path, true)) {
					return true;
				} else {
					return false;
				}
			}
		} else {
			return false;
		}
	} else {
		return false;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_status
 Purpose:   Internal redirects
-------------------------------------------------------------*/
function adrotate_status($status, $args = null) {

	$defaults = array(
		'ad' => '',
		'group' => '',
		'file' => '',
		'error' => ''
	);
	$arguments = wp_parse_args($args, $defaults);

	switch($status) {
		// Management messages
		case '200' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad saved", 'adrotate-pro') ."</p></div>";
		break;

		case '201' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Group saved", 'adrotate-pro') ."</p></div>";
		break;

		case '202' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Banner image saved", 'adrotate-pro') ."</p></div>";
		break;

		case '203' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad(s) deleted", 'adrotate-pro') ."</p></div>";
		break;

		case '204' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Group deleted", 'adrotate-pro') ."</p></div>";
		break;

		case '205' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Advertiser updated", 'adrotate-pro') ."</p></div>";
		break;

		case '206' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("File/folder deleted", 'adrotate-pro') ."</p></div>";
		break;

		case '207' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Group(s) duplicated", 'adrotate-pro') ."</p></div>";
		break;

		case '208' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad(s) statistics reset", 'adrotate-pro') ."</p></div>";
		break;

		case '209' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad(s) renewed", 'adrotate-pro') ."</p></div>";
		break;

		case '210' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad(s) deactivated", 'adrotate-pro') ."</p></div>";
		break;

		case '211' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad(s) activated", 'adrotate-pro') ."</p></div>";
		break;

		case '212' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Email(s) with reports successfully sent", 'adrotate-pro') ."</p></div>";
		break;

		case '213' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Group including it\'s Ads deleted", 'adrotate-pro') ."</p></div>";
		break;

		case '214' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Weight changed", 'adrotate-pro') ."</p></div>";
		break;

		case '215' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Export created", 'adrotate-pro') .'. <a href="' . WP_CONTENT_URL . '/'.$adrotate_config['report_folder'].'/'.$arguments['file'].'">Download</a>.</p></div>';
		break;

		case '216' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Adverts imported", 'adrotate-pro') .'</div>';
		break;

		case '217' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Schedule saved", 'adrotate-pro') .'</div>';
		break;

		case '218' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Schedule(s) deleted", 'adrotate-pro') .'</div>';
		break;

		case '219' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Advert(s) duplicated", 'adrotate-pro') .'</div>';
		break;

		case '220' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Advert(s) archived", 'adrotate-pro') .'</div>';
		break;

		case '221' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Advert(s) moved to the trash", 'adrotate-pro') .'</div>';
		break;

		case '222' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Advert(s) restored from trash", 'adrotate-pro') .'</div>';
		break;

		case '223' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Folder created.", 'adrotate-pro') ."</p></div>";
		break;

		case '224' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Schedule(s) duplicated.", 'adrotate-pro') ."</p></div>";
		break;

		case '225' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Uploaded CSV file seems incompatible. Make sure the export is from a recent AdRotate Pro version or if you made it yourself that ALL columns match up and the version is set to '0'.", 'adrotate-pro') ."</p></div>";
		break;

		case '226' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Advert HTML generated and placed in the AdCode field. Configure your advert below. Do not forget to check all settings and add or select a schedule.", 'adrotate-pro') .'</div>';
		break;

		case '227' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Header & ads.txt updated.", 'adrotate-pro') .'</div>';
		break;

		// Advertiser messages
		case '300' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Your message has been sent. Someone will be in touch shortly.", 'adrotate-pro') ."</p></div>";
		break;

		case '301' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Advert submitted for review", 'adrotate-pro') ."</p></div>";
		break;

		case '302' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Advert updated and awaiting review", 'adrotate-pro') ."</p></div>";
		break;

		case '303' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Email(s) with reports successfully sent", 'adrotate-pro') ."</p></div>";
		break;

		case '304' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad approved", 'adrotate-pro') ."</p></div>";
		break;

		case '305' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad(s) rejected", 'adrotate-pro') ."</p></div>";
		break;

		case '306' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ad(s) queued", 'adrotate-pro') ."</p></div>";
		break;

		// Settings
		case '400' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Settings saved", 'adrotate-pro') ."</p></div>";
		break;

		case '401' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("If any maintenance tasks were missing they have been scheduled", 'adrotate-pro') ."</p></div>";
		break;

		case '402' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Database and settings have been updated", 'adrotate-pro') ."</p></div>";
		break;

		case '403' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Geo Targeting has been reset", 'adrotate-pro') ."</p></div>";
		break;

		case '405' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Ads evaluated and statuses have been corrected where required", 'adrotate-pro') ."</p></div>";
		break;

		case '407' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Test notification sent", 'adrotate-pro') ."</p></div>";
		break;

		case '409' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("AdRotate tried to create a banners, Reports folder and an ads.txt file", 'adrotate-pro') ."</p></div>";
		break;

		// (all) Error messages
		case '500' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Action prohibited", 'adrotate-pro') ."</p></div>";
		break;

		case '501' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("The ad was saved but has an issue which might prevent it from working properly. Review the colored ad.", 'adrotate-pro') ."</p></div>";
		break;

		case '502' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("The ad was saved but has an issue which might prevent it from working properly. Please contact staff.", 'adrotate-pro') ."</p></div>";
		break;

		case '503' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("No data found in selected time period", 'adrotate-pro') ."</p></div>";
		break;

		case '504' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Database can only be optimized or cleaned once every hour", 'adrotate-pro') ."</p></div>";
		break;

		case '505' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Form can not be (partially) empty!", 'adrotate-pro') ."</p></div>";
		break;

		case '506' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("No file uploaded.", 'adrotate-pro') ."</p></div>";
		break;

		case '507' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("The file could not be read.", 'adrotate-pro') ."</p></div>";
		break;

		case '509' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("No ads found.", 'adrotate-pro') ."</p></div>";
		break;

		case '510' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Wrong file type. No file uploaded.", 'adrotate-pro') ."</p></div>";
		break;

		case '511' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("No file selected or file is too large.", 'adrotate-pro') ."</p></div>";
		break;

		case '512' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("There was an error unzipping the file. Please try again later.", 'adrotate-pro') .'<br />Error: '.$arguments['error']."</p></div>";
		break;

		case '513' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("The advert hash is not usable or is missing required data. Please copy the hash correctly and try again.", 'adrotate-pro') ."</p></div>";
		break;

		case '514' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("The advert hash can not be used on the same site as it originated from or is not a valid hash for importing.", 'adrotate-pro') ."</p></div>";
		break;

		case '515' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Folder name is empty or too long, please keep it under 100 characters.", 'adrotate-pro') ."</p></div>";
		break;

		case '516' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Something went wrong creating the folder, try again.", 'adrotate-pro') ."</p></div>";
		break;

		case '517' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Something went wrong deleting the file or folder. Make sure the file exists and that your file permissions are correct.", 'adrotate-pro') ."</p></div>";
		break;

		// Licensing
		case '600' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Invalid request", 'adrotate-pro') ."</p></div>";
		break;

		case '601' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("No license key or email provided", 'adrotate-pro') ."</p></div>";
		break;

		case '602' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("The request did not get through or the response was invalid. Please contact support.", 'adrotate-pro') .'<br />'.$arguments['error']."</p></div>";
		break;

		case '603' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("The email provided is invalid.", 'adrotate-pro') ."</p></div>";
		break;

		case '604' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Unknown license key or wrong license and email address combination.", 'adrotate-pro') ."</p></div>";
		break;

		case '605' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("The purchase matching this product is not complete. Please contact support.", 'adrotate-pro') ."</p></div>";
		break;

		case '606' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("No remaining activations for this license. But another license or manage your license activations from your account on ajdg.solutions.", 'adrotate-pro') ."</p></div>";
		break;

		case '607' :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Could not de-activate key. Key may already be de-activated. Please contact support if you think this is not correct.", 'adrotate-pro') ."</p></div>";
		break;

		case '608' :
			echo "<div class=\"ajdg-notification notice\"><div class=\"ajdg-notification-logo\" style=\"background-image: url('".plugins_url('/images/notification.png', __FILE__)."');\"><span></span></div><div class=\"ajdg-notification-message\"><strong>Woohoo! Your license is now active!</strong><br />A license is valid for one year from the purchase date. During this period you'll get plugin updates and of-course you'll have access to AdRotate Geo and Ticket Support.<br />You'll receive up to 3 email reminders when the license is about to expire.</div></div>";
		break;

		case '609' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Your license is now de-activated", 'adrotate-pro') ."</p></div>";
		break;

		case '610' :
			echo "<div id=\"message\" class=\"updated\"><p>". esc_html__("Thank you. Your licenses have been reset", 'adrotate-pro') ."</p></div>";
		break;

		// Support
		case '701' :
			echo '<div class="ajdg-notification notice"><div class="ajdg-notification-logo" style="background-image: url(\''.plugins_url('/images/notification.png', __FILE__).'\');"><span></span></div><div class="ajdg-notification-message"><strong>Support ticket sent.</strong><br />I will be in touch within one or two business days! Meanwhile, please check out the <a href="https://support.ajdg.net/knowledgebase.php?mtm_campaign=adrotatepro&mtm_keyword=support_banner" target="_blank">AdRotate knowledgebase</a>.<br /><strong>Please do not send multiple messages with the same question. This will clutter up my inbox and delays my response to you!</strong></div></div>';
		break;

		default :
			echo "<div id=\"message\" class=\"error\"><p>". esc_html__("Unexpected error", 'adrotate-pro') ."</p></div>";
		break;
	}

	unset($arguments, $args);
}
?>
