<?php
/* ------------------------------------------------------------------------------------
*  COPYRIGHT AND TRADEMARK NOTICE
*  Copyright 2008-2026 Arnan de Gans. All Rights Reserved.
*  The software AdRotate, related services and ADROTATE the name are owned by Arnan de Gans.

*  COPYRIGHT NOTICES AND ALL THE COMMENTS SHOULD REMAIN INTACT.
*  By using this code you agree to indemnify Arnan de Gans from any
*  liability that might arise from its use.
------------------------------------------------------------------------------------ */

/*-------------------------------------------------------------
 Name:      adrotate_generate_adcode
 Purpose:   Generate advert code based on user input
-------------------------------------------------------------*/
function adrotate_generate_adcode() {
	global $wpdb, $adrotate_config;

	if(wp_verify_nonce($_POST['adrotate_nonce'], 'adrotate_generate_ad')) {
		// Mandatory
		$id = '';
		if(isset($_POST['adrotate_id'])) $id = sanitize_key($_POST['adrotate_id']);

		// Basic advert
		$fullsize_image = $targeturl = '';
		if(isset($_POST['adrotate_fullsize_dropdown'])) $fullsize_image = strip_tags(trim($_POST['adrotate_fullsize_dropdown']));
		if(isset($_POST['adrotate_targeturl'])) $targeturl = strip_tags(trim($_POST['adrotate_targeturl']));

		$small_image = $medium_image = $large_image = '';
		if(isset($_POST['adrotate_small_dropdown'])) $small_image = strip_tags(trim($_POST['adrotate_small_dropdown']));
		if(isset($_POST['adrotate_medium_dropdown'])) $medium_image = strip_tags(trim($_POST['adrotate_medium_dropdown']));
		if(isset($_POST['adrotate_large_dropdown'])) $large_image = strip_tags(trim($_POST['adrotate_large_dropdown']));

		$video_aspect_ratio = 0;
		$video_autoplay = $video_muted = '';
		if(isset($_POST['adrotate_video_ratio'])) $video_aspect_ratio = strip_tags(trim($_POST['adrotate_video_ratio']));
		if(isset($_POST['adrotate_video_autoplay'])) $video_autoplay = strip_tags(trim($_POST['adrotate_video_autoplay']));
		if(isset($_POST['adrotate_video_muted'])) $video_muted = strip_tags(trim($_POST['adrotate_video_muted']));

		$new_window = $nofollow = $title_attr = $alt_attr = '';
		if(isset($_POST['adrotate_newwindow'])) $new_window = strip_tags(trim($_POST['adrotate_newwindow']));
		if(isset($_POST['adrotate_nofollow'])) $nofollow = strip_tags(trim($_POST['adrotate_nofollow']));
		if(isset($_POST['adrotate_title_attr'])) $title_attr = strip_tags(trim($_POST['adrotate_title_attr']));

		$portability = '';
		if(isset($_POST['adrotate_portability'])) $portability = strip_tags(trim($_POST['adrotate_portability']));

		if(current_user_can('adrotate_ad_manage')) {
			if(strlen($portability) == 0) {
				// Fullsize Image
				$fullsize_path = WP_CONTENT_URL.'/'.$adrotate_config['banner_folder'].'/'.$fullsize_image;

				// url?
				if(isset($targeturl) AND strlen($targeturl) != 0) {
					if(!preg_match('/^http(s)?:\/\//', $targeturl)) {
						$targeturl = 'https://'.$targeturl; // Assume https
					}
				} else {
					$targeturl = '#';
				}

				// Video ad, aspect ratio?
				if($video_aspect_ratio > 0) {
					if($video_aspect_ratio == 1) $padding = '12.82%';
					else if($video_aspect_ratio == 2) $padding = '12.36%';
					else if($video_aspect_ratio == 3) $padding = '100%';
					else if($video_aspect_ratio == 4) $padding = '56.25%';
					else if($video_aspect_ratio == 5) $padding = '177.77%';
					else if($video_aspect_ratio == 6) $padding = '75%';
					else $padding = '12.36%';
				}

				// Video player options
				if(isset($video_muted) AND strlen($video_muted) != 0) $video_muted = " muted";
					else $video_muted = "";

				// Open in a new window?
				if(isset($new_window) AND strlen($new_window) != 0) {
					$new_window = " target=\"_blank\"";
				} else {
					$new_window = "";
				}

				// Set nofollow?
				if(isset($nofollow) AND strlen($nofollow) != 0) {
					$nofollow = " rel=\"nofollow\"";
				} else {
					$nofollow = "";
				}

				// Add alt and title attributes?
				if(isset($title_attr) AND strlen($title_attr) != 0) {
					$fileinfo = pathinfo($fullsize_path);

					$title_attr = " title=\"".$fileinfo['filename']."\"";
					$alt_attr = " alt=\"".$fileinfo['filename']."\"";
				} else {
					$title_attr = $alt_attr = "";
				}

				// Viewports
				$srcset = $sizes = array();
				if(strlen($large_image) > 0) {
					$large_path = WP_CONTENT_URL.'/'.$adrotate_config['banner_folder'].'/'.$large_image;
					$srcset[] = "<source srcset=\"".$large_path."\" media=\"(min-width:1280px)\">";
					unset($large_path, $large_image);
				}
				if(strlen($medium_image) > 0) {
					$medium_path = WP_CONTENT_URL.'/'.$adrotate_config['banner_folder'].'/'.$medium_image;
					$srcset[] = "<source srcset=\"".$medium_path."\" media=\"(min-width:960px)\">";
					unset($medium_path, $medium_image);
				}
				if(strlen($small_image) > 0) {
					$small_path = WP_CONTENT_URL.'/'.$adrotate_config['banner_folder'].'/'.$small_image;
					$srcset[] = "<source srcset=\"".$small_path."\" media=\"(min-width:620px)\">";
					unset($small_path, $small_image);
				}

				// Generate ad code
				if(count($srcset) > 0) {
					$bannercode = "<a href=\"".$targeturl."\"".$new_window.$nofollow.$title_attr.">\r<picture>\r\t".implode("\r\t", $srcset)."\r\t<img src=\"%asset%\" style=\"width:auto;\"".$alt_attr." />\r</picture>\r</a>";
				} else if($video_aspect_ratio > 0) {
					$bannercode = "<div style=\"position:relative; padding-bottom:".$padding."; height:0; background-color:#fefefe; cursor:pointer;\">\r\t<video style=\"position:absolute; top:0; left:0; width:100%; height:100%;\" ".$video_muted." autoplay loop>\r\t\t<source src=\"%asset%\" type=\"video/mp4\">\r\t</video>\r\t<a href=\"".$targeturl."\"><span style=\"position:absolute; width:100%; height:100%; top:0; left:0; z-index:1;\"></span></a>\r</div>";
				} else {
					$bannercode = "<a href=\"".$targeturl."\"".$new_window.$nofollow.$title_attr."><img src=\"%asset%\" style=\"width:auto;\"".$alt_attr." /></a>";
				}

				$bannercode = htmlspecialchars($bannercode, ENT_QUOTES);

				// Determine image settings
				$imagetype = 'dropdown';
				$image = WP_CONTENT_URL.'/%folder%/'.$fullsize_image;

				// Save the ad to the DB
				$wpdb->update($wpdb->prefix.'adrotate', array('bannercode' => $bannercode, 'imagetype' => $imagetype, 'image' => $image), array('id' => $id));
			} else {
				$portability = adrotate_portable_hash('import', $portability);

				// Save the advert to the DB
				$wpdb->update($wpdb->prefix.'adrotate', array('title' => $portability['title'], 'bannercode' => $portability['bannercode'], 'thetime' => $portability['thetime'], 'updated' => current_time('timestamp'), 'author' => $portability['author'],  'imagetype' => $portability['imagetype'], 'image' => $portability['image'], 'tracker' => $portability['tracker'], 'show_everyone' => $portability['show_everyone'], 'desktop' => $portability['desktop'], 'mobile' => $portability['mobile'], 'tablet' => $portability['tablet'], 'os_ios' => $portability['os_ios'], 'os_android' => $portability['os_android'], 'weight' => $portability['weight'], 'autodelete' => $portability['autodelete'], 'budget' => $portability['budget'], 'crate' => $portability['crate'], 'irate' => $portability['irate'], 'state_req' => $portability['state_req'], 'cities' => $portability['cities'], 'states' => $portability['states'], 'countries' => $portability['countries']), array('id' => $id));
			}

			wp_redirect('admin.php?page=adrotate&status=226&view=edit&ad='.$id);
			exit;
		} else {
			wp_redirect('admin.php?page=adrotate&status=500');
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_save_header
 Purpose:   Generate header code based on user input
-------------------------------------------------------------*/
function adrotate_save_header() {
	global $wpdb, $adrotate_config;

	if(wp_verify_nonce($_POST['adrotate_nonce_header'], 'adrotate_nonce')) {
		// Mandatory
		$gamcode = $headercode = $adstxt = '';
		if(isset($_POST['adrotate_gam'])) $gamcode = trim($_POST['adrotate_gam']); // Sanitized later
		if(isset($_POST['adrotate_header'])) $headercode = trim($_POST['adrotate_header']); // Sanitized later
		if(isset($_POST['adrotate_adstxt'])) $adstxt = htmlspecialchars(trim($_POST['adrotate_adstxt'], '\t '), ENT_QUOTES);

		if(current_user_can('adrotate_ad_manage')) {
			// -- ads.txt

			// Format new lines
			$adstxt = str_ireplace("\r\n", "\n", $adstxt);
			$adstxt = str_ireplace("\n\n", "\n", $adstxt);
			$adstxt = explode("\n", $adstxt);

			if(is_array($adstxt)) {
				foreach($adstxt as $ad) {
					$adpieces = explode(",", $ad);
					foreach($adpieces as $key => $piece) {
						$piece = trim($piece, "\t\r\n, ");
						$piece = preg_replace("/\s+/", " ", $piece);
						$adpieces[$key] = $piece;
					}
					$ad = implode(", ", $adpieces);
					$lines[] = $ad."\n";
					unset($ad, $adpieces);
				}

				// Write new rules
				$fp = fopen(ABSPATH.'ads.txt', 'w');
				foreach($lines as $line){
					fwrite($fp, $line);
				}
				fclose($fp);
				unset($fp);
			}

			// -- Header snippets

			// Clean things up
			if(preg_match('/%RANDOM%/', $gamcode)) $gamcode = str_replace('%RANDOM%', '%random%', $gamcode);
			if(preg_match('/%RANDOM%/', $headercode)) $headercode = str_replace('%RANDOM%', '%random%', $headercode);

			// Strip out 'dangerous' tags and crap?
			if(!current_user_can('unfiltered_html')) {
				$gamcode = wp_kses_post($gamcode);
				$headercode = wp_kses_post($headercode);
			}
			$gamcode = htmlspecialchars($gamcode, ENT_QUOTES);
			$headercode = htmlspecialchars($headercode, ENT_QUOTES);

			// Save the code
			update_option('adrotate_gam_output', (string) $gamcode);
			update_option('adrotate_header_output', (string) $headercode);

			wp_redirect('admin.php?page=adrotate&status=227&view=advanced');
			exit;
		} else {
			wp_redirect('admin.php?page=adrotate&status=500');
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_insert_advert
 Purpose:   Prepare input form on saving new or updated banners
-------------------------------------------------------------*/
function adrotate_insert_advert() {
	global $wpdb, $adrotate_config;

	if(wp_verify_nonce($_POST['adrotate_nonce'], 'adrotate_save_ad')) {
		// Mandatory
		$id = $author = $title = $bannercode = $active = '';
		if(isset($_POST['adrotate_id'])) $id = sanitize_key($_POST['adrotate_id']);
		if(isset($_POST['adrotate_username'])) $author = sanitize_key($_POST['adrotate_username']);
		if(isset($_POST['adrotate_title'])) $title = sanitize_text_field(trim($_POST['adrotate_title']));
		if(isset($_POST['adrotate_bannercode'])) $bannercode = trim($_POST['adrotate_bannercode']); // Sanitized later
		$thetime = current_time('timestamp');
		if(isset($_POST['adrotate_active'])) $active = sanitize_key($_POST['adrotate_active']);

		// Schedule
		$schedule_start_date = $schedule_start_hour = $schedule_start_minute = $schedule_end_date = $schedule_end_hour = $schedule_end_minute = $schedule_autodelete = '';
		if(isset($_POST['adrotate_schedule_start_date'])) $schedule_start_date = sanitize_key(trim($_POST['adrotate_schedule_start_date']));
		if(isset($_POST['adrotate_schedule_start_hour'])) $schedule_start_hour = sanitize_key(trim($_POST['adrotate_schedule_start_hour']));
		if(isset($_POST['adrotate_schedule_start_minute'])) $schedule_start_minute = sanitize_key(trim($_POST['adrotate_schedule_start_minute']));
		if(isset($_POST['adrotate_schedule_end_date'])) $schedule_end_date = sanitize_key(trim($_POST['adrotate_schedule_end_date']));
		if(isset($_POST['adrotate_schedule_end_hour'])) $schedule_end_hour = sanitize_key(trim($_POST['adrotate_schedule_end_hour']));
		if(isset($_POST['adrotate_schedule_end_minute'])) $schedule_end_minute = sanitize_key(trim($_POST['adrotate_schedule_end_minute']));
		if(isset($_POST['adrotate_schedule_autodelete'])) $schedule_autodelete = sanitize_key($_POST['adrotate_schedule_autodelete']);

		// Schedules and group selection
		$schedules = $groups = array();
		if(isset($_POST['scheduleselect'])) $schedules = $_POST['scheduleselect'];
		if(isset($_POST['groupselect'])) $groups = $_POST['groupselect'];

		// Advert options
		$image_field = $image_dropdown = $tracker_clicks = $tracker_impressions = $desktop = $mobile = $tablet = $os_ios = $os_android = $type = $weight = $autodelete = '';
		if(isset($_POST['adrotate_image'])) $image_field = strip_tags(trim($_POST['adrotate_image']));
		if(isset($_POST['adrotate_image_dropdown'])) $image_dropdown = strip_tags(trim($_POST['adrotate_image_dropdown']));
		if(isset($_POST['adrotate_tracker_clicks'])) $tracker_clicks = sanitize_key($_POST['adrotate_tracker_clicks']);
		if(isset($_POST['adrotate_tracker_impressions'])) $tracker_impressions = sanitize_key($_POST['adrotate_tracker_impressions']);
		if(isset($_POST['adrotate_show_everyone'])) $show_everyone = sanitize_key($_POST['adrotate_show_everyone']);
		if(isset($_POST['adrotate_desktop'])) $desktop = sanitize_key($_POST['adrotate_desktop']);
		if(isset($_POST['adrotate_mobile'])) $mobile = sanitize_key($_POST['adrotate_mobile']);
		if(isset($_POST['adrotate_tablet'])) $tablet = sanitize_key($_POST['adrotate_tablet']);
		if(isset($_POST['adrotate_ios'])) $os_ios = sanitize_key($_POST['adrotate_ios']);
		if(isset($_POST['adrotate_android'])) $os_android = sanitize_key($_POST['adrotate_android']);
		if(isset($_POST['adrotate_type'])) $type = sanitize_key($_POST['adrotate_type']);
		if(isset($_POST['adrotate_weight'])) $weight = sanitize_key($_POST['adrotate_weight']);
		if(isset($_POST['adrotate_autodelete'])) $autodelete = sanitize_key($_POST['adrotate_autodelete']);
		if(isset($_POST['adrotate_autodisable'])) $autodisable = sanitize_key($_POST['adrotate_autodisable']);

		// GeoTargeting
		$cities = $states = $state_req = '';
		$countries = $countries_westeurope = $countries_easteurope = $countries_northamerica = $countries_southamerica = $countries_southeastasia = array();
		if(isset($_POST['adrotate_geo_state_required'])) $state_req = strip_tags(trim($_POST['adrotate_geo_state_required']));
		if(isset($_POST['adrotate_geo_cities'])) $cities = strip_tags(trim($_POST['adrotate_geo_cities']));
		if(isset($_POST['adrotate_geo_states'])) $states = strip_tags(trim($_POST['adrotate_geo_states']));
		if(isset($_POST['adrotate_geo_countries'])) $countries = $_POST['adrotate_geo_countries'];
		if(isset($_POST['adrotate_geo_westeurope'])) $countries_westeurope = array('AD', 'AT', 'BE', 'DK', 'FR', 'DE', 'GR', 'IS', 'IE', 'IT', 'LI', 'LU', 'MT', 'MC', 'NL', 'NO', 'PT', 'SM', 'ES', 'SE', 'CH', 'VA', 'GB');
		if(isset($_POST['adrotate_geo_easteurope'])) $countries_easteurope = array('AL', 'AM', 'AZ', 'BY', 'BA', 'BG', 'HR', 'CY', 'CZ', 'EE', 'FI', 'GE', 'HU', 'LV', 'LT', 'MK', 'MD', 'PL', 'RO', 'RS', 'SK', 'SI', 'TR', 'UA');
		if(isset($_POST['adrotate_geo_northamerica'])) $countries_northamerica = array('AG', 'BS', 'BB', 'BZ', 'CA', 'CR', 'CU', 'DM', 'DO', 'SV', 'GD', 'GT', 'HT', 'HN', 'JM', 'MX', 'NI', 'PA', 'KN', 'LC', 'VC', 'TT', 'US');
		if(isset($_POST['adrotate_geo_southamerica'])) $countries_southamerica = array('AR', 'BO', 'BR', 'CL', 'CO', 'EC', 'GY', 'PY', 'PE', 'SR', 'UY', 'VE');
		if(isset($_POST['adrotate_geo_southeastasia'])) $countries_southeastasia = array('AU', 'BN', 'KH', 'TL', 'ID', 'LA', 'MY', 'MM', 'NZ', 'PH', 'SG', 'TH', 'VN');

		// advertiser
		$advertiser = $budget = $crate = $irate = 0;
		if(isset($_POST['adrotate_advertiser'])) $advertiser = $_POST['adrotate_advertiser'];
		if(isset($_POST['adrotate_budget'])) $budget = strip_tags(trim($_POST['adrotate_budget']));
		if(isset($_POST['adrotate_crate'])) $crate = strip_tags(trim($_POST['adrotate_crate']));
		if(isset($_POST['adrotate_irate'])) $irate = strip_tags(trim($_POST['adrotate_irate']));

		if(current_user_can('adrotate_ad_manage')) {
			if(strlen($title) < 1) $title = 'Advert '.$id;
			$title = str_replace('"', '', $title);

			// Strip out 'dangerous' tags and crap?
			if(!current_user_can('unfiltered_html')) {
				$bannercode = wp_kses_post($bannercode);
			}
			$bannercode = htmlspecialchars($bannercode, ENT_QUOTES);

			// Clean up bannercode
			if(preg_match('/%ID%/', $bannercode)) $bannercode = str_replace('%ID%', '%id%', $bannercode);
			if(preg_match('/%ASSET%/', $bannercode)) $bannercode = str_replace('%ASSET%', '%asset%', $bannercode);
			if(preg_match('/%IMAGE%/', $bannercode)) $bannercode = str_replace('%IMAGE%', '%image%', $bannercode);
			if(preg_match('/%TITLE%/', $bannercode)) $bannercode = str_replace('%TITLE%', '%title%', $bannercode);
			if(preg_match('/%RANDOM%/', $bannercode)) $bannercode = str_replace('%RANDOM%', '%random%', $bannercode);

			// Sort out start dates
			if(strlen($schedule_start_date) > 0) {
				list($schedule_start_day, $schedule_start_month, $schedule_start_year) = explode('-', $schedule_start_date);
				$schedule_start_month = (!is_numeric($schedule_start_month)) ? date('m', strtotime($schedule_start_month.'-'.$schedule_start_year)) : $schedule_start_month; // Convert month to number
			} else {
				$schedule_start_year = $schedule_start_month = $schedule_start_day = 0;
			}

			if(($schedule_start_year > 0 AND $schedule_start_month > 0 AND $schedule_start_day > 0) AND strlen($schedule_start_hour) == 0) $schedule_start_hour = '00';
			if(($schedule_start_year > 0 AND $schedule_start_month > 0 AND $schedule_start_day > 0) AND strlen($schedule_start_minute) == 0) $schedule_start_minute = '00';

			if($schedule_start_month > 0 AND $schedule_start_day > 0 AND $schedule_start_year > 0) {
				$schedule_start_date = mktime($schedule_start_hour, $schedule_start_minute, 0, $schedule_start_month, $schedule_start_day, $schedule_start_year);
			} else {
				$schedule_start_date = 0;
			}

			// Sort out end dates
			if(strlen($schedule_end_date) > 0) {
				list($schedule_end_day, $schedule_end_month, $schedule_end_year) = explode('-', $schedule_end_date);
				$schedule_end_month = (!is_numeric($schedule_end_month)) ? date('m', strtotime($schedule_end_month.'-'.$schedule_end_year)) : $schedule_end_month; // Convert month to number?
			} else {
				$schedule_end_year = $schedule_end_month = $schedule_end_day = 0;
			}

			if(($schedule_end_year > 0 AND $schedule_end_month > 0 AND $schedule_end_day > 0) AND strlen($schedule_end_hour) == 0) $schedule_end_hour = '00';
			if(($schedule_end_year > 0 AND $schedule_end_month > 0 AND $schedule_end_day > 0) AND strlen($schedule_end_minute) == 0) $schedule_end_minute = '00';

			if($schedule_end_month > 0 AND $schedule_end_day > 0 AND $schedule_end_year > 0) {
				$schedule_end_date = mktime($schedule_end_hour, $schedule_end_minute, 0, $schedule_end_month, $schedule_end_day, $schedule_end_year);
			} else {
				$schedule_end_date = 0;
			}

			// Enddate is too early, reset to default
			if($schedule_end_date <= $schedule_start_date) $schedule_end_date = $schedule_start_date + 7257600; // 84 days (12 weeks)

			// Auto-delete schedule
			if(isset($schedule_autodelete) AND strlen($schedule_autodelete) != 0) {
				$schedule_autodelete = 'Y';
			} else {
				$schedule_autodelete = 'N';
			}

			// Save the schedule to the DB
			if($schedule_start_date > 0 AND $schedule_end_date > 0) {
				$wpdb->insert($wpdb->prefix.'adrotate_schedule', array('name' => 'Schedule for '.$id.' - '.$title, 'starttime' => $schedule_start_date, 'stoptime' => $schedule_end_date, 'maxclicks' => 0, 'maximpressions' => 0, 'spread' => 'N', 'spread_all' => 'N', 'autodelete' => $schedule_autodelete));
				$schedules[] = $wpdb->insert_id;
			}

			// Statistics
			if(isset($tracker_clicks) AND strlen($tracker_clicks) != 0) $tracker_clicks = 'Y';
				else $tracker_clicks = 'N';
			if(isset($tracker_impressions) AND strlen($tracker_impressions) != 0) $tracker_impressions = 'Y';
				else $tracker_impressions = 'N';

			if($tracker_clicks == 'Y' AND $tracker_impressions == 'Y') {
				$tracker = 'Y';
			} else if($tracker_clicks == 'Y' AND $tracker_impressions == 'N') {
				$tracker = 'C';
			} else if($tracker_clicks == 'N' AND $tracker_impressions == 'Y') {
				$tracker = 'I';
			} else {
				$tracker = 'N';
			}

			// Advert options/features
			if($weight != 2 AND $weight != 4 AND $weight != 6 AND $weight != 8 AND $weight != 10) $weight = 6;
			if(isset($show_everyone) AND strlen($show_everyone) != 0) $show_everyone = 'Y';
				else $show_everyone = 'N';
			if(isset($desktop) AND strlen($desktop) != 0) $desktop = 'Y';
				else $desktop = 'N';
			if(isset($mobile) AND strlen($mobile) != 0) $mobile = 'Y';
				else $mobile = 'N';
			if(isset($tablet) AND strlen($tablet) != 0) $tablet = 'Y';
				else $tablet = 'N';
			if(isset($os_ios) AND strlen($os_ios) != 0) $os_ios = 'Y';
				else $os_ios = 'N';
			if(isset($os_android) AND strlen($os_android) != 0) $os_android = 'Y';
				else $os_android = 'N';
			if(isset($autodelete) AND strlen($autodelete) != 0) $autodelete = 'Y';
				else $autodelete = 'N';
			if(isset($autodisable) AND strlen($autodisable) != 0) $autodisable = 'Y';
				else $autodisable = 'N';

			// Rate and Budget settings
			if(!is_numeric($crate) OR $crate < 0 OR $crate > 999) $crate = 0;
			if(!is_numeric($irate) OR $irate < 0 OR $irate > 999) $irate = 0;
			if($advertiser == 0 AND $crate == 0 AND $irate == 0) $budget = 0;

			$budget = number_format($budget, 4, '.', '');
			$crate = number_format($crate, 4, '.', '');
			$irate = number_format($irate, 4, '.', '');

			// Determine image settings ($image_field has priority!)
			if(strlen($image_field) > 1) {
				$imagetype = 'field';
				$image = $image_field;
			} else if(strlen($image_dropdown) > 1) {
				$imagetype = 'dropdown';
				$image = WP_CONTENT_URL.'/%folder%/'.$image_dropdown;
			} else {
				$imagetype = $image = '';
			}

			// Geo Targeting
			if(isset($state_req) AND strlen($state_req) != 0) $state_req = 'Y';
				else $state_req = 'N';

			if(strlen($cities) > 0) {
				$cities = str_replace('"', '', $cities);
				$cities = str_replace("'", '', $cities);

				$cities = explode(',', strtolower($cities));
				foreach($cities as $key => $value) {
					$cities_clean[] = trim($value);
					unset($value);
				}
				unset($cities);
				$cities = serialize($cities_clean);
			} else {
				$cities = serialize(array());
			}

			if(strlen($states) > 0) {
				$states = str_replace('"', '', $states);
				$states = str_replace("'", '', $states);

				$states = explode(',', strtolower($states));
				foreach($states as $key => $value) {
					$states_clean[] = trim($value);
					unset($value);
				}
				unset($states);
				$states = serialize($states_clean);
			} else {
				$states = serialize(array());
			}

			$countries = array_merge($countries, $countries_westeurope, $countries_easteurope, $countries_northamerica, $countries_southamerica, $countries_southeastasia);
			$countries = array_unique($countries);
			if(count($countries) == 0) {
				$countries = serialize(array());
			} else {
				foreach($countries as $key => $value) {
					$countries_clean[] = trim($value);
					unset($value);
				}
				unset($countries);
				$countries = serialize($countries_clean);
			}

			// Save the ad to the DB
			$wpdb->update($wpdb->prefix.'adrotate', array('title' => $title, 'bannercode' => $bannercode, 'updated' => $thetime, 'author' => $author, 'imagetype' => $imagetype, 'image' => $image, 'tracker' => $tracker, 'show_everyone' => $show_everyone, 'desktop' => $desktop, 'mobile' => $mobile, 'tablet' => $tablet, 'os_ios' => $os_ios, 'os_android' => $os_android, 'type' => $active, 'weight' => $weight, 'autodelete' => $autodelete, 'autodisable' => $autodisable, 'budget' => $budget, 'crate' => $crate, 'irate' => $irate, 'state_req' => $state_req, 'cities' => $cities, 'states' => $states, 'countries' => $countries), array('id' => $id));

			// Fetch group records for the ad
			$groupmeta = $wpdb->get_results($wpdb->prepare("SELECT `group` FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `user` = 0 AND `schedule` = 0;", $id));
			$group_array = array();
			foreach($groupmeta as $meta) {
				$group_array[] = $meta->group;
				unset($meta);
			}

			// Add new groups to this ad
			$insert = array_diff($groups, $group_array);
			foreach($insert as &$value) {
				$wpdb->insert($wpdb->prefix.'adrotate_linkmeta', array('ad' => $id, 'group' => $value, 'user' => 0, 'schedule' => 0));
			}
			unset($insert, $value);

			// Remove groups from this ad
			$delete = array_diff($group_array, $groups);
			foreach($delete as &$value) {
				$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `group` = %d AND `user` = 0 AND `schedule` = 0;", $id, $value));
			}
			unset($delete, $value, $groupmeta, $group_array);

			// Fetch schedules for the ad
			$schedulemeta = $wpdb->get_results($wpdb->prepare("SELECT `schedule` FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `group` = 0 AND `user` = 0;", $id));
			$schedule_array = array();
			foreach($schedulemeta as $meta) {
				$schedule_array[] = $meta->schedule;
				unset($meta);
			}

			// Add new schedules to this ad
			$insert = array_diff($schedules, $schedule_array);
			foreach($insert as &$value) {
				$wpdb->insert($wpdb->prefix.'adrotate_linkmeta', array('ad' => $id, 'group' => 0, 'user' => 0, 'schedule' => $value));
			}
			unset($insert, $value);

			// Remove schedules from this ad
			$delete = array_diff($schedule_array, $schedules);
			foreach($delete as &$value) {
				$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `group` = 0 AND `user` = 0 AND `schedule` = %d;", $id, $value));
			}
			unset($delete, $value, $schedulemeta, $schedule_array);

			// Fetch records for the ad, see if a publisher is set
			$linkmeta = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `group` = 0 AND `user` > 0 AND `schedule` = 0;", $id));

			// Add/update/remove advertiser on this ad
			if($linkmeta == 0 AND $advertiser > 0) $wpdb->insert($wpdb->prefix.'adrotate_linkmeta', array('ad' => $id, 'group' => 0, 'user' => $advertiser, 'schedule' => 0));
			if($linkmeta == 1 AND $advertiser > 0) $wpdb->query($wpdb->prepare("UPDATE `{$wpdb->prefix}adrotate_linkmeta` SET `user` = {$advertiser} WHERE `ad` = %d AND `group` = 0 AND `schedule` = 0;", $id));
			if($linkmeta == 1 AND $advertiser == 0) $wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `group` = 0 AND `schedule` = 0;", $id));

			// Verify ad (Must be as late in the process as possible)
			$adstate = adrotate_evaluate_ad($id);
			if($adstate == 'error' OR $adstate == 'expired') {
				$action = 501;
				$active = 'error';
			} else {
				$action = 200;
			}
			$wpdb->update($wpdb->prefix.'adrotate', array('type' => $active), array('id' => $id));

			// Archive stats?
			if($active == 'archived' AND $adrotate_config['stats'] == 1) {
				adrotate_archive_stats($id);
				wp_redirect('admin.php?page=adrotate&status=220&view=archive');
				exit;
			}

			wp_redirect('admin.php?page=adrotate&status='.$action);
			exit;
		} else {
			wp_redirect('admin.php?page=adrotate&status=500');
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_insert_group
 Purpose:   Save provided data for groups, update linkmeta where required
 Since:		0.4
-------------------------------------------------------------*/
function adrotate_insert_group() {
	global $wpdb, $adrotate_config;

	if(wp_verify_nonce($_POST['adrotate_nonce'], 'adrotate_save_group')) {
		$id = $name = $modus = '';
		if(isset($_POST['adrotate_id'])) $id = sanitize_key($_POST['adrotate_id']);
		if(isset($_POST['adrotate_groupname'])) $name = sanitize_text_field($_POST['adrotate_groupname']);
		if(isset($_POST['adrotate_modus'])) $modus = sanitize_key(trim($_POST['adrotate_modus']));

		$rows = $columns = $adwidth = $adheight = $admargin_top = $admargin_bottom = $admargin_left = $admargin_right = $adspeed = $repeat_impressions = '';
		if(isset($_POST['adrotate_gridrows'])) $rows = sanitize_key(trim($_POST['adrotate_gridrows']));
		if(isset($_POST['adrotate_gridcolumns'])) $columns = sanitize_key(trim($_POST['adrotate_gridcolumns']));
		if(isset($_POST['adrotate_adwidth'])) $adwidth = sanitize_key(trim($_POST['adrotate_adwidth']));
		if(isset($_POST['adrotate_adheight'])) $adheight = sanitize_key(trim($_POST['adrotate_adheight']));
		if(isset($_POST['adrotate_admargin_top'])) $admargin_top = sanitize_key(trim($_POST['adrotate_admargin_top']));
		if(isset($_POST['adrotate_admargin_bottom'])) $admargin_bottom = sanitize_key(trim($_POST['adrotate_admargin_bottom']));
		if(isset($_POST['adrotate_admargin_left'])) $admargin_left = sanitize_key(trim($_POST['adrotate_admargin_left']));
		if(isset($_POST['adrotate_admargin_right'])) $admargin_right = sanitize_key(trim($_POST['adrotate_admargin_right']));
		if(isset($_POST['adrotate_adspeed'])) $adspeed = sanitize_key(trim($_POST['adrotate_adspeed']));
		if(isset($_POST['adrotate_repeat_impressions'])) $repeat_impressions = sanitize_key($_POST['adrotate_repeat_impressions']);

		$geo = $mobile = $fallback = $ads = $align = '';
		if(isset($_POST['adrotate_geo'])) $geo = strip_tags(trim($_POST['adrotate_geo']));
		if(isset($_POST['adrotate_mobile'])) $mobile = strip_tags(trim($_POST['adrotate_mobile']));
		if(isset($_POST['adrotate_fallback'])) $fallback = strip_tags(trim($_POST['adrotate_fallback']));
		if(isset($_POST['adselect'])) $ads = $_POST['adselect'];
		if(isset($_POST['adrotate_align'])) $align = strip_tags(trim($_POST['adrotate_align']));

		$categories = $category_loc = $category_par = $pages = $page_loc = $page_par = $woo_categories = $woo_loc = $forums = $forum_loc = '';
		if(isset($_POST['adrotate_categories'])) $categories = $_POST['adrotate_categories'];
		if(isset($_POST['adrotate_cat_location'])) $category_loc = sanitize_key($_POST['adrotate_cat_location']);
		if(isset($_POST['adrotate_cat_paragraph'])) $category_par = sanitize_key($_POST['adrotate_cat_paragraph']);
		if(isset($_POST['adrotate_pages'])) $pages = $_POST['adrotate_pages'];
		if(isset($_POST['adrotate_page_location'])) $page_loc = sanitize_key($_POST['adrotate_page_location']);
		if(isset($_POST['adrotate_page_paragraph'])) $page_par = sanitize_key($_POST['adrotate_page_paragraph']);
		if(isset($_POST['adrotate_woo_categories'])) $woo_categories = $_POST['adrotate_woo_categories'];
		if(isset($_POST['adrotate_woo_location'])) $woo_loc = sanitize_key($_POST['adrotate_woo_location']);
		if(isset($_POST['adrotate_bbpress_forums'])) $forums = $_POST['adrotate_bbpress_forums'];
		if(isset($_POST['adrotate_bbpress_location'])) $forum_loc = sanitize_key($_POST['adrotate_bbpress_location']);

		$wrapper_before = $wrapper_after = '';
		if(isset($_POST['adrotate_wrapper_before'])) $wrapper_before = trim($_POST['adrotate_wrapper_before']);
		if(isset($_POST['adrotate_wrapper_after'])) $wrapper_after = trim($_POST['adrotate_wrapper_after']);

		if(current_user_can('adrotate_group_manage')) {
			if(strlen($name) < 1) $name = 'Group '.$id;
			$name = str_replace('"', '', $name);

			if($modus < 0 OR $modus > 3) $modus = 0;
			if($adspeed < 0 OR $adspeed > 99999) $adspeed = 6000;
			if($align < 0 OR $align > 3) $align = 0;

			if(isset($repeat_impressions) AND strlen($repeat_impressions) != 0) $repeat_impressions = 'Y';
				else $repeat_impressions = 'N';
			if(is_numeric($geo)) $geo = 1;
				else $geo = 0;
			if(is_numeric($mobile)) $mobile = 1;
				else $mobile = 0;
			if(!is_numeric($fallback) OR $fallback == $id) $fallback = 0;

			// Sort out block shape and advert size
			if($rows < 1 OR $rows == '' OR !is_numeric($rows)) $rows = 2;
			if($columns < 1 OR $columns == '' OR !is_numeric($columns)) $columns = 2;
			if((is_numeric($adwidth) AND ($adwidth < 1 OR $adwidth > 9999)) OR $adwidth == '' OR (!is_numeric($adwidth) AND $adwidth != 'auto')) $adwidth = '728';
			if((is_numeric($adheight) AND ($adheight < 1 OR $adheight > 9999)) OR $adheight == '' OR (!is_numeric($adheight) AND $adheight != 'auto')) $adheight = '90';
			if($admargin_top < 0 OR $admargin_top > 99 OR $admargin_top == '' OR !is_numeric($admargin_top)) $admargin_top = 0;
			if($admargin_bottom < 0 OR $admargin_bottom > 99 OR $admargin_bottom == '' OR !is_numeric($admargin_bottom)) $admargin_bottom = 0;
			if($admargin_left < 0 OR $admargin_left > 99 OR $admargin_left == '' OR !is_numeric($admargin_left)) $admargin_left = 0;
			if($admargin_right < 0 OR $admargin_right > 99 OR $admargin_right == '' OR !is_numeric($admargin_right)) $admargin_right = 0;

			// Categories
			if(!is_array($categories)) $categories = array();
			$category = implode(',', $categories);
			if($category_par > 0) $category_loc = 4;
			if($category_loc != 4) $category_par = 0;

			// Pages
			if(!is_array($pages)) $pages = array();
			$page = implode(',', $pages);
			if($page_par > 0) $page_loc = 4;
			if($page_loc != 4) $page_par = 0;

			// WooCommerce
			if(!is_array($woo_categories)) $woo_categories = array();
			$woo_cat = implode(',', $woo_categories);

			// bbPress
			if(!is_array($forums)) $forums = array();
			$forum = implode(',', $forums);

			// Fetch records for the group
			$linkmeta = $wpdb->get_results($wpdb->prepare("SELECT `ad` FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `group` = %d AND `user` = 0;", $id));
			foreach($linkmeta as $meta) {
				$meta_array[] = $meta->ad;
			}

			if(empty($meta_array)) $meta_array = array();
			if(empty($ads)) $ads = array();

			// Add new ads to this group
			$insert = array_diff($ads,$meta_array);
			foreach($insert as &$value) {
				$wpdb->insert($wpdb->prefix.'adrotate_linkmeta', array('ad' => $value, 'group' => $id, 'user' => 0));
			}
			unset($value);

			// Remove ads from this group
			$delete = array_diff($meta_array,$ads);
			foreach($delete as &$value) {
				$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `group` = %d AND `user` = 0;", $value, $id));
			}
			unset($value);

			// Strip out 'dangerous' tags and crap?
			if(!current_user_can('unfiltered_html')) {
				$wrapper_before = wp_kses_post($wrapper_before);
				$wrapper_after = wp_kses_post($wrapper_after);
			}
			$wrapper_before = htmlspecialchars($wrapper_before, ENT_QUOTES);
			$wrapper_after = htmlspecialchars($wrapper_after, ENT_QUOTES);

			// Update the group itself
			$wpdb->update($wpdb->prefix.'adrotate_groups', array('name' => $name, 'modus' => $modus, 'fallback' => $fallback, 'cat' => $category, 'cat_loc' => $category_loc,  'cat_par' => $category_par, 'page' => $page, 'page_loc' => $page_loc, 'page_par' => $page_par, 'woo_cat' => $woo_cat, 'woo_loc' => $woo_loc, 'bbpress' => $forum, 'bbpress_loc' => $forum_loc, 'mobile' => $mobile, 'geo' => $geo, 'wrapper_before' => $wrapper_before, 'wrapper_after' => $wrapper_after, 'align' => $align, 'gridrows' => $rows, 'gridcolumns' => $columns, 'admargin' => $admargin_top, 'admargin_bottom' => $admargin_bottom, 'admargin_left' => $admargin_left, 'admargin_right' => $admargin_right, 'adwidth' => $adwidth, 'adheight' => $adheight, 'adspeed' => $adspeed, 'repeat_impressions' => $repeat_impressions), array('id' => $id));

			// Determine GeoLocation Library requirement
			$geo_count = $wpdb->get_var("SELECT COUNT(*) FROM `{$wpdb->prefix}adrotate_groups` WHERE `name` != '' AND `geo` = 1;");
			update_option('adrotate_geo_required', $geo_count);

			// Determine Dynamic Library requirement
			$dynamic_count = $wpdb->get_var("SELECT COUNT(*) FROM `{$wpdb->prefix}adrotate_groups` WHERE `name` != '' AND `modus` = 1;");
			update_option('adrotate_dynamic_required', $dynamic_count);

			// Generate CSS for group
			if($align == 0) { // None
				$group_align = '';
			} else if($align == 1) { // Left
				$group_align = ' float:left; clear:left;';
			} else if($align == 2) { // Right
				$group_align = ' float:right; clear:right;';
			} else if($align == 3) { // Center
				$group_align = ' margin: 0 auto;';
			}

			$output_css = "";
			if($modus == 0 AND ($admargin_top > 0 OR $admargin_right > 0 OR $admargin_bottom > 0 OR $admargin_left > 0 OR $align > 0)) { // Single ad group
				if($align < 3) {
					$output_css .= "\t.g".$adrotate_config['adblock_disguise']."-".$id." { margin:".$admargin_top."px ".$admargin_right."px ".$admargin_bottom."px ".$admargin_left."px;".$group_align." }\n";
				} else {
					$output_css .= "\t.g".$adrotate_config['adblock_disguise']."-".$id." { ".$group_align." }\n";
				}
			}

			if($modus == 1) { // Dynamic group
				if($adwidth != 'auto') {
					$width = " width:100%; max-width:".$adwidth."px;";
				} else {
					$width = " width:auto;";
				}

				if($adheight != 'auto') {
					$height = " height:100%; max-height:".$adheight."px;";
				} else {
					$height = " height:auto;";
				}

				if($align < 3) {
					$output_css .= "\t.g".$adrotate_config['adblock_disguise']."-".$id." { margin:".$admargin_top."px ".$admargin_right."px ".$admargin_bottom."px ".$admargin_left."px;".$width.$height.$group_align." }\n";
				} else {
					$output_css .= "\t.g".$adrotate_config['adblock_disguise']."-".$id." { ".$width." ".$height.$group_align." }\n";
				}

				unset($width_sum, $width, $height_sum, $height);
			}

			if($modus == 2) { // Block group
				if($adwidth != 'auto') {
					$width_sum = $columns * ($admargin_left + $adwidth + $admargin_right);
					$grid_width = "min-width:".$admargin_left."px; max-width:".$width_sum."px;";
				} else {
					$grid_width = "width:auto;";
				}

				$output_css .= "\t.g".$adrotate_config['adblock_disguise']."-".$id." { ".$grid_width.$group_align." }\n";
				$output_css .= "\t.b".$adrotate_config['adblock_disguise']."-".$id." { margin:".$admargin_top."px ".$admargin_right."px ".$admargin_bottom."px ".$admargin_left."px; }\n";
				unset($width_sum, $grid_width, $height_sum, $grid_height);
			}

			if($modus == 2) { // Wallpaper group
				$output_css .= "\t.g".$adrotate_config['adblock_disguise']."-".$id." { ".$grid_width.$group_align." }\n";
				$output_css .= "\t.b".$adrotate_config['adblock_disguise']."-".$id." { margin:".$admargin_top."px ".$admargin_right."px ".$admargin_bottom."px ".$admargin_left."px; }\n";
				unset($width_sum, $grid_width, $height_sum, $grid_height);
			}

			$group_css = get_option('adrotate_group_css');
			$group_css[$id] = $output_css;
			update_option('adrotate_group_css', $group_css);
			// End CSS

			wp_redirect('admin.php?page=adrotate-groups&status=201');
			exit;
		} else {
			wp_redirect('admin.php?page=adrotate-groups&status=500');
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_insert_schedule
 Purpose:   Prepare input form on saving new or updated schedules
 Since:		3.8.9
-------------------------------------------------------------*/
function adrotate_insert_schedule() {
	global $wpdb;

	if(wp_verify_nonce($_POST['adrotate_nonce'], 'adrotate_save_schedule')) {
		// Mandatory
		$id = $ad = '';
		if(isset($_POST['adrotate_id'])) $id = sanitize_key($_POST['adrotate_id']);
		if(isset($_POST['adrotate_schedulename'])) $name = sanitize_text_field(trim($_POST['adrotate_schedulename']));

		// Schedule variables
		$start_date = $start_hour = $start_minute = $end_date = $end_hour = $end_minute = '';
		if(isset($_POST['adrotate_start_date'])) $start_date = sanitize_key(trim($_POST['adrotate_start_date']));
		if(isset($_POST['adrotate_start_hour'])) $start_hour = sanitize_key(trim($_POST['adrotate_start_hour']));
		if(isset($_POST['adrotate_start_minute'])) $start_minute = sanitize_key(trim($_POST['adrotate_start_minute']));
		if(isset($_POST['adrotate_end_date'])) $end_date = sanitize_key(trim($_POST['adrotate_end_date']));
		if(isset($_POST['adrotate_end_hour'])) $end_hour = sanitize_key(trim($_POST['adrotate_end_hour']));
		if(isset($_POST['adrotate_end_minute'])) $end_minute = sanitize_key(trim($_POST['adrotate_end_minute']));

		$maxclicks = $maxshown = $spread = $spread_all = $autodelete = '';
		if(isset($_POST['adrotate_maxclicks'])) $maxclicks = sanitize_key(trim($_POST['adrotate_maxclicks']));
		if(isset($_POST['adrotate_maxshown'])) $maxshown = sanitize_key(trim($_POST['adrotate_maxshown']));
		if(isset($_POST['adrotate_spread'])) $spread = sanitize_key($_POST['adrotate_spread']);
		if(isset($_POST['adrotate_spread_all'])) $spread_all = sanitize_key($_POST['adrotate_spread_all']);
		if(isset($_POST['adrotate_autodelete'])) $autodelete = sanitize_key($_POST['adrotate_autodelete']);

		$start_day_hour = $start_day_minute = $end_day_hour = $end_day_minute = $day_mon = $day_tue = $day_wed = $day_thu = $day_fri = $day_sat = $day_sun = '';
		if(isset($_POST['adrotate_start_day_hour'])) $start_day_hour = sanitize_key(trim($_POST['adrotate_start_day_hour']));
		if(isset($_POST['adrotate_start_day_minute'])) $start_day_minute = sanitize_key(trim($_POST['adrotate_start_day_minute']));
		if(isset($_POST['adrotate_end_day_hour'])) $end_day_hour = sanitize_key(trim($_POST['adrotate_end_day_hour']));
		if(isset($_POST['adrotate_end_day_minute'])) $end_day_minute = sanitize_key(trim($_POST['adrotate_end_day_minute']));
		if(isset($_POST['adrotate_mon'])) $day_mon = sanitize_key($_POST['adrotate_mon']);
		if(isset($_POST['adrotate_tue'])) $day_tue = sanitize_key($_POST['adrotate_tue']);
		if(isset($_POST['adrotate_wed'])) $day_wed = sanitize_key($_POST['adrotate_wed']);
		if(isset($_POST['adrotate_thu'])) $day_thu = sanitize_key($_POST['adrotate_thu']);
		if(isset($_POST['adrotate_fri'])) $day_fri = sanitize_key($_POST['adrotate_fri']);
		if(isset($_POST['adrotate_sat'])) $day_sat = sanitize_key($_POST['adrotate_sat']);
		if(isset($_POST['adrotate_sun'])) $day_sun = sanitize_key($_POST['adrotate_sun']);

		$ads = '';
		if(isset($_POST['adselect'])) $ads = $_POST['adselect'];

		if(current_user_can('adrotate_schedule_manage')) {
			if(strlen($name) < 1) $name = 'Schedule '.$id;
			$name = str_replace('"', '', $name);

			// Sort out start dates
			if(strlen($start_date) > 0) {
				list($start_day, $start_month, $start_year) = explode('-', $start_date);
				$start_month = (!is_numeric($start_month)) ? date('m', strtotime($start_month.'-'.$start_year)) : $start_month; // Convert month to number
			} else {
				$start_year = $start_month = $start_day = 0;
			}

			if(($start_year > 0 AND $start_month > 0 AND $start_day > 0) AND strlen($start_hour) == 0) $start_hour = '00';
			if(($start_year > 0 AND $start_month > 0 AND $start_day > 0) AND strlen($start_minute) == 0) $start_minute = '00';

			if($start_month > 0 AND $start_day > 0 AND $start_year > 0) {
				$start_date = mktime($start_hour, $start_minute, 0, $start_month, $start_day, $start_year);
			} else {
				$start_date = 0;
			}

			// Sort out end dates
			if(strlen($end_date) > 0) {
				list($end_day, $end_month, $end_year) = explode('-', $end_date);
				$end_month = (!is_numeric($end_month)) ? date('m', strtotime($end_month.'-'.$end_year)) : $end_month; // Convert month to number
			} else {
				$end_year = $end_month = $end_day = 0;
			}

			if(($end_year > 0 AND $end_month > 0 AND $end_day > 0) AND strlen($end_hour) == 0) $end_hour = '00';
			if(($end_year > 0 AND $end_month > 0 AND $end_day > 0) AND strlen($end_minute) == 0) $end_minute = '00';

			if($end_month > 0 AND $end_day > 0 AND $end_year > 0) {
				$end_date = mktime($end_hour, $end_minute, 0, $end_month, $end_day, $end_year);
			} else {
				$end_date = 0;
			}

			// Enddate is too early, reset to default
			if($end_date <= $start_date) $end_date = $start_date + 7257600; // 84 days (12 weeks)

			// Sort out click and impressions restrictions
			if(strlen($maxclicks) < 1 OR !is_numeric($maxclicks)) $maxclicks = 0;
			if(strlen($maxshown) < 1 OR !is_numeric($maxshown))	$maxshown = 0;

			// Impression Spread
			if(isset($spread) AND strlen($spread) != 0) {
				$spread = 'Y';
			} else {
				$spread = 'N';
			}
			if(isset($spread_all) AND strlen($spread_all) != 0) {
				$spread_all = 'Y';
			} else {
				$spread_all = 'N';
			}

			// Auto delete
			if(isset($autodelete) AND strlen($autodelete) != 0) {
				$autodelete = 'Y';
			} else {
				$autodelete = 'N';
			}

			// Day schedules
			if(!isset($start_day_hour) AND $start_day_hour < 1) $start_day_hour = '00';
			if(!isset($start_day_minute) AND $start_day_minute < 1) $start_day_minute = '00';
			$day_start = str_pad($start_day_hour, 2, 0, STR_PAD_LEFT).str_pad($start_day_minute, 2, 0, STR_PAD_LEFT);

			if(!isset($end_day_hour) AND $end_day_hour < 1) $end_day_hour = '00';
			if(!isset($end_day_minute) AND $end_day_minute < 1) $end_day_minute = '00';
			$day_stop = str_pad($end_day_hour, 2, 0, STR_PAD_LEFT).str_pad($end_day_minute, 2, 0, STR_PAD_LEFT);

			if(isset($day_mon) AND strlen($day_mon) != 0) $day_mon = 'Y';
				else $day_mon = 'N';
			if(isset($day_tue) AND strlen($day_tue) != 0) $day_tue = 'Y';
				else $day_tue = 'N';
			if(isset($day_wed) AND strlen($day_wed) != 0) $day_wed = 'Y';
				else $day_wed = 'N';
			if(isset($day_thu) AND strlen($day_thu) != 0) $day_thu = 'Y';
				else $day_thu = 'N';
			if(isset($day_fri) AND strlen($day_fri) != 0) $day_fri = 'Y';
				else $day_fri = 'N';
			if(isset($day_sat) AND strlen($day_sat) != 0) $day_sat = 'Y';
				else $day_sat = 'N';
			if(isset($day_sun) AND strlen($day_sun) != 0) $day_sun = 'Y';
				else $day_sun = 'N';

			// Fetch records for the schedule
			$linkmeta = $wpdb->get_results($wpdb->prepare("SELECT `ad` FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `schedule` = %d AND `user` = 0;", $id));
			foreach($linkmeta as $meta) {
				$meta_array[] = $meta->ad;
			}

			if(empty($meta_array)) $meta_array = array();
			if(empty($ads)) $ads = array();

			// Add new ads to this schedule
			$insert = array_diff($ads, $meta_array);
			foreach($insert as &$value) {
				$wpdb->insert($wpdb->prefix.'adrotate_linkmeta', array('ad' => $value, 'group' => 0, 'user' => 0, 'schedule' => $id));
			}
			unset($value);

			// Remove ads from this schedule
			$delete = array_diff($meta_array, $ads);
			foreach($delete as &$value) {
				$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `group` = 0 AND `user` = 0 AND `schedule` = %d;", $value, $id));
			}
			unset($value);

			// Save the schedule to the DB
			$wpdb->update($wpdb->prefix.'adrotate_schedule', array('name' => $name, 'starttime' => $start_date, 'stoptime' => $end_date, 'maxclicks' => $maxclicks, 'maximpressions' => $maxshown, 'spread' => $spread, 'spread_all' => $spread_all, 'daystarttime' => $day_start, 'daystoptime' => $day_stop, 'day_mon' => $day_mon, 'day_tue' => $day_tue, 'day_wed' => $day_wed, 'day_thu' => $day_thu, 'day_fri' => $day_fri, 'day_sat' => $day_sat, 'day_sun' => $day_sun, 'autodelete' => $autodelete), array('id' => $id));

			wp_redirect('admin.php?page=adrotate-schedules&status=217');
			exit;
		} else {
			wp_redirect('admin.php?page=adrotate-schedules&status=500');
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_insert_media
 Purpose:   Upload a file to the banners folder.
 Since:		?
-------------------------------------------------------------*/
function adrotate_insert_media() {
	global $adrotate_config;

	if(wp_verify_nonce($_POST['adrotate_nonce'], 'adrotate_save_media')) {
		if(current_user_can('adrotate_ad_manage')) {

			if($_FILES['adrotate_image']['size'] > 0 AND $_FILES['adrotate_image']['size'] <= 512000) {
				$file_path = WP_CONTENT_DIR."/".esc_attr($_POST['adrotate_image_location'])."/";

				$file = pathinfo(sanitize_file_name($_FILES['adrotate_image']['name']));
				$file['mimetype'] = mime_content_type($_FILES['adrotate_image']['tmp_name']);

				// Everyone can upload these
				$allowed_extensions = array('jpg', 'jpeg', 'gif', 'png', 'svg', 'webp');
				$allowed_mimetypes = array('image/jpg', 'image/jpeg', 'image/gif', 'image/png', 'image/svg', 'image/webp');

				if(current_user_can('unfiltered_html')) {
					// Higher level and enabled users can also upload these.
					$allowed_extensions = array_merge($allowed_extensions, array('html', 'htm', 'js', 'zip'));
					$allowed_mimetypes = array_merge($allowed_mimetypes, array('text/html', 'text/htm', 'application/x-javascript', 'application/javascript', 'text/javascript', 'application/zip'));
				}

				if(in_array($file['extension'], $allowed_extensions) AND in_array($file['mimetype'], $allowed_mimetypes)) {
					if ($_FILES['adrotate_image']['error'] > 0) {
						if($_FILES['adrotate_image']['error'] == 1 OR $_FILES['adrotate_image']['error'] == 2) {
							$errorcode = 511;
						} else if($_FILES['adrotate_image']['error'] == 3) {
							$errorcode = 506;
						} else if($_FILES['adrotate_image']['error'] == 4) {
							$errorcode = 506;
						} else if($_FILES['adrotate_image']['error'] == 6 OR $_FILES['adrotate_image']['error'] == 7) {
							$errorcode = 506;
						} else {
							$errorcode = '';
						}

						wp_redirect('admin.php?page=adrotate-media&status='.$errorcode); // Other error
					} else {
						// Rename file if it already exists
						if(file_exists($file_path.$file['filename'].'.'.$file['extension'])) {
						    $file['filename'] = $file['filename'].'_'.rand(1000, 9999);
						}

						// Move file into place
						if(!move_uploaded_file($_FILES['adrotate_image']['tmp_name'], $file_path.$file['filename'].'.'.$file['extension'])) {
							wp_redirect('admin.php?page=adrotate-media&status=506'); // Upload error
						}

						if(($file['mimetype'] == 'application/zip' OR $file['mimetype'] == 'application/zip-compressed' OR $file['mimetype'] == 'application/x-zip-compressed') AND $file['extension'] == 'zip') {
							require_once(ABSPATH .'/wp-admin/includes/file.php');

							$creds = request_filesystem_credentials(wp_nonce_url('admin.php?page=adrotate-media'), '', false, $file_path, null);
						    if(!WP_Filesystem($creds)) {
								$creds = request_filesystem_credentials(wp_nonce_url('admin.php?page=adrotate-media'), '', true, $file_path, null);
							}

							// Unzip file to a folder based on the filename without extension
							$unzipfile = unzip_file($file_path.$file['filename'].'.'.$file['extension'], $file_path.$file['filename']);
							if(is_wp_error($unzipfile)) {
								wp_redirect('admin.php?page=adrotate-media&status=512&error='.$unzipfile->get_error_message()); // Can not unzip file
								exit;

							}

							// Delete unwanted files
							adrotate_clean_folder_contents($file_path.$file['filename']);

							// Delete the uploaded zip
							adrotate_unlink($file['basename']);
						}

						wp_redirect('admin.php?page=adrotate-media&status=202'); // Success
						exit;
					}
				} else {
					wp_redirect('admin.php?page=adrotate-media&status=510'); // Filetype
					exit;
				}
			} else {
				wp_redirect('admin.php?page=adrotate-media&status=511'); // Size
				exit;
			}
		} else {
			wp_redirect('admin.php?page=adrotate-media&status=500'); // No access/permission
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_insert_folder
 Purpose:   Create a folder
 Since:		5.8.4
-------------------------------------------------------------*/
function adrotate_insert_folder() {
	global $adrotate_config;

	if(wp_verify_nonce($_POST['adrotate_nonce'], 'adrotate_save_media')) {
		if(current_user_can('adrotate_ad_manage')) {

			$folder = (isset($_POST['adrotate_folder'])) ? esc_attr(strip_tags(trim($_POST['adrotate_folder']))) : '';

			if(strlen($folder) > 0 and strlen($folder) <= 100) {
				$folder = WP_CONTENT_DIR."/".$adrotate_config['banner_folder']."/".$folder;

				if(is_dir($folder)) {
					$folder = $folder.'_'.rand(1000, 9999);
				}

				if(wp_mkdir_p($folder)) {
					wp_redirect('admin.php?page=adrotate-media&status=223'); // Success
					exit;
				} else {
					wp_redirect('admin.php?page=adrotate-media&status=516'); // error
					exit;
				}
			} else {
				wp_redirect('admin.php?page=adrotate-media&status=515'); // name length
				exit;
			}
		} else {
			wp_redirect('admin.php?page=adrotate-media&status=500'); // No access/permission
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_insert_advertiser
 Purpose:   Create a new advertiser
 Since:		4.0
-------------------------------------------------------------*/
function adrotate_insert_advertiser() {
	global $wpdb;

	if(wp_verify_nonce($_POST['adrotate_nonce'], 'adrotate_save_advertiser')) {
		// Mandatory
		$id = $notes = $can_edit = $can_mobile = $can_geo = '';
		if(isset($_POST['adrotate_user'])) $user = sanitize_key($_POST['adrotate_user']);
		if(isset($_POST['adrotate_notes'])) $notes = sanitize_text_field(trim($_POST['adrotate_notes']));
		if(isset($_POST['adrotate_can_edit'])) $can_edit = strip_tags(trim($_POST['adrotate_can_edit']));
		if(isset($_POST['adrotate_can_mobile'])) $can_mobile = strip_tags(trim($_POST['adrotate_can_mobile']));
		if(isset($_POST['adrotate_can_geo'])) $can_geo = strip_tags(trim($_POST['adrotate_can_geo']));

		if(current_user_can('adrotate_advertiser_manage')) {
			// User
			if(!is_numeric($user)) $user = 0;

			// Premissions
			if(isset($can_edit) AND strlen($can_edit) != 0) {
				$can_edit = 'Y';
			} else {
				$can_edit = 'N';
			}
			if(isset($can_mobile) AND strlen($can_mobile) != 0) {
				$can_mobile = 'Y';
			} else {
				$can_mobile = 'N';
			}
			if(isset($can_geo) AND strlen($can_geo) != 0) {
				$can_geo = 'Y';
			} else {
				$can_geo = 'N';
			}
			update_user_meta($user, 'adrotate_permissions', array('edit' => $can_edit, 'mobile' => $can_mobile, 'geo' => $can_geo));
			update_user_meta($user, 'adrotate_notes', $notes);

			wp_redirect('admin.php?page=adrotate-advertisers&status=205&view=profile&user='.$user);
			exit;
		} else {
			wp_redirect('admin.php?page=adrotate-advertisers&status=500');
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_request_action
 Purpose:   Prepare action for banner or group from database
 Since:		2.2
-------------------------------------------------------------*/
function adrotate_request_action() {
	global $adrotate_config;

	$banner_ids = $group_ids = $schedule_ids = $actions = '';

	if(wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_ads_active')
	OR wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_ads_disable')
	OR wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_ads_error')
	OR wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_ads_queue')
	OR wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_ads_reject')
	OR wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_ads_archive')
	OR wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_ads_trash')
	OR wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_groups')
	OR wp_verify_nonce($_POST['adrotate_nonce'],'adrotate_bulk_schedules')) {
		if(!empty($_POST['adrotate_id'])) $banner_ids = array($_POST['adrotate_id']);
		if(!empty($_POST['bannercheck'])) $banner_ids = $_POST['bannercheck'];
		if(!empty($_POST['groupcheck'])) $group_ids = $_POST['groupcheck'];
		if(!empty($_POST['schedulecheck'])) $schedule_ids = $_POST['schedulecheck'];
		if(!empty($_POST['adrotate_action'])) $actions = strip_tags(trim($_POST['adrotate_action']));

		if(preg_match('/-/', $actions)) {
			list($action, $specific) = explode('-', $actions);
		} else {
		   	$action = $actions;
		}

		if($banner_ids != '') {
			if($action == 'export') {
				if(current_user_can('adrotate_ad_manage')) {
					adrotate_export($banner_ids);
					$status = array('status' => 215);
				} else {
					$status = array('status' => 500);
				}

				wp_redirect('admin.php?page=adrotate&'.http_build_query($status));
				exit;
			}

			foreach($banner_ids as $banner_id) {
				if($action == 'duplicate' AND current_user_can('adrotate_ad_manage')) {
					adrotate_duplicate($banner_id, 'banner');
					adrotate_evaluate_ads();
					$status = array('status' => 219);
				} else if($action == 'deactivate' AND current_user_can('adrotate_ad_manage')) {
					adrotate_active($banner_id, 'deactivate');
					$status = array('status' => 210);
				} else if($action == 'activate' AND current_user_can('adrotate_ad_manage')) {
					adrotate_active($banner_id, 'activate');
					adrotate_evaluate_ads();
					$status = array('status' => 211);
				} else if($action == 'archive' AND current_user_can('adrotate_ad_manage')) {
					adrotate_active($banner_id, 'archive');
					$status = array('status' => 220);
				} else if($action == 'trash' AND current_user_can('adrotate_ad_delete')) {
					adrotate_delete($banner_id, 'trash');
					$status = array('status' => 221);
				} else if($action == 'restore' AND current_user_can('adrotate_ad_delete')) {
					adrotate_active($banner_id, 'restore');
					adrotate_evaluate_ads();
					$status = array('status' => 222, 'view' => 'trash');
				} else if($action == 'delete' AND current_user_can('adrotate_ad_delete')) {
					adrotate_delete($banner_id, 'banner');
					$status = array('status' => 203);
				} else if($action == 'reset' AND current_user_can('adrotate_ad_delete')) {
					adrotate_reset($banner_id);
					$status = array('status' => 208);
				} else if($action == 'renew' AND current_user_can('adrotate_ad_manage')) {
					adrotate_renew($banner_id, $specific);
					adrotate_evaluate_ads();
					$status = array('status' => 209);
				} else if($action == 'weight' AND current_user_can('adrotate_ad_manage')) {
					adrotate_weight($banner_id, $specific);
					$status = array('status' => 214);
				} else if($action == 'approve' AND current_user_can('adrotate_moderate_approve')) {
					adrotate_approve($banner_id);
					$status = array('status' => 304, 'view' => 'queue');
				} else if($action == 'reject' AND current_user_can('adrotate_moderate')) {
					adrotate_reject($banner_id);
					$status = array('status' => 305, 'view' => 'queue');
				} else if($action == 'queue' AND current_user_can('adrotate_moderate')) {
					adrotate_queue($banner_id);
					$status = array('status' => 306, 'view' => 'queue');
				} else {
					$status = array('status' => 500);
				}
			}

			wp_redirect('admin.php?page=adrotate&'.http_build_query($status));
			exit;
		}

		if($group_ids != '') {
			foreach($group_ids as $group_id) {
				if($action == 'group_duplicate' AND current_user_can('adrotate_group_manage')) {
					adrotate_duplicate($group_id, 'group');
					$status = array('status' => 207);
				} else if($action == 'group_delete' AND current_user_can('adrotate_group_delete')) {
					adrotate_delete($group_id, 'group');
					$status = array('status' => 204);
				} else if($action == 'group_delete_banners' AND current_user_can('adrotate_group_delete')) {
					adrotate_delete($group_id, 'bannergroup');
					$status = array('status' => 213);
				} else {
					$status = array('status' => 500);
				}
			}

			wp_redirect('admin.php?page=adrotate-groups&'.http_build_query($status));
			exit;
		}

		if($schedule_ids != '') {
			foreach($schedule_ids as $schedule_id) {
				if($action == 'schedule_duplicate' AND current_user_can('adrotate_schedule_manage')) {
					adrotate_duplicate($schedule_id, 'schedule');
					$status = array('status' => 224);
				} else if($action == 'schedule_delete' AND current_user_can('adrotate_schedule_delete')) {
					adrotate_delete($schedule_id, 'schedule');
					$status = array('status' => 218);
				} else {
					$status = array('status' => 500);
				}
			}

			wp_redirect('admin.php?page=adrotate-schedules&'.http_build_query($status));
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_duplicate
 Purpose:   Duplicate a banner
 Since:		3.17
-------------------------------------------------------------*/
function adrotate_duplicate($id, $what) {
	global $wpdb;

	if($id > 0) {
		$thetime = current_time('timestamp');

		if($what == 'banner') {
			$duplicate_id = $wpdb->get_row("SELECT * FROM `{$wpdb->prefix}adrotate` WHERE `id` = {$id};");

			$wpdb->insert($wpdb->prefix.'adrotate', array(
				'title' => $duplicate_id->title.' (Copy of #'.$id.')',
				'bannercode' => $duplicate_id->bannercode,
				'thetime' => $thetime,
				'updated' => $thetime,
				'author' => $duplicate_id->author,
				'imagetype' => $duplicate_id->imagetype,
				'image' => $duplicate_id->image,
				'tracker' => $duplicate_id->tracker,
				'show_everyone' => $duplicate_id->show_everyone,
				'desktop' => $duplicate_id->desktop,
				'mobile' => $duplicate_id->mobile,
				'tablet' => $duplicate_id->tablet,
				'os_ios' => $duplicate_id->os_ios,
				'os_android' => $duplicate_id->os_android,
				'type' => $duplicate_id->type,
				'weight' => $duplicate_id->weight,
				'autodelete' => $duplicate_id->autodelete,
				'autodisable' => $duplicate_id->autodisable,
				'budget' => $duplicate_id->budget,
				'crate' => $duplicate_id->crate,
				'irate' => $duplicate_id->irate,
				'state_req' => $duplicate_id->state_req,
				'cities' => $duplicate_id->cities,
				'countries' => $duplicate_id->countries
			));
			$new_id = $wpdb->insert_id;

			$duplicate_meta = $wpdb->get_results("SELECT * FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = {$id};");
			foreach($duplicate_meta as $meta) {
				$schedule = ($meta->schedule > 0) ? $meta->schedule : 0;
				$group = ($meta->group > 0) ? $meta->group : 0;
				$user = ($meta->user > 0) ? $meta->user : 0;
				$wpdb->insert("{$wpdb->prefix}adrotate_linkmeta", array(
					'ad' => $new_id,
					'group' => $group,
					'user' => $user,
					'schedule' => $schedule
				));

				unset($schedule, $group, $user);
			}
		} else if($what == 'group') {
			$duplicate_id = $wpdb->get_row("SELECT * FROM `{$wpdb->prefix}adrotate_groups` WHERE `id` = {$id};");

			$wpdb->insert($wpdb->prefix.'adrotate_groups', array(
				'name' => $duplicate_id->name.' (Copy of #'.$id.')',
				'modus' => $duplicate_id->modus,
				'fallback' => $duplicate_id->fallback,
				'cat' => $duplicate_id->cat,
				'cat_loc' => $duplicate_id->cat_loc,
				'cat_par' => $duplicate_id->cat_par,
				'page' => $duplicate_id->page,
				'page_loc' => $duplicate_id->page_loc,
				'page_par' => $duplicate_id->page_par,
				'woo_cat' => $duplicate_id->woo_cat,
				'woo_loc' => $duplicate_id->woo_loc,
				'bbpress' => $duplicate_id->bbpress,
				'bbpress_loc' => $duplicate_id->bbpress_loc,
				'mobile' => $duplicate_id->mobile,
				'geo' => $duplicate_id->geo,
				'wrapper_before' => $duplicate_id->wrapper_before,
				'wrapper_after' => $duplicate_id->wrapper_after,
				'align' => $duplicate_id->align,
				'gridrows' => $duplicate_id->gridrows,
				'gridcolumns' => $duplicate_id->gridcolumns,
				'admargin' => $duplicate_id->admargin,
				'admargin_bottom' => $duplicate_id->admargin_bottom,
				'admargin_left' => $duplicate_id->admargin_left,
				'admargin_right' => $duplicate_id->admargin_right,
				'adwidth' => $duplicate_id->adwidth,
				'adheight' => $duplicate_id->adheight,
				'adspeed' => $duplicate_id->adspeed,
				'repeat_impressions' => $duplicate_id->repeat_impressions
			));
	    	$new_id = $wpdb->insert_id;

			$duplicate_meta = $wpdb->get_results("SELECT * FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `group` = {$id};");
			foreach($duplicate_meta as $meta) {
				$ad_id = ($meta->ad > 0) ? $meta->ad : 0;
				$wpdb->insert("{$wpdb->prefix}adrotate_linkmeta", array(
					'ad' => $ad_id,
					'group' => $new_id,
					'user' => 0,
					'schedule' => 0
				));

				unset($ad_id);
			}
		} else if($what == 'schedule') {
			$duplicate_id = $wpdb->get_row("SELECT * FROM `{$wpdb->prefix}adrotate_schedule` WHERE `id` = {$id};");

			$wpdb->insert($wpdb->prefix.'adrotate_schedule', array(
				'name' => $duplicate_id->name.' (Copy of #'.$id.')',
				'starttime' => $duplicate_id->starttime,
				'stoptime' => $duplicate_id->stoptime,
				'maxclicks' => $duplicate_id->maxclicks,
				'maximpressions' => $duplicate_id->maximpressions,
				'spread' => $duplicate_id->spread,
				'spread_all' => $duplicate_id->spread_all,
				'daystarttime' => $duplicate_id->daystarttime,
				'daystoptime' => $duplicate_id->daystoptime,
				'day_mon' => $duplicate_id->day_mon,
				'day_tue' => $duplicate_id->day_tue,
				'day_wed' => $duplicate_id->day_wed,
				'day_thu' => $duplicate_id->day_thu,
				'day_fri' => $duplicate_id->day_fri,
				'day_sat' => $duplicate_id->day_sat,
				'day_sun' => $duplicate_id->day_sun,
				'autodelete' => $duplicate_id->autodelete
			));
	    	$new_id = $wpdb->insert_id;

			$duplicate_meta = $wpdb->get_results("SELECT * FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `schedule` = {$id};");
			foreach($duplicate_meta as $meta) {
				$ad_id = ($meta->ad > 0) ? $meta->ad : 0;
				$wpdb->insert("{$wpdb->prefix}adrotate_linkmeta", array(
					'ad' => $ad_id,
					'group' => 0,
					'user' => 0,
					'schedule' => $new_id
				));

				unset($ad_id);
			}
		}
	}
}


/*-------------------------------------------------------------
 Name:      adrotate_delete
 Purpose:   Remove banner or group from database
 Since:		0.1
-------------------------------------------------------------*/
function adrotate_delete($id, $what) {
	global $wpdb;

	$now = current_time('timestamp');

	if($id > 0) {
		if($what == 'banner') {
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate` WHERE `id` = %d;", $id));
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d;", $id));
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_stats` WHERE `ad` = %d;", $id));
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_stats_archive` WHERE `ad` = %d;", $id));
		} else if($what == 'trash') {
			$wpdb->update("{$wpdb->prefix}adrotate", array('type' => 'trash', 'updated' => $now), array('id' => $id));
		} else if($what == 'group') {
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_groups` WHERE `id` = %d;", $id));
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `group` = %d;", $id));
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_stats` WHERE `group` = %d;", $id));
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_stats_archive` WHERE `group` = %d;", $id));
			$wpdb->update($wpdb->prefix.'adrotate_groups', array('fallback' => 0), array('fallback' => $id));

			// Remove CSS from group
			$group_css = get_option('adrotate_group_css');
			unset($group_css[$id]);
			update_option('adrotate_group_css', $group_css);
		} else if($what == 'bannergroup') {
			$linkmeta = $wpdb->get_results($wpdb->prepare("SELECT `ad` FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `group` = %d AND `user` = '0' AND `schedule` = '0';", $id));
			foreach($linkmeta as $meta) {
				adrotate_delete($meta->ad, 'trash');
			}
			unset($linkmeta);
			adrotate_delete($id, 'group');
		} else if($what == 'schedule') {
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_schedule` WHERE `id` = %d;", $id));
			$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `schedule` = %d;", $id));
		}
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_active
 Purpose:   Activate or Deactivate a banner
 Since:		0.1
-------------------------------------------------------------*/
function adrotate_active($id, $what) {
	global $wpdb, $adrotate_config;

	if($id > 0) {
		if($what == 'deactivate') {
			$wpdb->update("{$wpdb->prefix}adrotate", array('type' => 'disabled'), array('id' => $id));
		} else if($what == 'activate') {
			// Determine status of ad
			$adstate = adrotate_evaluate_ad($id);
			$adtype = ($adstate == 'error' OR $adstate == 'expired') ? 'error' : 'active';

			$wpdb->update("{$wpdb->prefix}adrotate", array('type' => $adtype), array('id' => $id));
		} else if($what == 'archive') {
			$wpdb->update("{$wpdb->prefix}adrotate", array('type' => 'archived'), array('id' => $id));
			if($adrotate_config['stats'] == 1) {
				adrotate_archive_stats($id);
			}
		} else if($what == 'restore') {
			// Determine status of ad
			$updated = current_time('timestamp');
			$adstate = adrotate_evaluate_ad($id);
			$adtype = ($adstate == 'error' OR $adstate == 'expired') ? 'error' : 'active';

			$wpdb->update("{$wpdb->prefix}adrotate", array('type' => $adtype, 'updated' => $updated), array('id' => $id));
		}
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_reset
 Purpose:   Reset statistics for a banner
 Since:		2.2
-------------------------------------------------------------*/
function adrotate_reset($id) {
	global $wpdb;

	if($id > 0) {
		$wpdb->query($wpdb->prepare("DELETE FROM `{$wpdb->prefix}adrotate_stats` WHERE `ad` = %d;", $id));
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_renew
 Purpose:   Renew the end date of a banner with a new schedule starting where the last ended
 Since:		2.2
-------------------------------------------------------------*/
function adrotate_renew($id, $howlong = 2592000) {
	global $wpdb;

	if($id > 0) {
		$schedule_id = $wpdb->get_var($wpdb->prepare("SELECT `schedule` FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = %d AND `group` = 0 AND `user` = 0 ORDER BY `id` DESC LIMIT 1;", $id));
		if($schedule_id > 0) {
			$schedule = $wpdb->get_row($wpdb->prepare("SELECT * FROM `{$wpdb->prefix}adrotate_schedule` WHERE `id` = %d ORDER BY `id` DESC LIMIT 1;", $schedule_id));
			$stoptime = $schedule->stoptime + $howlong;
			$wpdb->insert($wpdb->prefix.'adrotate_schedule', array('name' => 'Schedule for ad '.$id, 'starttime' => $schedule->stoptime, 'stoptime' => $stoptime, 'maxclicks' => $schedule->maxclicks, 'maximpressions' => $schedule->maximpressions, 'spread' => $schedule->spread, 'spread_all' => $schedule->spread_all, 'daystarttime' => $schedule->daystarttime, 'daystoptime' => $schedule->daystoptime, 'day_mon' => $schedule->day_mon, 'day_tue' => $schedule->day_tue, 'day_wed' => $schedule->day_wed, 'day_thu' => $schedule->day_thu, 'day_fri' => $schedule->day_fri, 'day_sat' => $schedule->day_sat, 'day_sun' => $schedule->day_sun));
		} else {
			$now = current_time('timestamp');
			$stoptime = $now + $howlong;
			$wpdb->insert($wpdb->prefix.'adrotate_schedule', array('name' => 'Schedule for ad '.$id, 'starttime' => $now, 'stoptime' => $stoptime, 'maxclicks' => 0, 'maximpressions' => 0, 'spread' => 'N', 'spread_all' => 'N', 'daystarttime' => '0000', 'daystoptime' => '0000', 'day_mon' => 'Y', 'day_tue' => 'Y', 'day_wed' => 'Y', 'day_thu' => 'Y', 'day_fri' => 'Y', 'day_sat' => 'Y', 'day_sun' => 'Y'));
		}
		$wpdb->insert($wpdb->prefix.'adrotate_linkmeta', array('ad' => $id, 'group' => 0, 'user' => 0, 'schedule' => $wpdb->insert_id));
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_weight
 Purpose:   Renew the end date of a banner
 Since:		3.6
-------------------------------------------------------------*/
function adrotate_weight($id, $weight = 6) {
	global $wpdb;

	if($id > 0) {
		$wpdb->update($wpdb->prefix.'adrotate', array('weight' => $weight), array('id' => $id));
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_approve
 Purpose:   Approve a queued banner
 Since:		3.8.4
-------------------------------------------------------------*/
function adrotate_approve($id) {
	global $wpdb;

	if($id > 0) {
		$wpdb->update($wpdb->prefix.'adrotate', array('type' => 'active'), array('id' => $id));
		adrotate_notifications('approved', $id);
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_reject
 Purpose:   Reject a queued banner
 Since:		3.8.4
-------------------------------------------------------------*/
function adrotate_reject($id) {
	global $wpdb;

	if($id > 0) {
		$wpdb->update($wpdb->prefix.'adrotate', array('type' => 'reject'), array('id' => $id));
		adrotate_notifications('rejected', $id);
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_queue
 Purpose:   Queue a rejected banner
 Since:		3.8.4
-------------------------------------------------------------*/
function adrotate_queue($id) {
	global $wpdb;

	if($id > 0) {
		$wpdb->update($wpdb->prefix.'adrotate', array('type' => 'queue'), array('id' => $id));
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_export
 Purpose:   Export selected banners
 Since:		3.8.5
-------------------------------------------------------------*/
function adrotate_export($ids) {
	if(is_array($ids)) {
		adrotate_export_ads($ids);
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_save_options
 Purpose:   Save options from dashboard
 Since:		0.1
-------------------------------------------------------------*/
function adrotate_save_options() {
	if(wp_verify_nonce($_POST['adrotate_nonce_settings'],'adrotate_settings')) {

		$settings_tab = esc_attr($_POST['adrotate_settings_tab']);

		if($settings_tab == 'general') {
			$config = get_option('adrotate_config');

			$duplicate_adverts_filter = $textwidget_shortcodes = $live_preview = $jquery = $jsfooter = '';
			if(isset($_POST['adrotate_duplicate_adverts_filter'])) $duplicate_adverts_filter = sanitize_text_field($_POST['adrotate_duplicate_adverts_filter']);
			if(isset($_POST['adrotate_textwidget_shortcodes'])) $textwidget_shortcodes = sanitize_text_field($_POST['adrotate_textwidget_shortcodes']);
			if(isset($_POST['adrotate_live_preview'])) $live_preview = sanitize_text_field($_POST['adrotate_live_preview']);
			if(isset($_POST['adrotate_jquery'])) $jquery = sanitize_text_field($_POST['adrotate_jquery']);
			if(isset($_POST['adrotate_jsfooter'])) $jsfooter = sanitize_text_field($_POST['adrotate_jsfooter']);

			$adblock_disguise = $banner_folder = $report_folder = $adstxt_file = $crawlers = '';
			if(isset($_POST['adrotate_adblock_disguise'])) $adblock_disguise = sanitize_text_field($_POST['adrotate_adblock_disguise']);
			if(isset($_POST['adrotate_banner_folder'])) $banner_folder = sanitize_text_field($_POST['adrotate_banner_folder']);
			if(isset($_POST['adrotate_report_folder'])) $report_folder = sanitize_text_field($_POST['adrotate_report_folder']);
			if(isset($_POST['adrotate_adstxt_file'])) $adstxt_file = sanitize_text_field($_POST['adrotate_adstxt_file']);
			if(isset($_POST['adrotate_crawlers'])) $crawlers = sanitize_text_field($_POST['adrotate_crawlers']);

			$config['duplicate_adverts_filter'] = ($duplicate_adverts_filter == 'Y') ? 'Y' : 'N';
			$config['textwidget_shortcodes'] = ($textwidget_shortcodes == 'Y') ? 'Y' : 'N';
			$config['live_preview'] = ($live_preview == 'Y') ? 'Y' : 'N';
			$config['jquery'] = ($jquery == 'Y') ? 'Y' : 'N';
			$config['jsfooter'] = ($jsfooter == 'Y') ? 'Y' : 'N';

			$config['adblock_disguise'] = (strlen($adblock_disguise) > 0) ? preg_replace('/[^a-z]/', '', strtolower(substr($adblock_disguise, 0, 10))) : '';
			$config['banner_folder'] = (strlen($banner_folder) > 0) ? preg_replace('/[^a-zA-Z0-9\/\-_]/', '', $banner_folder) : 'banners';
			$config['report_folder'] = (strlen($report_folder) > 0) ? preg_replace('/[^a-zA-Z0-9\/\-_]/', '', $report_folder) : 'reports';
			$config['adstxt_file'] = (strlen($adstxt_file) > 0) ? preg_replace('/[^a-zA-Z0-9\/\-_]/', '', $adstxt_file) : '';

			// Mabe add a trailing slash
			if(strlen($config['adstxt_file']) > 0 AND substr($config['adstxt_file'], -1) != '/') {
				$config['adstxt_file'] = $config['adstxt_file'].'/';
			}
			update_option('adrotate_config', $config);

			// Sort out crawlers
			$crawlers = explode(',', $crawlers);
			$new_crawlers = array();
			foreach($crawlers as $crawler) {
				$crawler = preg_replace('/[^a-zA-Z0-9\[\]\-_:; ]/i', '', trim($crawler));
				if(strlen($crawler) > 0) $new_crawlers[] = $crawler;
			}
			update_option('adrotate_crawlers', $new_crawlers);

			// Update Group CSS if Adblock Disguise changes
			$adblock_disguise_current = $config['adblock_disguise'];

			if($adblock_disguise_current != $adblock_disguise) {
				$group_css = get_option('adrotate_group_css');
				if(count($group_css) > 0) {
					foreach($group_css as $id => $css) {
						$group_css_updated[$id] = str_replace($adblock_disguise_current, $adblock_disguise, $css);
					}
					update_option('adrotate_group_css', $group_css_updated);
				}
			}
		}

		if($settings_tab == 'notifications') {
			$notifications = get_option('adrotate_notifications');

			$notification_dash = $notification_dash_expired = $notification_dash_soon = $notification_dash_week = $notification_dash_schedules = '';
			if(isset($_POST['adrotate_notification_dash'])) $notification_dash = sanitize_text_field($_POST['adrotate_notification_dash']);
			if(isset($_POST['adrotate_notification_dash_expired'])) $notification_dash_expired = sanitize_text_field($_POST['adrotate_notification_dash_expired']);
			if(isset($_POST['adrotate_notification_dash_soon'])) $notification_dash_soon = sanitize_text_field($_POST['adrotate_notification_dash_soon']);
			if(isset($_POST['adrotate_notification_dash_week'])) $notification_dash_week = sanitize_text_field($_POST['adrotate_notification_dash_week']);
			if(isset($_POST['adrotate_notification_schedules'])) $notification_dash_schedules = sanitize_text_field($_POST['adrotate_notification_schedules']);

			$notification_mail = $notification_mail_geo = $notification_mail_status = $notification_mail_queue = $notification_mail_approved = $notification_mail_rejected = $notification_emails = '';
			if(isset($_POST['adrotate_notification_mail'])) $notification_mail = sanitize_text_field($_POST['adrotate_notification_mail']);
			if(isset($_POST['adrotate_notification_mail_geo'])) $notification_mail_geo = sanitize_text_field($_POST['adrotate_notification_mail_geo']);
			if(isset($_POST['adrotate_notification_mail_status'])) $notification_mail_status = sanitize_text_field($_POST['adrotate_notification_mail_status']);
			if(isset($_POST['adrotate_notification_mail_queue'])) $notification_mail_queue = sanitize_text_field($_POST['adrotate_notification_mail_queue']);
			if(isset($_POST['adrotate_notification_mail_approved'])) $notification_mail_approved = sanitize_text_field($_POST['adrotate_notification_mail_approved']);
			if(isset($_POST['adrotate_notification_mail_rejected'])) $notification_mail_rejected = sanitize_text_field($_POST['adrotate_notification_mail_rejected']);
			if(isset($_POST['adrotate_notification_mail_publisher'])) $notification_emails = sanitize_text_field($_POST['adrotate_notification_mail_publisher']);

			$notifications['notification_dash'] = ($notification_dash == 'Y') ? 'Y' : 'N';
			$notifications['notification_dash_expired'] = ($notification_dash_expired == 'Y') ? 'Y' : 'N';
			$notifications['notification_dash_soon'] = ($notification_dash_soon == 'Y') ? 'Y' : 'N';
			$notifications['notification_dash_week'] = ($notification_dash_week == 'Y') ? 'Y' : 'N';
			$notifications['notification_schedules'] = ($notification_dash_schedules == 'Y') ? 'Y' : 'N';

			$notifications['notification_email'] = ($notification_mail == 'Y') ? 'Y' : 'N';
			$notifications['notification_mail_geo'] = ($notification_mail_geo == 'Y') ? 'Y' : 'N';
			$notifications['notification_mail_status'] = ($notification_mail_status == 'Y') ? 'Y' : 'N';
			$notifications['notification_mail_queue'] = ($notification_mail_queue == 'Y') ? 'Y' : 'N';
			$notifications['notification_mail_approved'] = ($notification_mail_approved == 'Y') ? 'Y' : 'N';
			$notifications['notification_mail_rejected'] = ($notification_mail_rejected == 'Y') ? 'Y' : 'N';

			if(strlen($notification_emails) > 0) {
				$notification_emails = explode(',', trim($notification_emails));
				foreach($notification_emails as $notification_email) {
					$notification_email = trim($notification_email);
					if(strlen($notification_email) > 0) {
		  				if(is_email($notification_email) ) {
							$clean_email[] = $notification_email;
						}
					}
				}
				$notifications['notification_email_publisher'] = array_unique(array_slice($clean_email, 0, 5));
				unset($clean_email);
			} else {
				$notifications['notification_email_publisher'] = array(get_option('admin_email'));
			}

			update_option('adrotate_notifications', $notifications);
		}

		if($settings_tab == 'stats') {
			$config = get_option('adrotate_config');

			$stats = $admin_stats = $loggedin_impressions = $loggedin_clicks = $impression_timer = $click_timer = '';
			if(isset($_POST['adrotate_stats'])) $stats = sanitize_text_field($_POST['adrotate_stats']);
			if(isset($_POST['adrotate_enable_admin_stats'])) $admin_stats = sanitize_text_field($_POST['adrotate_enable_admin_stats']);
			if(isset($_POST['adrotate_enable_loggedin_impressions'])) $loggedin_impressions = sanitize_text_field($_POST['adrotate_enable_loggedin_impressions']);
			if(isset($_POST['adrotate_enable_loggedin_clicks'])) $loggedin_clicks = sanitize_text_field($_POST['adrotate_enable_loggedin_clicks']);
			if(isset($_POST['adrotate_impression_timer'])) $impression_timer = sanitize_text_field($_POST['adrotate_impression_timer']);
			if(isset($_POST['adrotate_click_timer'])) $click_timer = sanitize_text_field($_POST['adrotate_click_timer']);

			$config['stats'] = (is_numeric($stats) AND $stats >= 0 AND $stats <= 5) ? $stats : 1;
			$config['enable_admin_stats'] = ($admin_stats == 'Y') ? 'Y' : 'N';
			$config['enable_loggedin_impressions'] = ($loggedin_impressions == 'Y') ? 'Y' : 'N';
			$config['enable_loggedin_clicks'] = ($loggedin_clicks == 'Y') ? 'Y' : 'N';
			$config['impression_timer'] = (is_numeric($impression_timer) AND $impression_timer >= 10 AND $impression_timer <= 3600) ? $impression_timer : 60;
			$config['click_timer'] = (is_numeric($click_timer) AND $click_timer >= 60 AND $click_timer <= 86400) ? $click_timer : 86400;

			update_option('adrotate_config', $config);
		}

		if($settings_tab == 'geo') {
			$config = get_option('adrotate_config');

			$geo = $geo_email = $geo_pass = '';
			if(isset($_POST['adrotate_enable_geo'])) $geo = sanitize_text_field($_POST['adrotate_enable_geo']);
			if(isset($_POST['adrotate_geo_email'])) $geo_email = sanitize_text_field($_POST['adrotate_geo_email']);
			if(isset($_POST['adrotate_geo_pass'])) $geo_pass = sanitize_text_field($_POST['adrotate_geo_pass']);

			$config['enable_geo'] = (is_numeric($geo) AND $geo >= 0 AND $geo <= 7) ? $geo : 0;
			$config['geo_email'] = (strlen($geo_email) > 0) ? $geo_email : '';
			$config['geo_pass'] = (strlen($geo_pass) > 0) ? $geo_pass : '';

			// If no quotas are kept or Geo Targeting is disabled, reset possible old values to 0
			if($config['enable_geo'] == 0 OR $config['enable_geo'] == 6 OR $config['enable_geo'] == 7) {
				update_option('adrotate_geo_requests', 0);
				$dayend = gmmktime(0, 0, 0) + 86400;
				set_transient('adrotate_geo_reset', 1, $dayend);
			}

			update_option('adrotate_config', $config);
		}

		if($settings_tab == 'roles') {
			$config = get_option('adrotate_config');

			$global_report = $ad_manage = $ad_delete = $group_manage = $group_delete = $schedule_manage = $schedule_delete = $advertiser_manage = $moderate = $moderate_approve = '';
			if(isset($_POST['adrotate_global_report'])) $global_report = sanitize_text_field($_POST['adrotate_global_report']);
			if(isset($_POST['adrotate_ad_manage'])) $ad_manage = sanitize_text_field($_POST['adrotate_ad_manage']);
			if(isset($_POST['adrotate_ad_delete'])) $ad_delete = sanitize_text_field($_POST['adrotate_ad_delete']);
			if(isset($_POST['adrotate_group_manage'])) $group_manage = sanitize_text_field($_POST['adrotate_group_manage']);
			if(isset($_POST['adrotate_group_delete'])) $group_delete = sanitize_text_field($_POST['adrotate_group_delete']);
			if(isset($_POST['adrotate_schedule_manage'])) $schedule_manage = sanitize_text_field($_POST['adrotate_schedule_manage']);
			if(isset($_POST['adrotate_schedule_delete'])) $schedule_delete = sanitize_text_field($_POST['adrotate_schedule_delete']);
			if(isset($_POST['adrotate_advertiser_manage'])) $advertiser_manage = sanitize_text_field($_POST['adrotate_advertiser_manage']);
			if(isset($_POST['adrotate_moderate'])) $moderate = sanitize_text_field($_POST['adrotate_moderate']);
			if(isset($_POST['adrotate_moderate_approve'])) $moderate_approve = sanitize_text_field($_POST['adrotate_moderate_approve']);

			$enable_advertisers = $role = $advertiser = '';
			if(isset($_POST['adrotate_enable_advertisers'])) $enable_advertisers = sanitize_text_field($_POST['adrotate_enable_advertisers']);
			if(isset($_POST['adrotate_role'])) $role = sanitize_text_field($_POST['adrotate_role']);
			if(isset($_POST['adrotate_advertiser'])) $advertiser = sanitize_text_field($_POST['adrotate_advertiser']);

			adrotate_set_capability($global_report, 'adrotate_global_report');
			adrotate_set_capability($ad_manage, 'adrotate_ad_manage');
			adrotate_set_capability($ad_delete, 'adrotate_ad_delete');
			adrotate_set_capability($group_manage, 'adrotate_group_manage');
			adrotate_set_capability($group_delete, 'adrotate_group_delete');
			adrotate_set_capability($schedule_manage, 'adrotate_schedule_manage');
			adrotate_set_capability($schedule_delete, 'adrotate_schedule_delete');
			adrotate_set_capability($advertiser_manage, 'adrotate_advertiser_manage');
			adrotate_set_capability($moderate, 'adrotate_moderate');
			adrotate_set_capability($moderate_approve, 'adrotate_moderate_approve');

			$config['global_report'] = $global_report;
			$config['ad_manage'] = $ad_manage;
			$config['ad_delete'] = $ad_delete;
			$config['group_manage'] = $group_manage;
			$config['group_delete'] = $group_delete;
			$config['schedule_manage'] = $schedule_manage;
			$config['schedule_delete'] = $schedule_delete;
			$config['advertiser_manage'] = $advertiser_manage;
			$config['moderate'] = $moderate;
			$config['moderate_approve']	= $moderate_approve;

			$config['enable_advertisers'] = ($enable_advertisers == "Y") ? 'Y' : 'N';

			if($role == "Y") {
				adrotate_prepare_roles('add');
			} else {
				adrotate_prepare_roles('remove');
			}

			adrotate_set_capability($advertiser, 'adrotate_advertiser');
			$config['advertiser'] = $advertiser;

			update_option('adrotate_config', $config);
		}

		if($settings_tab == 'misc') {
			$config = get_option('adrotate_config');

			$zindex = $widgetalign = $widgetpadding = $hide_schedules = $w3caching = $borlabscache = '';
			if(isset($_POST['adrotate_zindex'])) $zindex = sanitize_text_field($_POST['adrotate_zindex']);
			if(isset($_POST['adrotate_widgetalign'])) $widgetalign = sanitize_text_field($_POST['adrotate_widgetalign']);
			if(isset($_POST['adrotate_widgetpadding'])) $widgetpadding = sanitize_text_field($_POST['adrotate_widgetpadding']);
			if(isset($_POST['adrotate_hide_schedules'])) $hide_schedules = sanitize_text_field($_POST['adrotate_hide_schedules']);
			if(isset($_POST['adrotate_w3caching'])) $w3caching = sanitize_text_field($_POST['adrotate_w3caching']);
			if(isset($_POST['adrotate_borlabscache'])) $borlabscache = sanitize_text_field($_POST['adrotate_borlabscache']);

			$config['zindex'] = $zindex;
			$config['widgetalign'] = ($widgetalign == "Y") ? 'Y' : 'N';
			$config['widgetpadding'] = ($widgetpadding == "Y") ? 'Y' : 'N';
			$config['hide_schedules'] = ($hide_schedules == "Y") ? 'Y' : 'N';
			$config['w3caching'] = ($w3caching == "Y") ? 'Y' : 'N';
			$config['borlabscache'] = ($borlabscache == "Y") ? 'Y' : 'N';

			update_option('adrotate_config', $config);
		}

		// Return to dashboard
		wp_redirect('admin.php?page=adrotate-settings&status=400&tab='.$settings_tab);
		exit;
	} else {
		adrotate_nonce_error();
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_save_profile_fields
 Purpose:   Save custom profile fields
 Since:		3.22.2b1
-------------------------------------------------------------*/
function adrotate_save_profile_fields($user_id){
    if(current_user_can('adrotate_advertiser_manage')) {
		// Is the user an advertiser?
		$advertiser = (isset($_POST['adrotate_is_advertiser']) AND strlen($_POST['adrotate_is_advertiser']) != 0) ? 'Y' : 'N';
	    update_user_meta($user_id, 'adrotate_is_advertiser', $advertiser);

		// Set user permissions
		$permissions['create'] = (isset($_POST['adrotate_can_create']) AND strlen($_POST['adrotate_can_create']) != 0 AND $advertiser == 'Y') ? 'Y' : 'N';
		$permissions['edit'] = (isset($_POST['adrotate_can_edit']) AND strlen($_POST['adrotate_can_edit']) != 0 AND $advertiser == 'Y') ? 'Y' : 'N';

		$permissions['advanced'] = (isset($_POST['adrotate_can_advanced']) AND strlen($_POST['adrotate_can_advanced']) != 0 AND $advertiser == 'Y') ? 'Y' : 'N';
		$permissions['geo'] = (isset($_POST['adrotate_can_geo']) AND strlen($_POST['adrotate_can_geo']) != 0 AND $advertiser == 'Y') ? 'Y' : 'N';
		$permissions['group'] = (isset($_POST['adrotate_can_group']) AND strlen($_POST['adrotate_can_group']) != 0 AND $advertiser == 'Y') ? 'Y' : 'N';
		$permissions['schedule'] = (isset($_POST['adrotate_can_schedule']) AND strlen($_POST['adrotate_can_schedule']) != 0 AND $advertiser == 'Y') ? 'Y' : 'N';
	    update_user_meta($user_id, 'adrotate_permissions', $permissions);

		// User notes
		$notes = htmlspecialchars(trim($_POST['adrotate_notes']), ENT_QUOTES);
	    update_user_meta($user_id, 'adrotate_notes', $notes);
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_prepare_roles
 Purpose:   Prepare user roles for WordPress
 Since:		3.0
-------------------------------------------------------------*/
function adrotate_prepare_roles($action) {

	if($action == 'add') {
		add_role('adrotate_advertiser', __('AdRotate Advertiser', 'adrotate-pro'), array('read' => 1));
	}
	if($action == 'remove') {
		remove_role('adrotate_advertiser');
	}
}
?>
