<?php
/* ------------------------------------------------------------------------------------
*  COPYRIGHT AND TRADEMARK NOTICE
*  Copyright 2008-2026 Arnan de Gans. All Rights Reserved.
*  The software AdRotate, related services and ADROTATE the name are owned by Arnan de Gans.

*  COPYRIGHT NOTICES AND ALL THE COMMENTS SHOULD REMAIN INTACT.
*  By using this code you agree to indemnify Arnan de Gans from any
*  liability that might arise from its use.
------------------------------------------------------------------------------------ */

/*-------------------------------------------------------------
 Name:      adrotate_is_human
 Purpose:   Check if visitor is a bot
-------------------------------------------------------------*/
function adrotate_is_human() {
	global $adrotate_crawlers;

	if(is_array($adrotate_crawlers)) {
		$crawlers = $adrotate_crawlers;
	} else {
		$crawlers = array();
	}

	$useragent = adrotate_get_useragent();

	$nocrawler = array(true);
	if(strlen($useragent) > 0) {
		foreach($crawlers as $key => $crawler) {
			if(preg_match('/'.$crawler.'/i', $useragent)) $nocrawler[] = false;
		}
	}
	$nocrawler = (!in_array(false, $nocrawler)) ? true : false; // If no bool false in array it's not a bot

	// Returns true if no bot.
	return $nocrawler;
}

/*-------------------------------------------------------------
 Name:      adrotate_is_ios
 Purpose:   Check if OS is iOS
-------------------------------------------------------------*/
function adrotate_is_ios() {
	if(!class_exists('\Detection\MobileDetect')) {
		require_once(dirname(__FILE__).'/library/mobile-detect.php');
	}
	$detect = new \Detection\MobileDetect;

	if($detect->isiOS() AND !$detect->isAndroidOS()) {
		return true;
	}
	return false;
}

/*-------------------------------------------------------------
 Name:      adrotate_is_android
 Purpose:   Check if OS is Android
-------------------------------------------------------------*/
function adrotate_is_android() {
	if(!class_exists('\Detection\MobileDetect')) {
		require_once(dirname(__FILE__).'/library/mobile-detect.php');
	}
	$detect = new \Detection\MobileDetect;

	if(!$detect->isiOS() AND $detect->isAndroidOS()) {
		return true;
	}
	return false;
}

/*-------------------------------------------------------------
 Name:      adrotate_is_mobile
 Purpose:   Check if visitor is on a smartphone
-------------------------------------------------------------*/
function adrotate_is_mobile() {
	if(!class_exists('\Detection\MobileDetect')) {
		require_once(dirname(__FILE__).'/library/mobile-detect.php');
	}
	$detect = new \Detection\MobileDetect;

	if($detect->isMobile() AND !$detect->isTablet()) {
		return true;
	}
	return false;
}

/*-------------------------------------------------------------
 Name:      adrotate_is_tablet
 Purpose:   Check if visitor is on a tablet
-------------------------------------------------------------*/
function adrotate_is_tablet() {
	if(!class_exists('\Detection\MobileDetect')) {
		require_once(dirname(__FILE__).'/library/mobile-detect.php');
	}
	$detect = new \Detection\MobileDetect;

	if($detect->isTablet()) {
		return true;
	}
	return false;
}

/*-------------------------------------------------------------
 Name:      adrotate_filter_duplicates
 Purpose:   Remove adverts that already show elsewhere on the page
-------------------------------------------------------------*/
function adrotate_filter_duplicates($banner_id, $session_page) {
	// Get session if it exists
	$adrotate_session = get_transient('adrotate_'.md5(adrotate_get_remote_ip()));
	if($adrotate_session === false) $adrotate_session = array();

	// Check session data
	if(!isset($adrotate_session['duplicate-ads'][$session_page]) OR (isset($adrotate_session['duplicate-ads'][$session_page]['timeout']) AND $adrotate_session['duplicate-ads'][$session_page]['timeout'] < current_time('timestamp'))) {
		$adrotate_session['duplicate-ads'][$session_page] = array('timeout' => current_time('timestamp'), 'adverts' => array());

		$ttl = ini_get('session.gc_maxlifetime');
		if(empty($ttl)) $ttl = 1440; // Set some default if PHP doesn't know
		set_transient('adrotate_'.md5(adrotate_get_remote_ip()), $adrotate_session, $ttl);
	} else {
		if(in_array($banner_id, $adrotate_session['duplicate-ads'][$session_page]['adverts'])) return true;
	}

	return false;
}

/*-------------------------------------------------------------
 Name:      adrotate_filter_show_everyone
 Purpose:   Remove adverts that don't show to logged in users
-------------------------------------------------------------*/
function adrotate_filter_show_everyone($banner) {
	if(($banner['show_everyone'] == 'N') AND is_user_logged_in()) {
		return true;
	}

	return false;
}

/*-------------------------------------------------------------
 Name:      adrotate_filter_schedule
 Purpose:   Weed out ads that are over the limit of their schedule
-------------------------------------------------------------*/
function adrotate_filter_schedule($banner) {
	global $wpdb, $adrotate_config;

	$now = current_time('timestamp');
	$day = date('D', $now);
	$hour = date('Hi', $now);

	$schedules = $wpdb->get_results("SELECT `{$wpdb->prefix}adrotate_schedule`.* FROM `{$wpdb->prefix}adrotate_schedule`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `schedule` = `{$wpdb->prefix}adrotate_schedule`.`id` AND `ad` = {$banner['id']} ORDER BY `starttime` ASC;");

	$current = array();
	foreach($schedules as $schedule) {
		if($schedule->starttime > $now OR $schedule->stoptime < $now) {
			$current[] = 0;
		} else if(($schedule->day_mon != 'Y' AND $day == 'Mon') OR ($schedule->day_tue != 'Y' AND $day == 'Tue') OR ($schedule->day_wed != 'Y' AND $day == 'Wed') OR ($schedule->day_thu != 'Y' AND $day == 'Thu') OR ($schedule->day_fri != 'Y' AND $day == 'Fri') OR ($schedule->day_sat != 'Y' AND $day == 'Sat') OR ($schedule->day_sun != 'Y' AND $day == 'Sun')) {
			$current[] = 0;
		} else if(($schedule->daystarttime > 0 OR $schedule->daystoptime > 0) AND ($schedule->daystarttime > $hour OR $schedule->daystoptime < $hour)) {
			$current[] = 0;
		} else {
			if($adrotate_config['stats'] == 1 AND $banner['tracker'] != 'N') {
				$stat = adrotate_get_stats($banner['id'], $schedule->starttime, $schedule->stoptime);
				if($schedule->spread_all == 'Y') {
					$group_impressions = $wpdb->get_var("SELECT SUM(`impressions`) FROM `{$wpdb->prefix}adrotate_stats`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `schedule` = {$schedule->id} AND `{$wpdb->prefix}adrotate_linkmeta`.`ad` = `{$wpdb->prefix}adrotate_stats`.`ad` AND `thetime` >= {$schedule->starttime} AND `thetime` <= {$schedule->stoptime};");
				} else {
					$group_impressions = 0;
				}
				$temp_max_impressions = floor($schedule->maximpressions / ($schedule->stoptime - $schedule->starttime) * ($now - $schedule->starttime));

				if(!is_array($stat)) $stat = array('clicks' => 0, 'impressions' => 0);

				if($stat['clicks'] >= $schedule->maxclicks AND $schedule->maxclicks > 0) { // Click limit reached?
					$current[] = 0;
					$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = 'limit' WHERE `id` = {$banner['id']};"); // Set advert expired
				} else if($stat['impressions'] >= $schedule->maximpressions AND $schedule->maximpressions > 0) { // Impression limit reached?
					$current[] = 0;
					$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = 'limit' WHERE `id` = {$banner['id']};"); // Set advert expired
				} else if($schedule->spread == 'Y' AND $stat['impressions'] > $temp_max_impressions) { // Impression spread
					$current[] = 0;
				} else if($schedule->spread_all == 'Y' AND $group_impressions > $temp_max_impressions) { // Impression spread all (campaigns)
					$current[] = 0;
				} else { // Everything else
					$current[] = 1;
				}
			} else {
				$current[] = 1;
			}
		}
	}

	// Remove advert from array if all schedules are false (0)
	if(!in_array(1, $current)) {
		return true;
	}
	unset($current, $schedules);

	return false;
}

/*-------------------------------------------------------------
 Name:      adrotate_filter_budget
 Purpose:   Weed out ads that are over the limit of their schedule
-------------------------------------------------------------*/
function adrotate_filter_budget($banner) {
	global $wpdb;

	$result = false;

	if($banner['budget'] == null) $banner['budget'] = '0';
	if($banner['crate'] == null) $banner['crate'] = '0';
	if($banner['irate'] == null) $banner['irate'] = '0';

	if(($banner['budget'] <= 0 AND $banner['crate'] > 0) OR ($banner['budget'] <= 0 AND $banner['irate'] > 0)) {
		// Set advert expired
		$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `type` = 'limit' WHERE `id` = {$banner['id']};");

		$result = true;
	}
	if($banner['budget'] > 0 AND $banner['irate'] > 0) {
		$cpm = number_format($banner['irate'] / 1000, 4, '.', '');
		$wpdb->query("UPDATE `{$wpdb->prefix}adrotate` SET `budget` = `budget` - {$cpm} WHERE `id` = {$banner['id']};");
	}

	return $result;
}

/*-------------------------------------------------------------
 Name:      adrotate_filter_location
 Purpose:   Determine the users location, the ads geo settings and filter out ads
-------------------------------------------------------------*/
function adrotate_filter_location($banner) {
	// Get session if it exists
	$adrotate_session = get_transient('adrotate_'.md5(adrotate_get_remote_ip()));
	if($adrotate_session === false) $adrotate_session = array();

	if(isset($adrotate_session['geo'])) {
		$geo = $adrotate_session['geo'];

		// Good Geo Response?
		if($geo['code'] == 200) {
			$keep = array();

			$cities = unserialize(stripslashes($banner['cities']));
			$states = unserialize(stripslashes($banner['states']));
			$countries = unserialize(stripslashes($banner['countries']));
			if(!is_array($cities)) $cities = array();
			if(!is_array($states)) $states = array();
			if(!is_array($countries)) $countries = array();

			// Match a city
			if(count($cities) > 0) {
				if(count(array_intersect($cities, array($geo['city'], $geo['dma']))) > 0) {
					$keep[] = 'C';
				}
			}

			// Match a state
			if(count($states) > 0) {
				if(count(array_intersect($states, array($geo['state'], $geo['statecode']))) > 0) {
					$keep[] = 'S';
				}
			}

			// Match a city in a state
			if(count($cities) > 0 AND count($states) > 0 AND $banner['state_req'] == 'Y') {
				if(in_array('C', $keep) AND in_array('S', $keep)) {
					$keep[] = 'Y';
				}
			}

			// Match a country
			if(count($countries) > 0) {
				if(in_array($geo['countrycode'], $countries)) {
					$keep[] = 'Y';
				}
			}

			unset($adrotate_session, $geo);

			// Remove advert from pool?
			if(count($keep) == 0) {
				return true;
			}
		}
	}

	return false;
}

/*-------------------------------------------------------------
 Name:      adrotate_filter_content
 Purpose:   Deal with quotes, pre and embed codes
-------------------------------------------------------------*/
function adrotate_filter_content($content) {
	// Deal with <blockquote>
    $array = preg_split('/<blockquote>/', $content);
    $content = array_shift($array);
    foreach ($array as $string) {
        $content .= "<blockquote>";
        $array2 = preg_split(',</blockquote>,', $string);
        $content .= preg_replace('/./', ' ', array_shift($array2));
        $content .= "</blockquote>";
        if (!empty($array2)) {
            $content .= $array2[0];
        }
    }
    unset($array, $array2, $string);

	// Deal with <pre>
    $array = preg_split('/<pre>/', $content);
    $content = array_shift($array);
    foreach ($array as $string) {
        $content .= "<pre>";
        $array2 = preg_split(',</pre>,', $string);
        $content .= preg_replace('/./', ' ', array_shift($array2));
        $content .= "</pre>";
        if (!empty($array2)) {
            $content .= $array2[0];
        }
    }
    unset($array, $array2, $string);

	// Deal with <code>
    $array = preg_split('/<code>/', $content);
    $content = array_shift($array);
    foreach ($array as $string) {
        $content .= "<code>";
        $array2 = preg_split(',</code>,', $string);
        $content .= preg_replace('/./', ' ', array_shift($array2));
        $content .= "</code>";
        if (!empty($array2)) {
            $content .= $array2[0];
        }
    }
    unset($array, $array2, $string);

    return $content;
}

/*-------------------------------------------------------------
 Name:      adrotate_session
 Purpose:   Start a 'session' for Geo Targeting and Duplicate ads
-------------------------------------------------------------*/
function adrotate_session() {
	if(!wp_doing_cron() AND !defined('WP_CLI')) {
		global $adrotate_config;

		$geo_required = get_option('adrotate_geo_required');
		$visitor_ip = md5(adrotate_get_remote_ip());
		$adrotate_session = get_transient('adrotate_'.$visitor_ip);
		if($adrotate_session === false) $adrotate_session = array();

		if(($geo_required > 0 OR $adrotate_config['duplicate_adverts_filter'] == "Y") AND !$adrotate_session) {
			$ttl = ini_get('session.gc_maxlifetime');
			if(empty($ttl)) $ttl = 1440; // Set some default if PHP doesn't know
			set_transient('adrotate_'.$visitor_ip, array('geo' => array(), 'duplicate-ads' => array()), $ttl);
		}

		if($geo_required > 0) {
			adrotate_geolocation();
		}
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_geolocation
 Purpose:   Find the location of the visitor
-------------------------------------------------------------*/
function adrotate_geolocation() {
	// Get session if it exists
	$visitor_ip = adrotate_get_remote_ip();
	$adrotate_session = get_transient('adrotate_'.md5($visitor_ip));
	if($adrotate_session === false) $adrotate_session = array();

	if(!wp_doing_cron() AND !defined('WP_CLI') AND adrotate_is_human() AND empty($adrotate_session['geo'])) {
		$adrotate_config = get_option('adrotate_config');
		$geo_result = array();

		if(!function_exists('get_plugins')) require_once(ABSPATH . 'wp-admin/includes/plugin.php');
		$plugin_version = get_plugins();
		$useragent = 'adrotate-pro/'.$plugin_version['adrotate-pro/adrotate-pro.php']['Version'].';';

		/*-- MaxMind ------------------------------------------------*/
		if($adrotate_config['enable_geo'] == 3 OR $adrotate_config['enable_geo'] == 4) {
			if($adrotate_config['enable_geo'] == 3) {
				$service_type = 'country';
			}
			if($adrotate_config['enable_geo'] == 4) {
				$service_type = 'city';
			}

			$args = array(
				'headers' => array(
					'user-agent' => $useragent,
					'accept' => 'application/json, */*;q=0.8',
					'authorization' => 'Basic '.base64_encode($adrotate_config['geo_email'].':'.$adrotate_config['geo_pass'])
				),
				'timeout' => 5,
				'sslverify' => false
			);
			$response = wp_remote_get('https://geoip.maxmind.com/geoip/v2.1/'.$service_type.'/'.$visitor_ip, $args);

			$geo_result['provider'] = 'MaxMind '.$service_type;
		    if(!is_wp_error($response)) {
			    $data = json_decode($response['body'], true);

				$geo_result['code'] = '200';
				$geo_result['message'] = 'OK';
				$geo_result['city'] = (isset($data['city']['names']['en'])) ? strtolower($data['city']['names']['en']) : '';
				$geo_result['dma'] = (isset($data['location']['metro_code'])) ? strtolower($data['location']['metro_code']) : '';
				$geo_result['countrycode'] = (isset($data['country']['iso_code'])) ? $data['country']['iso_code'] : '';
				$geo_result['state'] = (isset($data['subdivisions'][0]['names']['en'])) ? strtolower($data['subdivisions'][0]['names']['en']) : '';
				$geo_result['statecode'] = (isset($data['subdivisions'][0]['iso_code'])) ? strtolower($data['subdivisions'][0]['iso_code']) : '';

				$data['lookups_used'] = $data['maxmind']['queries_remaining'];
			} else {
				$geo_result['code'] = $response->get_error_code();
				$geo_result['message'] = $response->get_error_message();

				$data['lookups_used'] = 0;
			}

			update_option('adrotate_geo_requests', $data['lookups_used']);
		}
	    unset($response, $data);

		/*-- AdRotate Geo -------------------------------------------*/
		if($adrotate_config['enable_geo'] == 5) {
			$lookups = get_option('adrotate_geo_requests');

			// Maybe poke AdRotate Geo to figure out if a lookup can be made
			if($lookups < 1 AND !get_transient('adrotate_geo_reset')) {
				$lookups = 5; // 5-ish attempts to reset quota
			}

			if($lookups > 0) {
				$a = get_option('ajdg_activate_adrotate-pro');

				$args = array(
					'headers' => array(
						'accept' => 'multipart/form-data',
						'user-agent' => $useragent
					),
					'body' => array(
						'r' => serialize(array(
							'slug' => "adrotate-pro",
							'instance' => $a['instance'],
							'platform' => strtolower(get_option('siteurl')),
							'action' => 'geo_lookup',
							'et' => microtime(true),
							'ip' => $visitor_ip
						))
					),
					'sslverify' => true,
					'timeout' => 5
				);

				$response = wp_remote_post('https://ajdg.solutions/api/1/api.php', $args);

				$geo_result['provider'] = 'AdRotate Geo';
			    if(!is_wp_error($response)) {
				    $data = unserialize($response['body']);

					$geo_result['code'] = '200';
					$geo_result['message'] = 'OK';
					$geo_result['city'] = (isset($data['city'])) ? strtolower($data['city']) : '';
					$geo_result['dma'] = (isset($data['dma'])) ? strtolower($data['dma']) : '';
					$geo_result['countrycode'] = (isset($data['countrycode'])) ? $data['countrycode'] : '';
					$geo_result['state'] = (isset($data['state'])) ? strtolower($data['state']) : '';
					$geo_result['statecode'] = (isset($data['statecode'])) ? strtolower($data['statecode']) : '';

					$data['lookups_used'] = (isset($data['lookups_used'])) ? 30000 - $data['lookups_used'] : 0;
				} else {
					$geo_result['code'] = $response->get_error_code();
					$geo_result['message'] = $response->get_error_message();
					$data['lookups_used'] = 0;

					if($geo_result['code'] === 401) {
						$adrotate_config['enable_geo'] = 0; // Disable AdRotate Geo
						update_option('adrotate_config', $adrotate_config);
					}
				}

				update_option('adrotate_geo_requests', $data['lookups_used']);

				$until_day_end = (gmmktime(0, 0, 0) + 86400) - current_time('timestamp', 1);
				set_transient('adrotate_geo_reset', 1, $until_day_end);
			}
		}
	    unset($response, $data);

		/*-- CloudFlare ---------------------------------------------*/
		if($adrotate_config['enable_geo'] == 6) {
			$geo_result['provider'] = 'CloudFlare';
		    if(isset($_SERVER['HTTP_CF_IPCOUNTRY'])) {
				$geo_result['code'] = '200';
				$geo_result['message'] = 'OK';
				$geo_result['city'] = '';
				$geo_result['dma'] = '';
				$geo_result['countrycode'] = ($_SERVER['HTTP_CF_IPCOUNTRY'] == 'xx') ? '' : $_SERVER['HTTP_CF_IPCOUNTRY'];
				$geo_result['state'] = '';
				$geo_result['statecode'] = '';
			} else {
				$geo_result['code'] = 503;
				$geo_result['message'] = 'Header not found, check if Geo feature in CloudFlare is enabled.';
			}
		}

		/*-- ipstack ------------------------------------------------*/
		if($adrotate_config['enable_geo'] == 7) {
			// Does not report lookups
			$args = array(
				'headers' => array(
					'user-agent' => $useragent,
					'accept' => 'application/json, */*;q=0.8'
				),
				'timeout' => 5
			);
			$response = wp_remote_get('http://api.ipstack.com/'.$visitor_ip.'?access_key='.$adrotate_config['geo_pass'], $args);

			$geo_result['provider'] = 'ipstack';
		    if(!is_wp_error($response)) {
			    $data = json_decode($response['body'], true);

				if($response['response']['code'] === 200 AND !array_key_exists('error', $data)) {
					$geo_result['code'] = '200';
					$geo_result['message'] = 'OK';
					$geo_result['city'] = (isset($data['city'])) ? strtolower($data['city']) : '';
					$geo_result['dma'] = (isset($data['geoname_id'])) ? strtolower($data['geoname_id']) : '';
					$geo_result['countrycode'] = (isset($data['country_code'])) ? $data['country_code'] : '';
					$geo_result['state'] = (isset($data['region_name'])) ? strtolower($data['region_name']) : '';
					$geo_result['statecode'] = (isset($data['region_code'])) ? strtolower($data['region_code']) : '';
				} else {
					$geo_result['code'] = $data['error']['code'].' '.$data['error']['type'];
					$geo_result['message'] = $data['error']['info'];
				}
			} else {
				$geo_result['code'] = $response->get_error_code();
				$geo_result['message'] = $response->get_error_message();
			}
		}
	    unset($response, $data);

		// Add geo data to session
		$geo_result['retrieved'] = date_i18n("Y-m-d G:i", current_time('timestamp'));
		$adrotate_session['geo'] = $geo_result;

		// Store new session data
		$ttl = ini_get('session.gc_maxlifetime');
		if(empty($ttl)) $ttl = 1440; // Set some default if PHP doesn't know
		set_transient('adrotate_'.md5($visitor_ip), $adrotate_session, $ttl);
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_pick_weight
 Purpose:   Sort out and pick a random advert based on weight
-------------------------------------------------------------*/
function adrotate_pick_weight($ads) {
    $ad_ids = array_column($ads, 'id');
	$weight = array_column($ads, 'weight');

    $sum_of_weight = array_sum($weight)-1;
    if($sum_of_weight < 1) $sum_of_weight = 2;
    $random = mt_rand(0, $sum_of_weight);

    foreach($ad_ids as $key => $ad_id){
        if($random < $weight[$key]){
            return $key;
        }
        $random = $random - $weight[$key];
    }
    unset($ad_ids, $weight, $sum_of_weight, $random);
}

/*-------------------------------------------------------------
 Name:      adrotate_shuffle
 Purpose:   Randomize and slice an array but keep keys intact
-------------------------------------------------------------*/
function adrotate_shuffle($array, $amount = 20) {
	if(!is_array($array)) return $array;

	$keys = array_keys($array);
	shuffle($keys);

	$shuffle = array();
	foreach($keys as $key) {
		$shuffle[$key] = $array[$key];
	}
	return $shuffle;
}

/*-------------------------------------------------------------
 Name:      adrotate_get_remote_ip
 Purpose:   Get the remote IP from the visitor
-------------------------------------------------------------*/
function adrotate_get_remote_ip(){
	$accepted_headers = array(
		// Providers
		'HTTP_CF_CONNECTING_IP', // Cloudflare
		'HTTP_INCAP_CLIENT_IP', // Incapsula
		'HTTP_X_CLUSTER_CLIENT_IP', // RackSpace
		'HTTP_TRUE_CLIENT_IP', // Akamai
		// Proxies
		'HTTP_X_FORWARDED_FOR',
		'HTTP_X_FORWARDED',
		'HTTP_CLIENT_IP',
		'HTTP_X_REAL_IP',
		'HTTP_FORWARDED',
		'HTTP_FORWARDED_FOR',
		// Standard
		'REMOTE_ADDR'
	);

	foreach($accepted_headers as $header) {
		if(isset($_SERVER[$header]) AND is_string($_SERVER[$header])) {
			$remote_ip = explode(',', $_SERVER[$header], 2);
			$remote_ip = $remote_ip[0];

			if(filter_var($remote_ip, FILTER_VALIDATE_IP)) {
				return $remote_ip;
			}
		}
	}

	return $_SERVER['REMOTE_ADDR'];
}

/*-------------------------------------------------------------
 Name:      adrotate_get_useragent
 Purpose:   Get the useragent from the visitor
-------------------------------------------------------------*/
function adrotate_get_useragent(){
	if(isset($_SERVER['HTTP_USER_AGENT'])) {
		$useragent = $_SERVER['HTTP_USER_AGENT'];
		$useragent = trim($useragent, ' \t\r\n\0\x0B');
	} else {
		$useragent = '';
	}

	return $useragent;
}

/*-------------------------------------------------------------
 Name:      adrotate_notifications
 Purpose:   Contact admins/moderators about various things
-------------------------------------------------------------*/
function adrotate_notifications($action = false, $adid = false) {
	global $wpdb, $adrotate_config;

	$notifications = get_option('adrotate_notifications');
	$advert_status = get_option('adrotate_advert_status');

	$title = '';
	$message = array();

	if($action == 'test') {
		$title = esc_html__("Test notification", 'adrotate-pro');
		$message[] = esc_html__("This is a test notification from AdRotate Professional.", 'adrotate-pro');
		$message[] = esc_html__("Have a nice day!", 'adrotate-pro');
	} else {
		// Advert status
		if($notifications['notification_mail_status'] == 'Y') {
			$title = esc_html__("Status update", 'adrotate-pro');
			if($advert_status['error'] > 0) $message[] = $advert_status['error']." ".esc_html__("advert(s) with errors!", 'adrotate-pro');
			if($advert_status['expired'] > 0) $message[] = $advert_status['expired']." ".esc_html__("advert(s) expired!", 'adrotate-pro');
			if($advert_status['expiressoon'] > 0) $message[] = $advert_status['expiressoon']." ".esc_html__("advert(s) will expire in less than 2 days.", 'adrotate-pro');
			if($advert_status['expiresweek'] > 0) $message[] = $advert_status['expiresweek']." ".esc_html__("advert(s) will expire in less than a week.", 'adrotate-pro');
			if($advert_status['unknown'] > 0) $message[] = $advert_status['unknown']." ".esc_html__("advert(s) have an unknown status.", 'adrotate-pro');
		}

		// Geo Targeting
		if($notifications['notification_mail_geo'] == 'Y') {
			$geo_lookups = get_option('adrotate_geo_requests');

			if(($adrotate_config['enable_geo'] == 3 OR $adrotate_config['enable_geo'] == 4 OR $adrotate_config['enable_geo'] == 5) AND $geo_lookups < 1000) {
				$title = esc_html__("Geo targeting", 'adrotate-pro');
				if($geo_lookups > 0) $message[] = esc_html__("Your website has less than 1000 lookups left for Geo Targeting. If you run out of lookups, Geo Targeting will stop working.", 'adrotate-pro');
				if($geo_lookups < 1) $message[] = esc_html__("Your website has no lookups for Geo Targeting. Geo Targeting is currently not working.", 'adrotate-pro');
			}
		}

		// User (Advertiser) invoked actions (not on a timer)
		if($notifications['notification_mail_queue'] == 'Y') {
			if($action == 'queued') {
				$name = $wpdb->get_var("SELECT `title` FROM `{$wpdb->prefix}adrotate` WHERE `id` = {$adid};");
				$queued = $wpdb->get_var("SELECT COUNT(*) FROM `{$wpdb->prefix}adrotate` WHERE `type` = 'queue' OR `type` = 'reject';");

				$title = esc_html__("Moderation queue", 'adrotate-pro');
				$message[] = esc_html__("An advertiser has just queued one of their adverts for review.", 'adrotate-pro');
				$message[] = $name." (ID: ".$adid.")";
				$message[] = "Awaiting moderation: ".$queued." adverts.";
			}
		}

		if($notifications['notification_mail_approved'] == 'Y') {
			if($action == 'approved') {
				$name = $wpdb->get_var("SELECT `title` FROM `{$wpdb->prefix}adrotate` WHERE `id` = {$adid};");

				$title = esc_html__("Advert approved", 'adrotate-pro');
				$message[] = esc_html__("A moderator has just approved an advert;", 'adrotate-pro');
				$message[] = $name." (ID: ".$adid.")";
			}
		}

		if($notifications['notification_mail_rejected'] == 'Y') {
			if($action == 'rejected') {
				$name = $wpdb->get_var("SELECT `title` FROM `{$wpdb->prefix}adrotate` WHERE `id` = {$adid};");

				$title = esc_html__("Advert rejected", 'adrotate-pro');
				$message[] = esc_html__("A moderator has just rejected advert;", 'adrotate-pro');
				$message[] = $name." (ID: ".$adid.")";
			}
		}
	}

	// Maybe send some alerts (Test or real)
	if(count($message) > 0) {
		if($notifications['notification_email'] == 'Y') {
			$blogname = get_option('blogname');
			$dashboardurl = get_option('siteurl').'/wp-admin/admin.php?page=adrotate';
			$pluginurl = 'https://ajdg.solutions/';

			$subject = '[AdRotate Alert] '.$title;
			$message_content = "<p>".esc_html__("Hello", 'adrotate-pro').",</p>";
			$message_content .= "<p>".esc_html__("This notification is sent to you from your website", 'adrotate-pro')." '".$blogname."'.</p>";
			$message_content .= "<p>".implode('\n', $message)."</p>";
			$message_content .= "<p>".esc_html__("Access your dashboard here:", 'adrotate-pro')." ".$dashboardurl."</p>";
			$message_content .= "<p>".esc_html__("Have a nice day!", 'adrotate-pro')."<br />".esc_html__("Your AdRotate Notifier", 'adrotate-pro')."</p>";
			$message_content .= "<p><a href=\"".$pluginurl."\"> target=\"_blank\">ajdg.solutions</a></p>";

			$x = count($notifications['notification_email_publisher']);
			for($i=0;$i<$x;$i++) {
			    $headers = "Content-Type: text/html; charset=UTF-8\r\nFrom: AdRotate Plugin <".$notifications['notification_email_publisher'][$i].">" . "\r\n";
				wp_mail($notifications['notification_email_publisher'][$i], $subject, $message_content, $headers);
			}
		}
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_empty_trackerdata
 Purpose:   Removes old statistics
-------------------------------------------------------------*/
function adrotate_empty_trackerdata() {
	global $wpdb;

	$clicks = current_time('timestamp') - DAY_IN_SECONDS;
	$impressions = current_time('timestamp') - HOUR_IN_SECONDS;

	$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate_tracker` WHERE `timer` < {$impressions} AND `stat` = 'i';");
	$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate_tracker` WHERE `timer` < {$clicks} AND `stat` = 'c';");
	$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate_tracker` WHERE `ipaddress`  = 'unknown' OR `ipaddress`  = '';");
}

/*-------------------------------------------------------------
 Name:      adrotate_empty_trash
 Purpose:   Delete expired and trashed adverts
-------------------------------------------------------------*/
function adrotate_empty_trash() {
	global $wpdb;

	$threedaysago = current_time('timestamp') - 259200;

	$adverts = $wpdb->get_results("SELECT `id` FROM `{$wpdb->prefix}adrotate` WHERE `type` = 'trash' AND `updated` < {$threedaysago};");
	foreach($adverts as $advert) {
		$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate` WHERE `id` = {$advert->id};");
		$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = {$advert->id};");
		$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate_stats` WHERE `ad` = {$advert->id};");
		$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate_stats_archive` WHERE `ad` = {$advert->id};");
	}
	unset($adverts);
}

/*-------------------------------------------------------------
 Name:      adrotate_auto_deletedisable
 Purpose:   Auto trash selected adverts and schedules
-------------------------------------------------------------*/
function adrotate_auto_deletedisable() {
	global $wpdb;

	// Auto trash expired adverts
	$now = current_time('timestamp');
	$twentythreehoursago = $now - 82800;

	$adverts = $wpdb->get_results("SELECT `id`, `autodelete`, `autodisable` FROM `{$wpdb->prefix}adrotate` WHERE `type` = 'expired' AND (`autodelete` = 'Y' OR `autodisable` = 'Y');");
	foreach($adverts as $advert) {
		$stoptime = $wpdb->get_var("SELECT `stoptime` FROM `{$wpdb->prefix}adrotate_schedule`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = '{$advert->id}' AND  `schedule` = `{$wpdb->prefix}adrotate_schedule`.`id` ORDER BY `stoptime` DESC LIMIT 1;");

		if($advert->autodelete == "Y" AND $stoptime <= $twentythreehoursago) {
			$wpdb->update("{$wpdb->prefix}adrotate", array('type' => 'trash', 'updated' => $now), array('id' => $advert->id));
		}
		if($advert->autodisable == "Y" AND $stoptime <= $now) {
			$wpdb->update("{$wpdb->prefix}adrotate", array('type' => 'disabled', 'updated' => $now), array('id' => $advert->id));
		}
		unset($advert, $stoptime);
	}
	unset($adverts);

	// Auto delete expired schedules
	$schedules = $wpdb->get_results("SELECT `id` FROM `{$wpdb->prefix}adrotate_schedule` WHERE `stoptime` <= {$twentythreehoursago} AND `autodelete` = 'Y';");
	foreach($schedules as $schedule) {
		$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate_schedule` WHERE `id` = {$schedule->id};");
		$wpdb->query("DELETE FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `schedule` = {$schedule->id};");
	}

}

/*-------------------------------------------------------------
 Name:      adrotate_apply_jetpack_photon
 Purpose:   Use Jetpack Photon where available
-------------------------------------------------------------*/
function adrotate_apply_jetpack_photon($image) {
	if(class_exists('Jetpack_Photon') AND Jetpack::is_module_active('photon') AND function_exists('jetpack_photon_url')) {
		return jetpack_photon_url($image);
	} else {
		return $image;
	}
}
?>
