<?php
/* ------------------------------------------------------------------------------------
*  COPYRIGHT AND TRADEMARK NOTICE
*  Copyright 2008-2026 Arnan de Gans. All Rights Reserved.
*  The software AdRotate, related services and ADROTATE the name are owned by Arnan de Gans.

*  COPYRIGHT NOTICES AND ALL THE COMMENTS SHOULD REMAIN INTACT.
*  By using this code you agree to indemnify Arnan de Gans from any
*  liability that might arise from its use.
------------------------------------------------------------------------------------ */

/*-------------------------------------------------------------
 Name:      adrotate_ad
 Purpose:   Show requested ad
-------------------------------------------------------------*/
function adrotate_ad($banner_id, $opt = null) {
	global $wpdb, $adrotate_config, $adrotate_crawlers;

	$output = '';

	if($banner_id) {
		$defaults = array(
			'wrapper' => 'yes', // Group wrapper (yes|no, Default mode)
		);
		$options = wp_parse_args($opt, $defaults);
		$available = true;

		$a = get_option('ajdg_activate_adrotate-pro');

		$banner = $wpdb->get_row($wpdb->prepare("SELECT `id`, `title`, `bannercode`, `tracker`, `show_everyone`, `image`, `crate`, `irate`, `budget` FROM `{$wpdb->prefix}adrotate` WHERE `id` = %d AND (`type` = 'active' OR `type` = '2days' OR `type` = '7days');", $banner_id), ARRAY_A);

		if($banner) {
			if(adrotate_filter_show_everyone($banner)) {
				$available = false;
			}

			if(adrotate_filter_schedule($banner)) {
				$available = false;
			}

			if($adrotate_config['enable_advertisers'] == 'Y' AND ($banner['crate'] > 0 OR $banner['irate'] > 0)) {
				if(adrotate_filter_budget($banner)) {
					$available = false;
				}
			}
		} else {
			$available = false;
		}

		if($available) {
			$image = str_replace('%folder%', $adrotate_config['banner_folder'], $banner['image']);

			if($options['wrapper'] == 'yes') $output .= "<div class=\"a".$adrotate_config['adblock_disguise']."-single a".$adrotate_config['adblock_disguise']."-".$banner['id']."\">";
			$output .= adrotate_ad_output($banner['id'], 0, $banner['title'], $banner['bannercode'], $banner['tracker'], $image);
			if($options['wrapper'] == 'yes') $output .= "</div>";

			if($adrotate_config['stats'] == 1 AND ($banner['tracker'] == 'Y' OR $banner['tracker'] == 'I')) {
				adrotate_count_impression($banner['id'], 0);
			}
		} else {
			$output .= adrotate_error('ad_expired', array('banner_id' => $banner_id));
		}
		unset($banner);
	} else {
		$output .= adrotate_error('ad_no_id');
	}

	return $output;
}

/*-------------------------------------------------------------
 Name:      adrotate_group
 Purpose:   Group output
-------------------------------------------------------------*/
function adrotate_group($group_ids, $opt = null) {
	global $wpdb, $adrotate_config;

	$output = $group_select = $weightoverride = $mobileoverride = $mobileosoverride = $showoverride = '';
	if($group_ids) {
		$defaults = array(
			'fallback' => 0, // Fallback group ID
			'weight' => 0, // Minimum weight (0, 1-10)
		);
		$options = wp_parse_args($opt, $defaults);

		$a = get_option('ajdg_activate_adrotate-pro');
		$now = current_time('timestamp');

		$group_array = (preg_match('/,/is', $group_ids)) ? explode(',', $group_ids) : array($group_ids);
		$group_array = array_filter($group_array);

		foreach($group_array as $key => $value) {
			$group_select .= " `{$wpdb->prefix}adrotate_linkmeta`.`group` = ".$wpdb->prepare('%d', $value)." OR";
		}
		$group_select = rtrim($group_select, " OR");

		// Grab settings to use from first group
		$group = $wpdb->get_row($wpdb->prepare("SELECT * FROM `{$wpdb->prefix}adrotate_groups` WHERE `name` != '' AND `id` = %d;", $group_array[0]));

		if($group) {
			if($group->mobile == 1) {
				if(!adrotate_is_mobile() AND !adrotate_is_tablet()) { // Desktop
					$mobileoverride = "AND `{$wpdb->prefix}adrotate`.`desktop` = 'Y' ";
				} else if(adrotate_is_mobile()) { // Phones
					$mobileoverride = "AND `{$wpdb->prefix}adrotate`.`mobile` = 'Y' ";
				} else if(adrotate_is_tablet()) { // Tablets
					$mobileoverride = "AND `{$wpdb->prefix}adrotate`.`tablet` = 'Y' ";
				}

				if(adrotate_is_mobile() OR adrotate_is_tablet()) {
					if(adrotate_is_ios()) { // iOS
						$mobileosoverride = "AND `{$wpdb->prefix}adrotate`.`os_ios` = 'Y' ";
					} else if(adrotate_is_android()) { // Android
						$mobileosoverride = "AND `{$wpdb->prefix}adrotate`.`os_android` = 'Y' ";
					}
				}
			}

			$weightoverride = ($options['weight'] > 0) ? "AND `{$wpdb->prefix}adrotate`.`weight` >= {$options['weight']} " : "";
			$options['fallback'] = ($options['fallback'] == 0) ? $group->fallback : $options['fallback'];

			// Get all ads in all selected groups
			$ads = $wpdb->get_results(
				"SELECT
					`{$wpdb->prefix}adrotate`.`id`, `title`, `bannercode`, `image`, `tracker`, `show_everyone`, `weight`,
					`crate`, `irate`, `budget`, `state_req`, `cities`, `states`, `countries`, `{$wpdb->prefix}adrotate_linkmeta`.`group`
				FROM
					`{$wpdb->prefix}adrotate`,
					`{$wpdb->prefix}adrotate_linkmeta`
				WHERE
					({$group_select})
					AND `{$wpdb->prefix}adrotate_linkmeta`.`user` = 0
					AND `{$wpdb->prefix}adrotate`.`id` = `{$wpdb->prefix}adrotate_linkmeta`.`ad`
					{$mobileoverride}
					{$mobileosoverride}
					{$weightoverride}
					AND (`{$wpdb->prefix}adrotate`.`type` = 'active'
						OR `{$wpdb->prefix}adrotate`.`type` = '2days'
						OR `{$wpdb->prefix}adrotate`.`type` = '7days')
				GROUP BY `{$wpdb->prefix}adrotate`.`id`
				ORDER BY `{$wpdb->prefix}adrotate`.`id`;"
			, ARRAY_A);

			if($ads) {
				foreach($ads as $key => $ad) {
					if($adrotate_config['duplicate_adverts_filter'] == 'Y') {
						if (is_home() AND !in_the_loop()) {
					    	$session_page = get_option('page_for_posts');
						} elseif (is_post_type_archive() OR is_category()){
							$session_page = get_query_var('cat');
						} else {
							$session_page = get_the_ID();
						}
						$session_page = 'post-'.$session_page;

						// Remove advert if it's in session data
						if(adrotate_filter_duplicates($ad['id'], $session_page)) {
							unset($ads[$key]);
							continue;
						}
					}

					if(adrotate_filter_show_everyone($ad)) {
						unset($ads[$key]);
						continue;
					}

					if(adrotate_filter_schedule($ad)) {
						unset($ads[$key]);
						continue;
					}

					if($adrotate_config['enable_advertisers'] == 'Y' AND ($ad['crate'] > 0 OR $ad['irate'] > 0)) {
						if(adrotate_filter_budget($ad)) {
							unset($ads[$key]);
							continue;
						}
					}

					if($adrotate_config['enable_geo'] > 0 AND $group->geo == 1) {
						if(adrotate_filter_location($ad)) {
							unset($ads[$key]);
							continue;
						}
					}
					
					$ads[$key]['bannercode'] = stripslashes(htmlspecialchars_decode($ads[$key]['bannercode'], ENT_QUOTES));
				}

				$array_count = count($ads);
				if($array_count > 0) {
					$before = $after = '';
					$before = str_replace('%id%', $group_array[0], stripslashes(html_entity_decode($group->wrapper_before, ENT_QUOTES)));
					$after = str_replace('%id%', $group_array[0], stripslashes(html_entity_decode($group->wrapper_after, ENT_QUOTES)));

					$output .= "<div class=\"g".$adrotate_config['adblock_disguise']." g".$adrotate_config['adblock_disguise']."-".$group->id."\">";

					if($group->modus == 1) { // Dynamic ads
						$i = 1;

						// Randomize array
						$ads = adrotate_shuffle($ads);

						foreach($ads as $key => $banner) {
							$image = str_replace('%folder%', $adrotate_config['banner_folder'], $banner['image']);

							$output .= "<div class=\"g".$adrotate_config['adblock_disguise']."-dyn a".$adrotate_config['adblock_disguise']."-".$banner['id']." c-".$i."\">";
							$output .= $before.adrotate_ad_output($banner['id'], $group->id, $banner['title'], $banner['bannercode'], $banner['tracker'], $image).$after;
							$output .= "</div>";
							$i++;
						}
					} else if($group->modus == 2) { // Block of ads
						$block_count = $group->gridcolumns * $group->gridrows;
						if($array_count < $block_count) $block_count = $array_count;
						$columns = 1;

						for($i=1;$i<=$block_count;$i++) {
							// Reset keys, must be here or adrotate_pick_weight won't work
							$ads = array_values($ads);
							$array_key = adrotate_pick_weight($ads);

							$image = str_replace('%folder%', $adrotate_config['banner_folder'], $ads[$array_key]['image']);

							$output .= "<div class=\"g".$adrotate_config['adblock_disguise']."-col b".$adrotate_config['adblock_disguise']."-".$group->id." a".$adrotate_config['adblock_disguise']."-".$ads[$array_key]['id']."\">";
							$output .= $before.adrotate_ad_output($ads[$array_key]['id'], $group->id, $ads[$array_key]['title'], $ads[$array_key]['bannercode'], $ads[$array_key]['tracker'], $image).$after;
							$output .= "</div>";

							if($columns == $group->gridcolumns AND $i != $block_count) {
								$output .= "</div><div class=\"g".$adrotate_config['adblock_disguise']." g".$adrotate_config['adblock_disguise']."-".$group->id."\">";
								$columns = 1;
							} else {
								$columns++;
							}

							if($adrotate_config['stats'] == 1 AND ($ads[$array_key]['tracker'] == 'Y' OR $ads[$array_key]['tracker'] == 'I')) {
								adrotate_count_impression($ads[$array_key]['id'], $group->id);
							}

							// Store advert ID's in session
							if($adrotate_config['duplicate_adverts_filter'] == 'Y') {
								$adrotate_session = get_transient('adrotate_'.md5(adrotate_get_remote_ip()));
								if($adrotate_session === false) $adrotate_session = array();

								// Add new data
								$adrotate_session['duplicate-ads'][$session_page]['adverts'][] = $ads[$array_key]['id'];
								$adrotate_session['duplicate-ads'][$session_page]['timeout'] = current_time('timestamp');

								$ttl = ini_get('session.gc_maxlifetime');
								if(empty($ttl)) $ttl = 1440; // Set some default if PHP doesn't know
								set_transient('adrotate_'.md5(adrotate_get_remote_ip()), $adrotate_session, $ttl);
							}

							// Remove the advert so it cannot be picked again
							unset($ads[$array_key]);
						}
					} else if($group->modus == 3) { // Wallpaper ad
						$array_key = array_rand($ads, 1);

						$image = str_replace('%folder%', $adrotate_config['banner_folder'], $ads[$array_key]['image']);
						
						preg_match_all('/<a[^>](?:.*?)>/i', $ads[$array_key]['bannercode'], $matches, PREG_SET_ORDER);
						if(isset($matches[0])) {
							$ads[$array_key]['bannercode'] = str_replace('<a ', '<a style="margin:0 auto; position:fixed; top:0; width:100%; height:100%;" ', $ads[$array_key]['bannercode']);
						}

						$output .= "<div class=\"g-wall a-".$ads[$array_key]['id']."\" style=\"background: url(".$image.") no-repeat 50% 0;\">";
						$output .= adrotate_ad_output($ads[$array_key]['id'], $group->id, $ads[$array_key]['title'], $ads[$array_key]['bannercode'], $ads[$array_key]['tracker'], $image);
						$output .= "</div>";

						if($adrotate_config['stats'] == 1 AND $ads[$array_key]['tracker'] == 'Y') {
							adrotate_count_impression($ads[$array_key]['id'], $group->id);
						}
					} else { // Default (single ad)
						// Reset keys, must be here or adrotate_pick_weight won't work
						$ads = array_values($ads);
						$array_key = adrotate_pick_weight($ads);

						$image = str_replace('%folder%', $adrotate_config['banner_folder'], $ads[$array_key]['image']);

						$output .= "<div class=\"g".$adrotate_config['adblock_disguise']."-single a".$adrotate_config['adblock_disguise']."-".$ads[$array_key]['id']."\">";
						$output .= $before.adrotate_ad_output($ads[$array_key]['id'], $group->id, $ads[$array_key]['title'], $ads[$array_key]['bannercode'], $ads[$array_key]['tracker'], $image).$after;
						$output .= "</div>";

						if($adrotate_config['stats'] == 1 AND ($ads[$array_key]['tracker'] == 'Y' OR $ads[$array_key]['tracker'] == 'I')) {
							adrotate_count_impression($ads[$array_key]['id'], $group->id);
						}

						// Store advert ID's in session
						if($adrotate_config['duplicate_adverts_filter'] == 'Y') {
							$adrotate_session = get_transient('adrotate_'.md5(adrotate_get_remote_ip()));
							if($adrotate_session === false) $adrotate_session = array();

							// Add new data
							$adrotate_session['duplicate-ads'][$session_page]['adverts'][] = $ads[$array_key]['id'];
							$adrotate_session['duplicate-ads'][$session_page]['timeout'] = current_time('timestamp');

							$ttl = ini_get('session.gc_maxlifetime');
							if(empty($ttl)) $ttl = 1440; // Set some default if PHP doesn't know
							set_transient('adrotate_'.md5(adrotate_get_remote_ip()), $adrotate_session, $ttl);
						}
					}

					$output .= '</div>';

					unset($banner_id);
				} else {
					$output .= adrotate_fallback($options['fallback'], 'expired');
				}
			} else {
				$output .= adrotate_fallback($options['fallback'], 'unqualified');
			}
		} else {
			$output .= adrotate_error('group_not_found', array('group_id' => $group_array[0]));
		}
	} else {
		$output .= adrotate_error('group_no_id');
	}

	return $output;
}

/*-------------------------------------------------------------
 Name:      adrotate_shortcode
 Purpose:   Prepare function requests for calls on shortcodes
-------------------------------------------------------------*/
function adrotate_shortcode($atts, $content = null) {
	global $adrotate_config;

	$banner_id = (!empty($atts['banner'])) ? trim($atts['banner'], '\r\t ') : 0;
	$group_ids = (!empty($atts['group'])) ? trim($atts['group'], '\r\t ') : 0;
	$fallback = (!empty($atts['fallback'])) ? trim($atts['fallback'], '\r\t ') : 0; // Optional: for groups (ID)
	$weight	= (!empty($atts['weight']))	? trim($atts['weight'], '\r\t ') : 0; // Optional: for groups (0, 1-10)
	$wrapper = (!empty($atts['wrapper'])) ? trim($atts['wrapper'], '\r\t ') : 'yes'; // Optional: for inline advert (yes|no, single advert only)

	$output = '';
	if($adrotate_config['w3caching'] == 'Y') {
		$output .= "<!-- mfunc ".W3TC_DYNAMIC_SECURITY." -->";

		if($banner_id > 0 AND ($group_ids == 0 OR $group_ids > 0)) { // Show one Ad
			$output .= "echo adrotate_ad(".$banner_id.", array('wrapper' => '".$wrapper."'));";
		}

		if($banner_id == 0 AND $group_ids > 0) { // Show group
			$output .= "echo adrotate_group(".$group_ids.", array('fallback' => '".$fallback."', 'weight' => '".$weight."'));";
		}

		$output .= "<!-- /mfunc ".W3TC_DYNAMIC_SECURITY." -->";
	} else if($adrotate_config['borlabscache'] == 'Y' AND function_exists('BorlabsCacheHelper')) {
		if(BorlabsCacheHelper()->willFragmentCachingPerform()) {
			$borlabsphrase = BorlabsCacheHelper()->getFragmentCachingPhrase();

			$output .= "<!--[borlabs cache start: ".$borlabsphrase."]-->";
			if($banner_id > 0 AND ($group_ids == 0 OR $group_ids > 0)) { // Show one Ad
				$output .= "echo adrotate_ad(".$banner_id.", array('wrapper' => '".$wrapper."'));";
			}
			if($banner_id == 0 AND $group_ids > 0) { // Show group
				$output .= "echo adrotate_group(".$group_ids.", array('fallback' => '".$fallback."', 'weight' => '".$weight."'));";
			}
			$output .= "<!--[borlabs cache end: ".$borlabsphrase."]-->";

			unset($borlabsphrase);
		}
	} else {
		if($banner_id > 0 AND ($group_ids == 0 OR $group_ids > 0)) { // Show one Ad
			$output .= adrotate_ad($banner_id, array('wrapper' => $wrapper));
		}

		if($banner_id == 0 AND $group_ids > 0) { // Show group
			$output .= adrotate_group($group_ids, array('fallback' => $fallback, 'weight' => $weight));
		}
	}

	return $output;
}

/*-------------------------------------------------------------
 Name:      adrotate_inject_posts_cache_wrapper
 Purpose:   Wrap post injection return with caching code?
 Since:		5.10
-------------------------------------------------------------*/
function adrotate_inject_posts_cache_wrapper($group_id) {
	global $adrotate_config;

	if($adrotate_config['w3caching'] == 'Y') {
		$advert_output = "<!-- mfunc ".W3TC_DYNAMIC_SECURITY." -->";
		$advert_output .= "echo adrotate_group(".$group_id.");";
		$advert_output .= "<!-- /mfunc ".W3TC_DYNAMIC_SECURITY." -->";
	} else if($adrotate_config['borlabscache'] == 'Y' AND function_exists('BorlabsCacheHelper')) {
		if(BorlabsCacheHelper()->willFragmentCachingPerform()) {
			$borlabsphrase = BorlabsCacheHelper()->getFragmentCachingPhrase();

			$advert_output = "<!--[borlabs cache start: ".$borlabsphrase."]-->";
			$advert_output .= "echo adrotate_group(".$group_id.");";
			$advert_output .= "<!--[borlabs cache end: ".$borlabsphrase."]-->";

			unset($borlabsphrase);
		}
	} else {
		$advert_output = adrotate_group($group_id);
	}

	return $advert_output;
}

/*-------------------------------------------------------------
 Name:      adrotate_inject_posts
 Purpose:   Add an advert to a single page or post
-------------------------------------------------------------*/
function adrotate_inject_posts($post_content) {
	global $wpdb, $post;

	$categories_top = $categories_bottom = $categories_inside = array();
	if(is_page()) {
		// Inject ads into pages
		$groups = $wpdb->get_results("SELECT `id`, `page`, `page_loc`, `page_par` FROM `{$wpdb->prefix}adrotate_groups` WHERE `page_loc` > 0 AND  `page_loc` < 5;");

		foreach($groups as $group) {
			$pages_more = explode(',', $group->page);

			if(count($pages_more) > 0) {
				if(in_array($post->ID, $pages_more)) {
					if($group->page_loc == 1 OR $group->page_loc == 3) {
						$categories_top[$group->id] = $group->page_par;
					}
					if($group->page_loc == 2 OR $group->page_loc == 3) {
						$categories_bottom[$group->id] = $group->page_par;
					}
					if($group->page_loc == 4) {
						$categories_inside[$group->id] = $group->page_par;
					}
					unset($pages_more, $group);
				}
			}
		}
	}

	if(is_single()) {
		// Inject ads into posts in specified category
		$groups = $wpdb->get_results("SELECT `id`, `cat`, `cat_loc`, `cat_par` FROM `{$wpdb->prefix}adrotate_groups` WHERE `cat_loc` > 0 AND `cat_loc` < 5;");
		$wp_categories = wp_get_post_categories($post->ID, array('taxonomy' => 'category', 'fields' => 'ids'));

		foreach($groups as $group) {
			$categories_more = array_intersect($wp_categories, explode(',', $group->cat));

			if(count($categories_more) > 0) {
				if(has_category($categories_more, $post->ID)) {
					if(($group->cat_loc == 1 OR $group->cat_loc == 3)) {
						$categories_top[$group->id] = $group->cat_par;
					}
					if($group->cat_loc == 2 OR $group->cat_loc == 3) {
						$categories_bottom[$group->id] = $group->cat_par;
					}
					if($group->cat_loc == 4) {
						$categories_inside[$group->id] = $group->cat_par;
					}
					unset($categories_more, $group);
				}
			}
		}
	}

	// Advert in front of content
	if(count($categories_top) > 0) {
		$post_content = adrotate_inject_posts_cache_wrapper(array_rand($categories_top)).$post_content;
	}

	// Advert behind the content
	if(count($categories_bottom) > 0) {
		$post_content = $post_content.adrotate_inject_posts_cache_wrapper(array_rand($categories_bottom));
	}

	// Adverts inside the content
	if(count($categories_inside) > 0) {
		// Setup
		$categories_inside = adrotate_shuffle($categories_inside);
	    $post_content_exploded = explode('</p>', $post_content);
		$post_content_count = ceil(count($post_content_exploded));
		$inserted = array();

		// Determine after which paragraphs ads should show
		foreach($categories_inside as $group_id => $group_paragraph) {
			if($group_paragraph == 99) {
				$insert_paragraph = intval($post_content_count / 2); // Middle of content
			} else if($group_paragraph == 100) {
				$insert_paragraph = 3;
			} else if($group_paragraph == 101) {
				$insert_paragraph = 4;
			} else if($group_paragraph == 102) {
				$insert_paragraph = 5;
			} else if($group_paragraph == 103) {
				$insert_paragraph = 6;
			} else if($group_paragraph == 104) {
				$insert_paragraph = 7;
			} else if($group_paragraph == 105) {
				$insert_paragraph = 8;
			} else if($group_paragraph == 110) {
				$insert_paragraph = intval(floor($post_content_count / 3)); // 30% down
			} else if($group_paragraph == 111) {
				$insert_paragraph = intval(floor($post_content_count / 3) * 2); // 60% down
			} else {
				$insert_paragraph = intval($group_paragraph);
			}

			// Create $inserted with paragraphs numbers and link the group to it. This list is leading from this point on.
			if($group_paragraph > 99 AND $group_paragraph < 110) {
				for($i=$insert_paragraph;$i<=($post_content_count-4);$i+=$insert_paragraph) {
					if(!array_key_exists($i, $inserted)) {
						$inserted[$i] = $group_id;
					}
				}
				unset($i);
			} else {
				if(!array_key_exists($insert_paragraph, $inserted)) {
					$inserted[$insert_paragraph] = $group_id;
				}
			}
			unset($group_id, $insert_paragraph, $group_paragraph);
		}

		// Inject ads behind paragraphs based on $inserted created above, IF a group_id is set higher than 0
		foreach($post_content_exploded as $index => $paragraph) {
			$insert_here = $index + 1; // Deal with array offset
			if(array_key_exists($insert_here, $inserted)) {
				if($inserted[$insert_here] > 0) {
					$post_content_exploded[$index] .= adrotate_inject_posts_cache_wrapper($inserted[$insert_here]);
					$inserted[$insert_here] = 0;
				}
			}
			unset($index, $paragraph, $insert_here);
		}

		// Re-assemble post_content and clean up
	    $post_content = implode('', $post_content_exploded);
		unset($post_content_exploded, $post_content_count, $inserted);

	}

	unset($groups, $categories_top, $categories_bottom, $categories_inside);

	return $post_content;
}

/*-------------------------------------------------------------
 Name:      adrotate_inject_products
 Purpose:   Add adverts to a WooCommerce or Classic Commerce product page
 Since:		5.10
-------------------------------------------------------------*/
function adrotate_inject_products() {
	global $wpdb, $product;

	if(function_exists('is_product')) {
		if(is_product()) {
			$categories_top = $categories_bottom = array();

			// Inject ads into posts in specified product category
			$groups = $wpdb->get_results("SELECT `id`, `woo_cat`, `woo_loc` FROM `{$wpdb->prefix}adrotate_groups` WHERE `woo_loc` > 0 AND `woo_loc` < 4;");

			foreach($groups as $group) {
				$categories_more = array_intersect($product->get_category_ids(), explode(',', $group->woo_cat));

				if(count($categories_more) > 0) {
					if($group->woo_loc == 1 OR $group->woo_loc == 3) {
						$categories_top[$group->id] = $categories_more;
					}
					if($group->woo_loc == 2 OR $group->woo_loc == 3) {
						$categories_bottom[$group->id] = $categories_more;
					}
					unset($categories_more, $group);
				}
			}

			// Advert before the content
			if (count($categories_top) > 0 AND current_filter() == 'woocommerce_before_single_product') {
				echo adrotate_inject_posts_cache_wrapper(array_rand($categories_top));
			}

			// Advert behind the content
			if (count($categories_bottom) > 0 AND current_filter() == 'woocommerce_after_single_product') {
				echo adrotate_inject_posts_cache_wrapper(array_rand($categories_bottom));
			}

			unset($groups, $categories_top, $categories_bottom);
		}
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_inject_forums
 Purpose:   Add adverts to a bbPress forum page/topic
 Since:		5.10
-------------------------------------------------------------*/
function adrotate_inject_forums() {
	global $wpdb, $post;

	if(function_exists('is_bbpress')) {
		if(is_bbpress()) {
			$forums_top = $forums_bottom = array();

			// Inject ads into posts in specified forum
			$groups = $wpdb->get_results("SELECT `id`, `bbpress`, `bbpress_loc` FROM `{$wpdb->prefix}adrotate_groups` WHERE `bbpress_loc` > 0 AND `bbpress_loc` < 4;");

			foreach($groups as $group) {
				$forums_more = explode(',', $group->bbpress);

				if(count($forums_more) > 0) {
					if(in_array($post->ID, $forums_more) OR in_array($post->post_parent, $forums_more)) {
						if($group->bbpress_loc == 1 OR $group->bbpress_loc == 3) {
							$forums_top[$group->id] = $group->bbpress;
						}
						if($group->bbpress_loc == 2 OR $group->bbpress_loc == 3) {
							$forums_bottom[$group->id] = $group->bbpress;
						}
					}
					unset($forums_more, $group);
				}
			}

			// Advert before the content
			if(count($forums_top) > 0 AND (current_filter() == 'bbp_template_before_topics_loop' OR current_filter() == 'bbp_template_before_replies_loop')) {
				echo adrotate_inject_posts_cache_wrapper(array_rand($forums_top));
			}

			// Advert behind the content
			if(count($forums_bottom) > 0 AND (current_filter() == 'bbp_template_after_topics_loop' OR current_filter() == 'bbp_template_after_replies_loop')) {
				echo adrotate_inject_posts_cache_wrapper(array_rand($forums_bottom));
			}

			unset($groups, $forums_top, $forums_bottom);
		}
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_preview
 Purpose:   Show preview of selected ad (Dashboard)
-------------------------------------------------------------*/
function adrotate_preview($banner_id) {
	global $wpdb, $adrotate_config;

	if($banner_id) {
		$now = current_time('timestamp');

		$edit_banner = $wpdb->get_row($wpdb->prepare("SELECT * FROM `{$wpdb->prefix}adrotate` WHERE `id` = %d;", $banner_id));

		if($edit_banner) {
			if(preg_match_all('/<(ins|script)(.*?)>|onclick\=|onload\=/i', stripslashes(htmlspecialchars_decode($edit_banner->bannercode, ENT_QUOTES)), $things)) {
				$output = "<div class=\"preview-wrapper row_blue\"><div class=\"preview-inner\">Adverts with JavaScript or a &lt;ins&gt; tag can not be previewed!</div></div>";
			} else {
				$image = str_replace('%folder%', $adrotate_config['banner_folder'], $edit_banner->image);
				$output = adrotate_ad_output($edit_banner->id, 0, $edit_banner->title, $edit_banner->bannercode, $edit_banner->tracker, $image);
			}
		} else {
			$output = adrotate_error('ad_expired', array('banner_id' => $banner_id));
		}
	} else {
		$output = adrotate_error('ad_no_id');
	}

	return $output;
}

/*-------------------------------------------------------------
 Name:      adrotate_ad_output
 Purpose:   Prepare the output for viewing
-------------------------------------------------------------*/
function adrotate_ad_output($ad_id, $group_id, $name, $bannercode, $tracker, $image) {
	global $adrotate_config;

	$banner_output = $bannercode;
	$banner_output = stripslashes(htmlspecialchars_decode($banner_output, ENT_QUOTES));

	if($adrotate_config['stats'] > 0 AND $tracker != 'N') {
		// Internal tracker
		if($adrotate_config['stats'] == 1 AND ($tracker == 'Y' OR $tracker == 'C')) {
			preg_match_all('/<a[^>](?:.*?)>/i', $banner_output, $matches, PREG_SET_ORDER);
			if(isset($matches[0])) {
				$banner_output = str_replace('<a ', '<a data-track="'.base64_encode($ad_id.','.$group_id.','.$adrotate_config['impression_timer']).'" ', $banner_output);
				foreach($matches[0] as $value) {
					if(preg_match('/<a[^>]+class=\"(.+?)\"[^>]*>/i', $value, $regs)) {
					    $result = $regs[1]." gofollow";
						$banner_output = str_ireplace('class="'.$regs[1].'"', 'class="'.$result.'"', $banner_output);
					} else {
						$banner_output = str_ireplace('<a ', '<a class="gofollow" ', $banner_output);
					}
					unset($value, $regs, $result);
				}
			}
		}

		// Google Analytics || Matomo
		if($adrotate_config['stats'] >= 2 AND $adrotate_config['stats'] <= 5) { 
			$click_event = $impression_event = '';
			preg_match_all('/<(?:a|img|iframe)[^>](?:.*?)>/i', $banner_output, $matches, PREG_SET_ORDER);

			if(isset($matches[0])) {
				if($adrotate_config['stats'] == 2) { // Matomo
					if($tracker == 'Y' OR $tracker == 'C') {
						$click_event = "_paq.push(['trackEvent', 'Adverts', 'Click', '$name']);";
					}
					if($tracker == 'Y' OR $tracker == 'I') {
						$impression_event = "_paq.push(['trackEvent', 'Adverts', 'Impression', '$name']);";
					}
				}

				if($adrotate_config['stats'] == 3) { // gtag.js (Global Tag for GA4)
					if($tracker == 'Y' OR $tracker == 'C') {
						$click_event = "gtag('event', 'click', {'advert_name': '$name'});";
					}
					if($tracker == 'Y' OR $tracker == 'I') {
						$impression_event = "gtag('event', 'impression', {'advert_name': '$name'});";
					}
				}

				if($adrotate_config['stats'] == 5) { // gtm.js (Tag Manager for GA4)
					if($tracker == 'Y' OR $tracker == 'C') {
						$click_event = "dataLayer.push({'event': 'AdRotatePro', 'advert_interaction': 'click', 'advert_name': '$name'});";
					}
					if($tracker == 'Y' OR $tracker == 'I') {
						$impression_event = "dataLayer.push({'event': 'AdRotatePro', 'advert_interaction': 'impression', 'advert_name': '$name'});";
					}
				}

				// Image banner or Text banner
				if(strlen($click_event) > 0 AND stripos($banner_output, '<a') !== false) {
					if(!preg_match('/<a[^>]+onClick[^>]*>/i', $banner_output, $url)) {
						$banner_output = str_ireplace('<a ', '<a onClick="'.$click_event.'" ', $banner_output);
					}
				}

				// Add tracking pixel (Most ads, including <iframe> and <ins> ads)?
				if(strlen($impression_event) > 0) {
					$banner_output .= "<img class=\"pixel\" style=\"width:0 !important; height:0 !important;\" width=\"0\" height=\"0\" src=\"data:image/gif;base64,R0lGODlhAQABAIAAAP///////yH5BAEKAAEALAAAAAABAAEAAAICTAEAOw==\" onload=\"".$impression_event."\" alt=\"\" />";
				}
				unset($url, $img, $click_event, $impression_event);
			}
		}
		unset($matches);
	}

	$image = apply_filters('adrotate_apply_photon', $image);

	$banner_output = str_replace('%title%', $name, $banner_output);
	$banner_output = str_replace('%random%', rand(100000,999999), $banner_output);
	$banner_output = str_replace('%asset%', $image, $banner_output);
	$banner_output = str_replace('%id%', $ad_id, $banner_output);
	$banner_output = do_shortcode($banner_output);

	return $banner_output;
}

/*-------------------------------------------------------------
 Name:      adrotate_fallback
 Purpose:   Fall back to the set group or show an error if no fallback is set
-------------------------------------------------------------*/
function adrotate_fallback($group_id, $case) {

	$fallback_output = '';
	if($group_id > 0) {
		$fallback_output = adrotate_group($group_id);
	} else {
		if($case == 'expired') {
			$fallback_output = adrotate_error('ad_expired', array('banner_id' => 'n/a'));
		}

		if($case == 'unqualified') {
			$fallback_output = adrotate_error('ad_unqualified');
		}
	}

	return $fallback_output;
}

/*-------------------------------------------------------------
 Name:      adrotate_header
 Purpose:   Add required CSS to wp_head (action)
-------------------------------------------------------------*/
function adrotate_header() {
	if(!function_exists('get_plugins')){
		require_once(ABSPATH . 'wp-admin/includes/plugin.php');
	}

	$plugin_version = get_plugins();
	$plugin_version = $plugin_version['adrotate-pro/adrotate-pro.php']['Version'];

	$output = "\n<!-- This site is using AdRotate Professional v".$plugin_version." to display their advertisements - https://ajdg.solutions/ -->\n";
	$output .= adrotate_custom_css();

	$header = get_option('adrotate_header_output', false);
	if($header) {
		$header = stripslashes(htmlspecialchars_decode($header, ENT_QUOTES));
		$header = str_replace('%random%', rand(100000,999999), $header);
		$output .= $header."\n";
		unset($header);
	}

	$gam = get_option('adrotate_gam_output', false);
	if($gam) {
		$gam = stripslashes(htmlspecialchars_decode($gam, ENT_QUOTES));
		$gam = str_replace('%random%', rand(100000,999999), $gam);
		$output .= $gam."\n\n";
		unset($gam);
	}
	echo $output;
}

/*-------------------------------------------------------------
 Name:      adrotate_custom_css
 Purpose:   Add group CSS to adrotate_header()
-------------------------------------------------------------*/
function adrotate_custom_css() {
	global $wpdb, $adrotate_config;

	$a = get_option('ajdg_activate_adrotate-pro');
	$generated_css = get_option('adrotate_group_css', false);

	$output = "<!-- AdRotate CSS -->\n";
	$output .= "<style type=\"text/css\" media=\"screen\">\n";
	$output .= "\t.g".$adrotate_config['adblock_disguise']." { margin:0px; padding:0px; overflow:hidden; line-height:1; zoom:1; }\n";
	$output .= "\t.g".$adrotate_config['adblock_disguise']." img { height:auto; }\n";
	$output .= "\t.g".$adrotate_config['adblock_disguise']."-col { position:relative; float:left; }\n";
	$output .= "\t.g".$adrotate_config['adblock_disguise']."-col:first-child { margin-left: 0; }\n";
	$output .= "\t.g".$adrotate_config['adblock_disguise']."-col:last-child { margin-right: 0; }\n";
	$output .= "\t.g".$adrotate_config['adblock_disguise']."-wall { display:block; position:fixed; left:0; top:0; width:100%; height:100%; z-index:".$adrotate_config['zindex']."; }\n";
	$output .= "\t.woocommerce-page .g".$adrotate_config['adblock_disguise'].", .bbpress-wrapper .g".$adrotate_config['adblock_disguise']." { margin: 20px auto; clear:both; }\n";

	if($generated_css) {
		foreach($generated_css as $group_id => $css) {
			if(strlen($css) > 0) {
				$output .= $css;
			}
		}
		unset($generated_css);
	}

	$output .= "\t@media only screen and (max-width: 480px) {\n";
	$output .= "\t\t.g".$adrotate_config['adblock_disguise']."-col, .g".$adrotate_config['adblock_disguise']."-dyn, .g".$adrotate_config['adblock_disguise']."-single { width:100%; margin-left:0; margin-right:0; }\n";
	$output .= "\t\t.woocommerce-page .g".$adrotate_config['adblock_disguise'].", .bbpress-wrapper .g".$adrotate_config['adblock_disguise']." { margin: 10px auto; }\n";
	$output .= "\t}\n";

	if($adrotate_config['widgetpadding'] == 'Y') {
		$advert_string = get_option('adrotate_dynamic_widgets_advert', 'temp_1');
		$group_string = get_option('adrotate_dynamic_widgets_group', 'temp_2');
		$output .= ".ajdg_bnnrwidgets, .ajdg_grpwidgets { overflow:hidden; padding:0; }\n";
		$output .= ".".$advert_string.", .".$group_string." { overflow:hidden; padding:0; }\n";
	}

	$output .= "</style>\n";
	$output .= "<!-- /AdRotate CSS -->\n\n";

	return $output;
}

/*-------------------------------------------------------------
 Name:      adrotate_scripts
 Purpose:   Add required scripts to wp_enqueue_scripts (action)
-------------------------------------------------------------*/
function adrotate_scripts() {
	global $adrotate_config;

	$in_footer = ($adrotate_config['jsfooter'] == 'Y') ? true : false;

	if($adrotate_config['jquery'] == 'Y') {
		wp_enqueue_script('jquery', false, false, null, $in_footer);
	}

	if(get_option('adrotate_dynamic_required') > 0) {
		wp_enqueue_script('adrotate-groups', plugins_url('/library/jquery.groups.js', __FILE__), false, null, $in_footer);
	}

	if($adrotate_config['stats'] == 1) {
		wp_enqueue_script('adrotate-clicker', plugins_url('/library/jquery.clicker.js', __FILE__), false, null, $in_footer);
		wp_localize_script('adrotate-clicker', 'click_object', array('ajax_url' => admin_url('admin-ajax.php')));
		wp_localize_script('adrotate-groups', 'impression_object', array('ajax_url' => admin_url('admin-ajax.php')));
	}

	if(!$in_footer) {
		add_action('wp_head', 'adrotate_custom_javascript');
	} else {
		add_action('wp_footer', 'adrotate_custom_javascript', 100);
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_custom_javascript
 Purpose:   Add required JavaScript to adrotate_scripts()
-------------------------------------------------------------*/
function adrotate_custom_javascript() {
	global $wpdb, $adrotate_config;

	$a = get_option('ajdg_activate_adrotate-pro');
	$groups = $wpdb->get_results("SELECT `id`, `adspeed`, `repeat_impressions` FROM `{$wpdb->prefix}adrotate_groups` WHERE `name` != '' AND `modus` = 1 ORDER BY `id` ASC;", ARRAY_A);

	if(count($groups) > 0) {
		$output = "<!-- AdRotate JS -->\n";
		$output .= "<script type=\"text/javascript\">\n";
		$output .= "jQuery(document).ready(function(){if(jQuery.fn.gslider) {\n";
		foreach($groups as $group) {
			$output .= "\tjQuery('.g".$adrotate_config['adblock_disguise']."-".$group['id']."').gslider({groupid:".$group['id'].",speed:".$group['adspeed'].",repeat_impressions:'".$group['repeat_impressions']."'});\n";
		}
		$output .= "}});\n";
		$output .= "</script>\n";
		$output .= "<!-- /AdRotate JS -->\n\n";
		unset($groups);
		echo $output;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_custom_profile_fields
 Purpose:   Add profile fields to user creation and editing dashboards
-------------------------------------------------------------*/
function adrotate_custom_profile_fields($user) {
	global $adrotate_config;

    if(current_user_can('adrotate_advertiser_manage') AND $adrotate_config['enable_advertisers'] == 'Y') {
		if($user != 'add-new-user') {
		    $advertiser = get_user_meta($user->ID, 'adrotate_is_advertiser', 1);
		    $permissions = get_user_meta($user->ID, 'adrotate_permissions', 1);
		    // Check for gaps
		    if(empty($advertiser)) $advertiser = 'N';
		    if(empty($permissions)) $permissions = array('create' => 'N', 'edit' => 'N', 'advanced' => 'N', 'geo' => 'N', 'group' => 'N', 'schedule' => 'N');
			if(!isset($permissions['create'])) $permissions['create'] = 'N';
			if(!isset($permissions['edit'])) $permissions['edit'] = 'N';
			if(!isset($permissions['advanced'])) $permissions['advanced'] = 'N';
			if(!isset($permissions['geo'])) $permissions['geo'] = 'N';
			if(!isset($permissions['group'])) $permissions['group'] = 'N';
			if(!isset($permissions['schedule'])) $permissions['schedule'] = 'N';
		    $notes = get_user_meta($user->ID, 'adrotate_notes', 1);
		} else {
			$advertiser = 'N';
			$permissions = array('create' => 'N', 'edit' => 'N', 'advanced' => 'N', 'geo' => 'N', 'group' => 'N', 'schedule' => 'N');
			$notes = '';
		}
		?>
	    <h3><?php esc_html_e("AdRotate Advertiser", 'adrotate-pro'); ?></h3>
	    <table class="form-table">
	      	<tr>
		        <th valign="top"><?php esc_html_e("Enable", 'adrotate-pro'); ?></th>
		        <td>
		        	<label for="adrotate_is_advertiser"><input tabindex="100" type="checkbox" name="adrotate_is_advertiser" <?php if($advertiser == 'Y') { ?>checked="checked" <?php } ?> /> <?php esc_html_e("Is this user an AdRotate Advertiser?", 'adrotate-pro'); ?></label><br />
		        </td>
	      	</tr>
	      	<tr>
		        <th valign="top"><?php esc_html_e("Permissions", 'adrotate-pro'); ?></th>
		        <td>
		        	<label for="adrotate_can_create"><input tabindex="101" type="checkbox" name="adrotate_can_create" <?php if($permissions['create'] == 'Y') { ?>checked="checked" <?php } ?> /> <?php esc_html_e("Can create adverts?", 'adrotate-pro'); ?></label><br />
		        	<label for="adrotate_can_edit"><input tabindex="102" type="checkbox" name="adrotate_can_edit" <?php if($permissions['edit'] == 'Y') { ?>checked="checked" <?php } ?> /> <?php esc_html_e("Can edit their own adverts?", 'adrotate-pro'); ?></label>
		        </td>
	      	</tr>
	      	<tr>
		        <th valign="top"><?php esc_html_e("Advert Features", 'adrotate-pro'); ?></th>
		        <td>
		        	<label for="adrotate_can_advanced"><input tabindex="103" type="checkbox" name="adrotate_can_advanced" <?php if($permissions['advanced'] == 'Y') { ?>checked="checked" <?php } ?> /> <?php esc_html_e("Can change advanced settings in adverts?", 'adrotate-pro'); ?></label><br />
		        	<label for="adrotate_can_geo"><input tabindex="104" type="checkbox" name="adrotate_can_geo" <?php if($permissions['geo'] == 'Y') { ?>checked="checked" <?php } ?> /> <?php esc_html_e("Can change Geo Targeting?", 'adrotate-pro'); ?></label><br />
		        	<label for="adrotate_can_group"><input tabindex="105" type="checkbox" name="adrotate_can_group" <?php if($permissions['group'] == 'Y') { ?>checked="checked" <?php } ?> /> <?php esc_html_e("Can change schedules in adverts?", 'adrotate-pro'); ?></label><br />
		        	<label for="adrotate_can_schedule"><input tabindex="106" type="checkbox" name="adrotate_can_schedule" <?php if($permissions['schedule'] == 'Y') { ?>checked="checked" <?php } ?> /> <?php esc_html_e("Can change groups in adverts?", 'adrotate-pro'); ?></label>
		        </td>
	      	</tr>
		    <tr>
				<th valign="top"><label for="adrotate_notes"><?php esc_html_e("Notes", 'adrotate-pro'); ?></label></th>
				<td>
					<textarea tabindex="104" name="adrotate_notes" cols="50" rows="5"><?php echo esc_attr($notes); ?></textarea><br />
					<em><?php esc_html_e("Also visible in the advertiser profile.", 'adrotate-pro'); ?></em>
					</td>
			</tr>
	    </table>
<?php
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_nonce_error
 Purpose:   Display a formatted error if Nonce fails
-------------------------------------------------------------*/
function adrotate_nonce_error() {
	$message = "WordPress was unable to verify the authenticity of the url you have clicked. Verify if the url used is valid or log in via your browser.<br />".
	"Contact AdRotate Pro support if the issue persists: <a href=\"https://support.ajdg.net\" title=\"AdRotate Support\" target=\"_blank\">AJdG Solutions Support</a>.";
	wp_die($message);
}

/*-------------------------------------------------------------
 Name:      adrotate_error
 Purpose:   Show errors for problems in using AdRotate, should they occur
-------------------------------------------------------------*/
function adrotate_error($action, $arg = null) {
	switch($action) {
		// Ads
		case 'ad_expired' :
			$result = "<!-- ".sprintf(esc_html__("Error, Ad (%s) is not available at this time due to schedule/budgeting/geolocation/mobile restrictions!", 'adrotate-pro'), $arg['banner_id'])." -->";
			return $result;
		break;

		case 'ad_unqualified' :
			$result = "<!-- ".esc_html__("Either there are no banners, they are disabled or none qualified for this location!", 'adrotate-pro')." -->";
			return $result;
		break;

		case 'ad_no_id' :
			$result = "<span style=\"font-weight: bold; color: #f00;\">".esc_html__("Error, no Ad ID set! Check your syntax!", 'adrotate-pro')."</span>";
			return $result;
		break;

		// Groups
		case 'group_no_id' :
			$result = "<span style=\"font-weight: bold; color: #f00;\">".esc_html__("Error, no group ID set! Check your syntax!", 'adrotate-pro')."</span>";
			return $result;
		break;

		case 'group_not_found' :
			$result = "<span style=\"font-weight: bold; color: #f00;\">".esc_html__("Error, group does not exist! Check your syntax!", 'adrotate-pro')." (ID: ".$arg['group_id'].")</span>";
			return $result;
		break;

		// Database
		case 'db_error' :
			$result = "<span style=\"font-weight:bold; color:#f00;\">".esc_html__("There was an error locating the database tables for AdRotate. Please deactivate and re-activate AdRotate from the plugin page!!", 'adrotate-pro')."<br />".esc_html__("If this does not solve the issue please create a support ticket at", 'adrotate-pro')." <a href=\"https://support.ajdg.net\" target=\"_blank\">Ticket support</a></span>";
			return $result;
		break;

		// Possible XSS or malformed URL
		case 'error_loading_item' :
			$result = "<span style=\"font-weight:bold; color:#f00;\">".esc_html__("There was an error loading the page. Please try again by reloading the page via the menu on the left.", 'adrotate-pro')."<br />".esc_html__("If the issue persists please seek help at", 'adrotate-pro')." <a href=\"https://support.ajdg.net\" target=\"_blank\">Ticket support</a></span>";
			return $result;
		break;

		// Misc
		default:
			$result = "<span style=\"font-weight:bold; color:#f00;\">".esc_html__("An unknown error occured.", 'adrotate-pro')."</span>";
			return $result;
		break;
	}
}
?>
