<?php
/*
Plugin Name: AdRotate Professional
Plugin URI: https://ajdg.solutions/product-category/adrotate-pro/
Author: Arnan de Gans
Author URI: https://www.arnan.me/
Description: AdRotate Pro is the popular choice for monetizing your website with adverts while keeping things simple.
Version: 5.27
License: Limited License

Text Domain: adrotate-pro
Domain Path: /languages

Requires at least: 6.0
Requires PHP: 8.0
GooseUp: ajdg.solutions
*/

/* ------------------------------------------------------------------------------------
*  COPYRIGHT AND TRADEMARK NOTICE
*  Copyright 2008-2026 Arnan de Gans. All Rights Reserved.
*  The software AdRotate, related services and ADROTATE the name are owned by Arnan de Gans.

*  COPYRIGHT NOTICES AND ALL THE COMMENTS SHOULD REMAIN INTACT.
*  By using this code you agree to indemnify Arnan de Gans from any
*  liability that might arise from its use.
------------------------------------------------------------------------------------ */

/*--- AdRotate values ---------------------------------------*/
define("ADROTATE_VERSION", 408);
define("ADROTATE_DB_VERSION", 76);
$adrotate_path = plugin_dir_path(__FILE__);
/*-----------------------------------------------------------*/

/*--- Load Files --------------------------------------------*/
require_once($adrotate_path.'/adrotate-functions.php');
require_once($adrotate_path.'/adrotate-statistics.php');
require_once($adrotate_path.'/adrotate-output.php');
require_once($adrotate_path.'/adrotate-widgets.php');
if(function_exists('register_block_type')) include_once($adrotate_path.'/adrotate-block.php');
/*-----------------------------------------------------------*/

/*--- Setup -------------------------------------------------*/
register_activation_hook(__FILE__, 'adrotate_activate');
register_deactivation_hook(__FILE__, 'adrotate_deactivate');
register_uninstall_hook(__FILE__, 'adrotatepro_uninstall');
/*-----------------------------------------------------------*/

/*--- Check and Load config ---------------------------------*/
load_plugin_textdomain('adrotate-pro', false, 'adrotate-pro/languages');
$adrotate_config = get_option('adrotate_config');
$adrotate_crawlers = get_option('adrotate_crawlers');
$adrotate_version = get_option('adrotate_version');
$adrotate_db_version = get_option('adrotate_db_version');
/*-----------------------------------------------------------*/

/*--- Core --------------------------------------------------*/
add_action('adrotate_notification', 'adrotate_notifications');
add_action('adrotate_empty_trash', 'adrotate_empty_trash');
add_action('adrotate_empty_trackerdata', 'adrotate_empty_trackerdata');
add_action('adrotate_auto_deletedisable', 'adrotate_auto_deletedisable');
add_action('init', 'adrotate_session', 1);
add_action('widgets_init', 'adrotate_widgets');
add_filter('adrotate_apply_photon', 'adrotate_apply_jetpack_photon');
/*-----------------------------------------------------------*/

/*--- Front end ---------------------------------------------*/
if(!is_admin()) {
	add_action('wp_head', 'adrotate_header');
	add_action('wp_enqueue_scripts', 'adrotate_scripts');
	add_shortcode('adrotate', 'adrotate_shortcode');
	if($adrotate_config['textwidget_shortcodes'] == 'Y') add_filter('widget_text', 'do_shortcode');

	// Post Injection
	add_filter('the_content', 'adrotate_inject_posts', 12);
	add_action('woocommerce_before_single_product', 'adrotate_inject_products', 15);
	add_action('woocommerce_after_single_product', 'adrotate_inject_products', 15);
	add_action('bbp_template_before_topics_loop', 'adrotate_inject_forums', 15);
	add_action('bbp_template_after_topics_loop', 'adrotate_inject_forums', 15);
	add_action('bbp_template_before_replies_loop', 'adrotate_inject_forums', 15);
	add_action('bbp_template_after_replies_loop', 'adrotate_inject_forums', 15);
}

// AJAX Callbacks
if(is_array($adrotate_config) AND $adrotate_config['stats'] == 1){
	add_action('wp_ajax_adrotate_impression', 'adrotate_impression_callback');
	add_action('wp_ajax_nopriv_adrotate_impression', 'adrotate_impression_callback');
	add_action('wp_ajax_adrotate_click', 'adrotate_click_callback');
	add_action('wp_ajax_nopriv_adrotate_click', 'adrotate_click_callback');
}
/*-----------------------------------------------------------*/

/*--- Back end ----------------------------------------------*/
if(is_admin()) {
	include_once($adrotate_path.'/adrotate-setup.php');
	include_once($adrotate_path.'/adrotate-admin-functions.php');
	include_once($adrotate_path.'/adrotate-admin-manage.php');
	include_once($adrotate_path.'/adrotate-admin-statistics.php');
	include_once($adrotate_path.'/adrotate-admin-portability.php');
	include_once($adrotate_path.'/adrotate-admin-advertiser.php');

	/*--- Dashboard hooks ---*/
	add_action('wpmu_new_blog', 'adrotate_activate_new_blog', 10, 1);
	add_action('admin_menu', 'adrotate_dashboard');
	add_action('admin_enqueue_scripts', 'adrotate_dashboard_scripts');
	add_action('admin_notices', 'adrotate_notifications_dashboard');
	add_filter('plugin_row_meta', 'adrotate_meta_links', 10, 2);
	add_filter('plugin_action_links_' . plugin_basename( __FILE__ ), 'adrotate_action_links');

	/*--- Publisher ---*/
	if(isset($_POST['adrotate_advert_submit'])) add_action('init', 'adrotate_insert_advert');
	if(isset($_POST['adrotate_group_submit'])) add_action('init', 'adrotate_insert_group');
	if(isset($_POST['adrotate_schedule_submit'])) add_action('init', 'adrotate_insert_schedule');
	if(isset($_POST['adrotate_generate_adcode'])) add_action('init', 'adrotate_generate_adcode');
	if(isset($_POST['adrotate_save_header_code'])) add_action('init', 'adrotate_save_header');
	if(isset($_POST['adrotate_upload_media'])) add_action('init', 'adrotate_insert_media');
	if(isset($_POST['adrotate_create_folder'])) add_action('init', 'adrotate_insert_folder');
	if(isset($_POST['adrotate_advertiser_submit'])) add_action('init', 'adrotate_insert_advertiser');
	if(isset($_POST['adrotate_action_submit'])) add_action('init', 'adrotate_request_action');
	if(isset($_POST['adrotate_save_options'])) add_action('init', 'adrotate_save_options');
	if(isset($_POST['adrotate_contact_support'])) add_action('init', 'adrotate_mail_advertiser');
	if(isset($_POST['adrotate_import'])) add_action('init', 'adrotate_import_ads');
	if(isset($_POST['adrotate_export'])) add_action('init', 'adrotate_export_stats');

	/*--- Advertiser ---*/
	if(is_array($adrotate_config) AND $adrotate_config['enable_advertisers'] == 'Y') {
		if(isset($_POST['adrotate_advertiser_ad_submit'])) add_action('init', 'adrotate_advertiser_insert_input');
		add_action('show_user_profile', 'adrotate_custom_profile_fields');
		add_action('edit_user_profile', 'adrotate_custom_profile_fields');
		add_action('user_new_form', 'adrotate_custom_profile_fields');
		add_action('user_register', 'adrotate_save_profile_fields'); // Update new profile
		add_action('edit_user_profile_update', 'adrotate_save_profile_fields'); // Update others profile
		add_action('personal_options_update', 'adrotate_save_profile_fields'); // Update own profile
	}

	/*--- Licensing ---*/
	include_once($adrotate_path.'/library/license-api.php');
	if(isset($_POST['adrotate_license_activate'])) add_action('init', 'adrotate_license_activate');
	if(isset($_POST['adrotate_license_deactivate'])) add_action('init', 'adrotate_license_deactivate');
	if(isset($_POST['adrotate_support_submit'])) add_action('init', 'adrotate_support_request');
}
/*-----------------------------------------------------------*/

/*-------------------------------------------------------------
 Name:      adrotate_dashboard
 Purpose:   Add pages to admin menus
-------------------------------------------------------------*/
function adrotate_dashboard() {
	global $adrotate_config;

	add_menu_page('AdRotate Pro', 'AdRotate Pro', 'adrotate_ad_manage', 'adrotate', 'adrotate_manage_adverts', plugins_url('/images/icon-menu.png', __FILE__), '25.8');
	add_submenu_page('adrotate', 'AdRotate Pro · '.esc_html__("Manage Adverts", 'adrotate-pro'), esc_html__("Manage Adverts", 'adrotate-pro'), 'adrotate_ad_manage', 'adrotate', 'adrotate_manage_adverts');
	add_submenu_page('adrotate', 'AdRotate Pro · '.esc_html__("Manage Groups", 'adrotate-pro'), esc_html__("Manage Groups", 'adrotate-pro'), 'adrotate_group_manage', 'adrotate-groups', 'adrotate_manage_group');
	add_submenu_page('adrotate', 'AdRotate Pro · '.esc_html__("Manage Schedules", 'adrotate-pro'), esc_html__("Manage Schedules", 'adrotate-pro'), 'adrotate_schedule_manage', 'adrotate-schedules', 'adrotate_manage_schedules');
	add_submenu_page('adrotate', 'AdRotate Pro · '.esc_html__("Manage Files", 'adrotate-pro'), esc_html__("Manage Files", 'adrotate-pro'), 'adrotate_ad_manage', 'adrotate-media', 'adrotate_manage_media');
	if($adrotate_config['enable_advertisers'] == 'Y') {
		add_submenu_page('adrotate', 'AdRotate Pro · '.esc_html__("Manage Advertisers", 'adrotate-pro'), esc_html__("Manage Advertisers", 'adrotate-pro'), 'adrotate_advertiser_manage', 'adrotate-advertisers', 'adrotate_manage_advertisers');
	}
	if($adrotate_config['stats'] == 1) {
		add_submenu_page('adrotate', 'AdRotate Pro · '.esc_html__("Statistics", 'adrotate-pro'), esc_html__("Statistics", 'adrotate-pro'), 'adrotate_global_report', 'adrotate-statistics', 'adrotate_statistics');
	}
	add_submenu_page('adrotate', 'AdRotate Pro · '.esc_html__("Premium Support", 'adrotate-pro'), esc_html__("Premium Support", 'adrotate-pro'), 'adrotate_ad_manage', 'adrotate-support', 'adrotate_support');
	add_submenu_page('adrotate', 'AdRotate Pro · '.esc_html__("Settings", 'adrotate-pro'), esc_html__("Settings", 'adrotate-pro'), 'manage_options', 'adrotate-settings', 'adrotate_options');

	if($adrotate_config['enable_advertisers'] == 'Y') {
		$current_user = wp_get_current_user();
		$is_advertiser = get_user_meta($current_user->ID, 'adrotate_is_advertiser', 1);
		if($is_advertiser == 'Y') {
			add_menu_page(esc_html__("Advertiser", 'adrotate-pro'), esc_html__("Advertiser", 'adrotate-pro'), 'adrotate_advertiser', 'adrotate-advertiser', 'adrotate_advertiser', plugins_url('/images/icon-menu.png', __FILE__), '25.9');
			add_submenu_page('adrotate-advertiser', 'AdRotate Pro · '.esc_html__("Advertiser", 'adrotate-pro'), esc_html__("Advertiser", 'adrotate-pro'), 'adrotate_advertiser', 'adrotate-advertiser', 'adrotate_advertiser');
		}
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_manage_adverts
 Purpose:   Admin management page
-------------------------------------------------------------*/
function adrotate_manage_adverts() {
	global $wpdb, $userdata, $blog_id, $adrotate_config;

	$view = (isset($_GET['view'])) ? esc_attr($_GET['view']) : '';
	$status = (isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$ad_edit_id =(isset($_GET['ad'])) ? esc_attr($_GET['ad']) : '';
	$file = (isset($_GET['file'])) ? esc_attr($_GET['file']) : '';

	if(!is_numeric($status)) $status = 0;
	if(!is_numeric($ad_edit_id)) $ad_edit_id = 0;

	$now = current_time('timestamp');
	$today = adrotate_date_start('day');
	$in2days = $now + 172800;
	$in7days = $now + 604800;
	$in84days = $now + 7257600;

	if(isset($_GET['month']) AND isset($_GET['year'])) {
		$month = esc_attr($_GET['month']);
		$year = esc_attr($_GET['year']);
	} else {
		$month = date('m');
		$year = date('Y');
	}
	$monthstart = mktime(0, 0, 0, $month, 1, $year);
	$monthend = mktime(0, 0, 0, $month+1, 0, $year);
	?>
	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Adverts", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status, array('file' => $file)); ?>

		<div class="tablenav">
			<div class="alignleft actions">
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate');?>"><?php esc_html_e("Manage", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate&view=generator');?>"><?php esc_html_e("Advert Generator", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate&view=addnew');?>"><?php esc_html_e("New Advert", 'adrotate-pro'); ?></a> |
				<?php if($adrotate_config['enable_advertisers'] == 'Y') { ?>
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate&view=queue');?>"><?php esc_html_e("Queue", 'adrotate-pro'); ?></a> |
				<?php } ?>
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate&view=archive');?>"><?php esc_html_e("Archive", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate&view=trash');?>"><?php esc_html_e("Trash", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate&view=advanced');?>"><?php esc_html_e("Header & ads.txt", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate&view=import');?>"><?php esc_html_e("Import Adverts", 'adrotate-pro'); ?></a>
			</div>
		</div>

    	<?php

    	if($view == '') {
			include('dashboard/publisher/manage-adverts.php');
	   	} else if($view == 'generator') {
			include('dashboard/publisher/manage-adverts-generator.php');
	   	} else if($view == 'addnew' OR $view == 'edit') {
			include('dashboard/publisher/manage-adverts-edit.php');
		} else if($view == 'queue') {
			include('dashboard/publisher/manage-adverts-queue.php');
		} else if($view == 'archive') {
			include('dashboard/publisher/manage-adverts-archive.php');
		} else if($view == 'trash') {
			include('dashboard/publisher/manage-adverts-trash.php');
	   	} else if($view == 'advanced') {
		   	$adrotate_gam = get_option('adrotate_gam_output');
		   	$adrotate_header = get_option('adrotate_header_output');

			if(file_exists(ABSPATH.$adrotate_config['adstxt_file'].'ads.txt')){
				$adstxt_content = file(ABSPATH.$adrotate_config['adstxt_file'].'ads.txt');
			} else {
				$adstxt_content = '';
			}

			include('dashboard/publisher/manage-adverts-advanced.php');
		} else if($view == 'import') {
			include('dashboard/publisher/manage-adverts-import.php');
		}
		?>
	</div>
<?php
}

/*-------------------------------------------------------------
 Name:      adrotate_manage_group
 Purpose:   Manage groups
-------------------------------------------------------------*/
function adrotate_manage_group() {
	global $wpdb, $adrotate_config;

	$view = (isset($_GET['view'])) ? esc_attr($_GET['view']) : '';
	$status = (isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$group_edit_id = (isset($_GET['group'])) ? esc_attr($_GET['group']) : '';

	if(!is_numeric($status)) $status = 0;
	if(!is_numeric($group_edit_id)) $group_edit_id = 0;

	if(isset($_GET['month']) AND isset($_GET['year'])) {
		$month = esc_attr($_GET['month']);
		$year = esc_attr($_GET['year']);
	} else {
		$month = date('m');
		$year = date('Y');
	}
	$monthstart = mktime(0, 0, 0, $month, 1, $year);
	$monthend = mktime(0, 0, 0, $month+1, 0, $year);

	$now = current_time('timestamp');
	$today = adrotate_date_start('day');
	$in2days = $now + 172800;
	$in7days = $now + 604800;
	?>
	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Groups", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status); ?>

		<div class="tablenav">
			<div class="alignleft actions">
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate-groups');?>"><?php esc_html_e("Manage", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate-groups&view=addnew');?>"><?php esc_html_e("Add New", 'adrotate-pro'); ?></a>
			</div>
		</div>

    	<?php
	    if ($view == '') {
			include('dashboard/publisher/manage-groups.php');
	   	} else if($view == 'addnew' OR $view == 'edit') {
			include('dashboard/publisher/manage-groups-edit.php');
	   	}
	   	?>
	</div>
<?php
}

/*-------------------------------------------------------------
 Name:      adrotate_manage_schedules
 Purpose:   Manage schedules for ads
-------------------------------------------------------------*/
function adrotate_manage_schedules() {
	global $wpdb, $adrotate_config;

	$view =(isset($_GET['view'])) ? esc_attr($_GET['view']) : '';
	$status =(isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$schedule_edit_id =(isset($_GET['schedule'])) ? esc_attr($_GET['schedule']) : '';

	if(!is_numeric($status)) $status = 0;
	if(!is_numeric($schedule_edit_id)) $schedule_edit_id = 0;

	$now = current_time('timestamp');
	$today = adrotate_date_start('day');
	$in2days = $now + 172800;
	$in7days = $now + 604800;
	$in84days = $now + 7257600;
	?>
	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Schedules", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status); ?>

		<div class="tablenav">
			<div class="alignleft actions">
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate-schedules');?>"><?php esc_html_e("Manage", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate-schedules&view=addnew');?>"><?php esc_html_e("Add New", 'adrotate-pro'); ?></a>
			</div>
		</div>

    	<?php
	    if ($view == '') {
			include('dashboard/publisher/manage-schedules.php');
		} else if($view == 'addnew' OR $view == 'edit') {
			include('dashboard/publisher/manage-schedules-edit.php');
		}
		?>
	</div>
<?php
}

/*-------------------------------------------------------------
 Name:      adrotate_statistics
 Purpose:   Advert and Group stats
-------------------------------------------------------------*/
function adrotate_statistics() {
	global $wpdb, $adrotate_config;

	$view = (isset($_GET['view'])) ? esc_attr($_GET['view']) : '';
	$status = (isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$id = (isset($_GET['id'])) ? esc_attr($_GET['id']) : '';
	$file =(isset($_GET['file'])) ? esc_attr($_GET['file']) : '';

	if(!is_numeric($status)) $status = 0;
	if(!is_numeric($id)) $id = 0;

	if(isset($_GET['graph_start']) AND isset($_GET['graph_end'])) {
		list($start_day, $start_month, $start_year) = explode('-', esc_attr($_GET['graph_start']));
		list($end_day, $end_month, $end_year) = explode('-', esc_attr( $_GET['graph_end']));

		$graph_start_date = gmmktime(0, 0, 0, $start_month, $start_day, $start_year);
		$graph_end_date = gmmktime(0, 0, 0, $end_month, $end_day, $end_year);
	} else {
		$graph_end_date = current_time('timestamp');
		$graph_start_date = $graph_end_date - (28 * DAY_IN_SECONDS);
	}
	$now = current_time('timestamp');
	$today = adrotate_date_start('day');
	$in2days = $now + 172800;
	$in7days = $now + 604800;
	?>

	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Statistics", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status, array('file' => $file)); ?>

		<?php
	    if ($view == '') {
			include('dashboard/publisher/statistics-main.php');
		} else if($view == 'advert') {
			include('dashboard/publisher/statistics-advert.php');
		} else if($view == 'group') {
			include('dashboard/publisher/statistics-group.php');
		} else if($view == 'advertiser') {
			include('dashboard/publisher/statistics-advertiser.php');
		}
		?>
	</div>
<?php
}

/*-------------------------------------------------------------
 Name:      adrotate_manage_advertisers
 Purpose:   Manage advertisers
-------------------------------------------------------------*/
function adrotate_manage_advertisers() {
	global $wpdb, $userdata, $adrotate_config;

	$view =(isset($_GET['view'])) ? esc_attr($_GET['view']) : '';
	$status =(isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$user_id =(isset($_GET['user'])) ? esc_attr($_GET['user']) : '';

	if(!is_numeric($status)) $status = 0;
	if(!is_numeric($user_id)) $user_id = 0;

	$now = current_time('timestamp');
	$today = adrotate_date_start('day');
	$in2days = $now + 172800;
	$in7days = $now + 604800;
	?>
	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Advertisers", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status); ?>

		<div class="tablenav">
			<div class="alignleft actions">
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate-advertisers');?>"><?php esc_html_e("Manage", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('users.php?adrotate');?>"><?php esc_html_e("All Users", 'adrotate-pro'); ?></a> |
				<a class="row-title" href="<?php echo admin_url('user-new.php?adrotate');?>"><?php esc_html_e("New User", 'adrotate-pro'); ?></a>
			</div>
		</div>

		<?php
		$all_advertisers = get_users(array('fields' => array('ID', 'display_name', 'user_email'), 'meta_key' => 'adrotate_is_advertiser', 'meta_value' => 'Y', 'orderby' => 'ID', 'order' => 'ASC'));

		$advertisers = array();
		foreach($all_advertisers as $advertiser) {
			$has_adverts = $wpdb->get_var("SELECT COUNT(`{$wpdb->prefix}adrotate_linkmeta`.`id`) FROM `{$wpdb->prefix}adrotate`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `user` = '{$advertiser->ID}' AND `ad` = `{$wpdb->prefix}adrotate`.`id` AND `type` != 'empty' AND `type` != 'a_empty';");

			$advertisers[$advertiser->ID] = array(
				'name' => $advertiser->display_name,
				'email' => $advertiser->user_email,
				'has_adverts' => $has_adverts,
			);
			unset($advertiser);
		}

    	if ($view == '' OR $view == 'manage') {
			include('dashboard/publisher/advertisers-main.php');
		} else if($view == 'profile') {
			include('dashboard/publisher/advertisers-profile.php');
		} else if($view == 'contact') {
			include('dashboard/publisher/advertisers-contact.php');
		}
		?>
	</div>
<?php
}

/*-------------------------------------------------------------
 Name:      adrotate_manage_media
 Purpose:   Manage banner images for ads
-------------------------------------------------------------*/
function adrotate_manage_media() {
	global $wpdb, $adrotate_config;

	$status = (isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$file = (isset($_GET['file'])) ? esc_attr($_GET['file']) : '';
	$path = (isset($_GET['path'])) ? esc_attr($_GET['path']) : '';
	$error = (isset($_GET['error'])) ? esc_attr($_GET['error']) : '';

	if(!is_numeric($status)) $status = 0;

	if(strlen($file) > 0 AND wp_verify_nonce($_REQUEST['_wpnonce'], 'adrotate_delete_media_'.$file)) {
		if(adrotate_unlink($file, $path)) {
			$status = 206;
		} else {
			$status = 517;
		}
	}
	?>

	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Files and Reports", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status, array('error' => $error)); ?>

		<?php
		include('dashboard/publisher/media.php');
		?>
	</div>
<?php
}

/*-------------------------------------------------------------
 Name:      adrotate_support
 Purpose:   Get help
-------------------------------------------------------------*/
function adrotate_support() {
	global $wpdb, $adrotate_config;

	$status = (isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$file = (isset($_GET['file'])) ? esc_attr($_GET['file']) : '';

	if(!is_numeric($status)) $status = 0;

	$current_user = wp_get_current_user();
	$a = get_option('ajdg_activate_adrotate-pro');
	?>

	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Premium Support", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status); ?>

		<?php
		include('dashboard/support.php');
		?>

	</div>
<?php
}

/*-------------------------------------------------------------
 Name:      adrotate_options
 Purpose:   Admin options page
-------------------------------------------------------------*/
function adrotate_options() {
	global $wpdb, $wp_roles;

    $active_tab = (isset($_GET['tab'])) ? esc_attr($_GET['tab']) : 'general';
	$status = (isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$action = (isset($_GET['action'])) ? esc_attr($_GET['action']) : '';
	$error = (isset($_GET['error'])) ? esc_attr($_GET['error']) : '';

	if(!is_numeric($status)) $status = 0;

	if(isset($_GET['adrotate-nonce']) AND wp_verify_nonce($_GET['adrotate-nonce'], 'maintenance')) {
		if($action == 'check-all-ads') {
			adrotate_evaluate_ads();
			$status = 405;
		}

		if($action == 'update-db') {
			adrotate_finish_upgrade();
			$status = 402;
		}

		if($action == 'create-filefolders') {
			adrotate_maintenance_create_folders();
			$status = 409;
		}

		if($action == 'reset-tasks') {
			adrotate_check_schedules();
			$status = 401;
		}

		if($action == 'reset-geo') {
			set_transient('adrotate_geo_reset', 1, 1);
			update_option('adrotate_geo_requests', 5);
			adrotate_geolocation();
			$status = 403;
		}

		if($action == 'test-mail') {
			adrotate_notifications('test');
			$status = 407;
		}
	}
	?>

	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Settings", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status, array('error' => $error)); ?>

		<h2 class="nav-tab-wrapper">
            <a href="?page=adrotate-settings&tab=general" class="nav-tab <?php echo $active_tab == 'general' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e("General", 'adrotate-pro'); ?></a>
            <a href="?page=adrotate-settings&tab=notifications" class="nav-tab <?php echo $active_tab == 'notifications' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e("Notifications", 'adrotate-pro'); ?></a>
            <a href="?page=adrotate-settings&tab=stats" class="nav-tab <?php echo $active_tab == 'stats' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e("Statistics", 'adrotate-pro'); ?></a>
            <a href="?page=adrotate-settings&tab=geo" class="nav-tab <?php echo $active_tab == 'geo' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e("Geo Targeting", 'adrotate-pro'); ?></a>
            <a href="?page=adrotate-settings&tab=roles" class="nav-tab <?php echo $active_tab == 'roles' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e("Access Roles", 'adrotate-pro'); ?></a>
            <a href="?page=adrotate-settings&tab=misc" class="nav-tab <?php echo $active_tab == 'misc' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e("Miscellaneous", 'adrotate-pro'); ?></a>
            <a href="?page=adrotate-settings&tab=maintenance" class="nav-tab <?php echo $active_tab == 'maintenance' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e("Maintenance", 'adrotate-pro'); ?></a>
            <a href="?page=adrotate-settings&tab=license" class="nav-tab <?php echo $active_tab == 'license' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e("License", 'adrotate-pro'); ?></a>
        </h2>

		<?php
		$adrotate_config = get_option('adrotate_config');
		$a = get_option('ajdg_activate_adrotate-pro');

		if($active_tab == 'general') {
			$adrotate_crawlers = get_option('adrotate_crawlers');

			$crawlers = '';
			if(is_array($adrotate_crawlers)) {
				$crawlers = implode(', ', $adrotate_crawlers);
			}

			include('dashboard/settings/general.php');
		} elseif($active_tab == 'notifications') {
			$adrotate_notifications	= get_option('adrotate_notifications');

			$notification_mails = $advertiser_mails = '';
			if(is_array($adrotate_notifications['notification_email_publisher'])) {
				$notification_mails	= implode(', ', $adrotate_notifications['notification_email_publisher']);
			}

			include('dashboard/settings/notifications.php');
		} elseif($active_tab == 'stats') {
			include('dashboard/settings/statistics.php');
		} elseif($active_tab == 'geo') {
			$adrotate_geo_requests = get_option('adrotate_geo_requests', 0);

			include('dashboard/settings/geotargeting.php');
		} elseif($active_tab == 'roles') {
			include('dashboard/settings/roles.php');
		} elseif($active_tab == 'misc') {
			include('dashboard/settings/misc.php');
		} elseif($active_tab == 'maintenance') {
			$adrotate_notifications	= get_option('adrotate_notifications');

			$adrotate_version = get_option('adrotate_version');
			$adrotate_db_version = get_option('adrotate_db_version');
			$advert_status	= get_option("adrotate_advert_status");

			$adschedule = wp_next_scheduled('adrotate_notification');
			$trash = wp_next_scheduled('adrotate_empty_trash');
			$tracker = wp_next_scheduled('adrotate_empty_trackerdata');
			$autodeletedisable = wp_next_scheduled('adrotate_auto_deletedisable');

			include('dashboard/settings/maintenance.php');
		} elseif($active_tab == 'license') {
			include('dashboard/settings/license.php');
		}
		?>
	</div>
<?php
}

/*-------------------------------------------------------------
 Name:      adrotate_advertiser
 Purpose:   Advertiser page
-------------------------------------------------------------*/
function adrotate_advertiser() {
	global $wpdb, $adrotate_config;

	$view = (isset($_GET['view'])) ? esc_attr($_GET['view']) : '';
	$status = (isset($_GET['status'])) ? esc_attr($_GET['status']) : '';
	$ad_edit_id = (isset($_GET['ad'])) ? esc_attr($_GET['ad']) : '';
	$filename = (isset($_GET['file'])) ? esc_attr($_GET['file']) : '';

	if(!is_numeric($status)) $status = 0;
	if(!is_numeric($ad_edit_id)) $ad_edit_id = 0;

	$current_user = wp_get_current_user();
	$advertiser_permissions = get_user_meta($current_user->ID, 'adrotate_permissions', 1);
	if(!isset($advertiser_permissions['create'])) $advertiser_permissions['create'] = 'N';
	if(!isset($advertiser_permissions['edit'])) $advertiser_permissions['edit'] = 'N';
	if(!isset($advertiser_permissions['advanced'])) $advertiser_permissions['advanced'] = 'N';
	if(!isset($advertiser_permissions['geo'])) $advertiser_permissions['geo'] = 'N';
	if(!isset($advertiser_permissions['group'])) $advertiser_permissions['group'] = 'N';
	if(!isset($advertiser_permissions['schedules'])) $advertiser_permissions['schedules'] = 'N';

	if(isset($_GET['graph_start']) AND isset($_GET['graph_end'])) {
		list($start_day, $start_month, $start_year) = explode('-', esc_attr($_GET['graph_start']));
		list($end_day, $end_month, $end_year) = explode('-', esc_attr( $_GET['graph_end']));

		$graph_start_date = gmmktime(0, 0, 0, $start_month, $start_day, $start_year);
		$graph_end_date = gmmktime(0, 0, 0, $end_month, $end_day, $end_year);
	} else {
		$graph_end_date = current_time('timestamp');
		$graph_start_date = $graph_end_date - (28 * DAY_IN_SECONDS);
	}

	$now = current_time('timestamp');
	$today = adrotate_date_start('day');
	$in2days = $now + 172800;
	$in7days = $now + 604800;
	$in84days = $now + 7257600;
	?>
	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e("Advertiser", 'adrotate-pro'); ?></h1>
		<hr class="wp-header-end">

		<?php if($status > 0) adrotate_status($status, array('file' => $filename)); ?>

		<div class="tablenav">
			<div class="alignleft actions">
				<a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate-advertiser');?>"><?php esc_html_e("Manage", 'adrotate-pro'); ?></a>
				<?php if($advertiser_permissions['create'] == 'Y') { ?>
				 | <a class="row-title" href="<?php echo admin_url('/admin.php?page=adrotate-advertiser&view=addnew');?>"><?php esc_html_e("Add New", 'adrotate-pro'); ?></a>
				<?php  } ?>
			</div>
		</div>

		<?php
		if($view == '') {

			$ads = $wpdb->get_results($wpdb->prepare("SELECT `ad` FROM `{$wpdb->prefix}adrotate_linkmeta` WHERE `group` = 0 AND `user` = '%d' ORDER BY `ad` ASC;", $current_user->ID));

			if($ads) {
				$activebanners = $queuebanners = $disabledbanners = array();
				foreach($ads as $ad) {
					$banner = $wpdb->get_row("SELECT `id`, `title`, `type`, `desktop`, `mobile`, `tablet`, `budget`, `crate`, `irate` FROM `{$wpdb->prefix}adrotate` WHERE (`type` = 'active' OR `type` = '2days' OR `type` = '7days' OR `type` = 'disabled' OR `type` = 'error' OR `type` = 'a_error' OR `type` = 'expired' OR `type` = 'queue' OR `type` = 'reject') AND `id` = {$ad->ad};");

					// Skip if no ad
					if(!$banner) continue;

					$starttime = $stoptime = 0;
					$starttime = $wpdb->get_var("SELECT `starttime` FROM `{$wpdb->prefix}adrotate_schedule`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = {$banner->id} AND `schedule` = `{$wpdb->prefix}adrotate_schedule`.`id` ORDER BY `starttime` ASC LIMIT 1;");
					$stoptime = $wpdb->get_var("SELECT `stoptime` FROM `{$wpdb->prefix}adrotate_schedule`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `ad` = {$banner->id} AND `schedule` = `{$wpdb->prefix}adrotate_schedule`.`id` ORDER BY `stoptime` DESC LIMIT 1;");

					$type = $banner->type;
					if($type == 'active' AND $stoptime <= $now) $type = 'expired';
					if($type == 'active' AND $stoptime <= $in2days) $type = '2days';
					if($type == 'active' AND $stoptime <= $in7days) $type = '7days';

					if($type == 'active' OR $type == '2days' OR $type == '7days' OR $type == 'expired') {
						$activebanners[$banner->id] = array(
							'id' => $banner->id,
							'title' => $banner->title,
							'type' => $type,
							'desktop' => $banner->desktop,
							'mobile' => $banner->mobile,
							'tablet' => $banner->tablet,
							'firstactive' => $starttime,
							'lastactive' => $stoptime,
							'budget' => $banner->budget,
							'crate' => $banner->crate,
							'irate' => $banner->irate
						);
					}

					if($type == 'disabled') {
						$disabledbanners[$banner->id] = array(
							'id' => $banner->id,
							'title' => $banner->title,
							'type' => $type
						);
					}

					if($type == 'queue' OR $type == 'reject' OR $type == 'error' OR $type == 'a_error') {
						$queuebanners[$banner->id] = array(
							'id' => $banner->id,
							'title' => $banner->title,
							'type' => $type,
							'desktop' => $banner->desktop,
							'mobile' => $banner->mobile,
							'tablet' => $banner->tablet,
							'budget' => $banner->budget,
							'crate' => $banner->crate,
							'irate' => $banner->irate
						);
					}
				}

				// Show active ads, if any
				if(count($activebanners) > 0) {
					include('dashboard/advertiser/main.php');
				}

				// Show disabled ads, if any
				if(count($disabledbanners) > 0) {
					include('dashboard/advertiser/main-disabled.php');
				}

				// Show queued ads, if any
				if(count($queuebanners) > 0) {
					include('dashboard/advertiser/main-queue.php');
				}

				if($adrotate_config['stats'] == 1) {
					// Gather data for summary report
					$stats = adrotate_prepare_advertiser_report($current_user->ID, $activebanners);
					$stats_graph_month = $wpdb->get_row($wpdb->prepare("SELECT SUM(`clicks`) as `clicks`, SUM(`impressions`) as `impressions` FROM `{$wpdb->prefix}adrotate_stats`, `{$wpdb->prefix}adrotate_linkmeta` WHERE `{$wpdb->prefix}adrotate_stats`.`ad` = `{$wpdb->prefix}adrotate_linkmeta`.`ad` AND `{$wpdb->prefix}adrotate_linkmeta`.`user` = %d AND `{$wpdb->prefix}adrotate_stats`.`thetime` >= %d AND `{$wpdb->prefix}adrotate_stats`.`thetime` <= %d;", $current_user->ID, $graph_start_date, $graph_end_date), ARRAY_A);

					// Prevent gaps in display
					if(empty($stats['ad_amount'])) $stats['ad_amount'] = 0;
					if(empty($stats['thebest'])) $stats['thebest'] = 0;
					if(empty($stats['theworst'])) $stats['theworst'] = 0;
					if(empty($stats['total_impressions'])) $stats['total_impressions'] = 0;
					if(empty($stats['total_clicks'])) $stats['total_clicks'] = 0;
					if(empty($stats_graph_month['impressions'])) $stats_graph_month['impressions'] = 0;
					if(empty($stats_graph_month['clicks'])) $stats_graph_month['clicks'] = 0;

					// Get Click Through Rate
					$ctr = adrotate_ctr($stats['total_clicks'], $stats['total_impressions']);
					$ctr_graph_month = adrotate_ctr($stats_graph_month['clicks'], $stats_graph_month['impressions']);

					include('dashboard/advertiser/main-summary.php');
				}

			} else {
				?>
				<table class="widefat" style="margin-top:.5em;">
					<thead>
						<tr>
							<th><?php esc_html_e("Notice", 'adrotate-pro'); ?></th>
						</tr>
					</thead>
					<tbody>
					    <tr>
							<td><?php esc_html_e("No ads for user.", 'adrotate-pro'); ?></td>
						</tr>
					</tbody>
				</table>
				<?php
			}
		} else if($view == 'addnew' OR $view == 'edit') {

			include('dashboard/advertiser/edit.php');

		} else if($view == 'report') {

			include('dashboard/advertiser/report.php');

		} else if($view == 'message') {

			if(wp_verify_nonce($_REQUEST['_wpnonce'], 'adrotate_email_advertiser')) {
				include('dashboard/advertiser/message.php');
			} else {
				adrotate_nonce_error();
				exit;
			}

		}
		?>
		<br class="clear" />

		<?php adrotate_user_notice(); ?>

	</div>
<?php
}
?>
