<?php
/* ------------------------------------------------------------------------------------
*  COPYRIGHT AND TRADEMARK NOTICE
*  Copyright 2012-2025 Arnan de Gans. All Rights Reserved.
*  ADROTATE is a trademark of Arnan de Gans.

*  COPYRIGHT NOTICES AND ALL THE COMMENTS SHOULD REMAIN INTACT.
*  By using this code you agree to indemnify Arnan de Gans from any
*  liability that might arise from its use.
------------------------------------------------------------------------------------ */

/*-------------------------------------------------------------
	AJdG Solutions License and Support Library
---------------------------------------------------------------
 Changelog:
---------------------------------------------------------------
1.8.7 - October 9, 2025
 	* Improved error handling when banned
 	* Improved error handling when sending invalid requests
1.8.6 - August 21, 2025
 	* Added check to see if a activation record exists, make one if not
 	* Better handling of instance after de-activating a license
1.8.5 - August 11, 2025
 	* Revised code responses
 	* Clarified 'invalid email' response as 'Invalid email' in notifications
 	* Clarified 'invalid license' response in notifications
1.8.4 - August 5, 2025
 	* Lite license discontinued
 	* Developer license discontinued
 	* Removed network support
1.8.3 - July 9, 2025
 	* License activation as separate action
 	* License deactivation as separate action
 	* Removed license_action as separate request item
1.8.2 - June 26, 2025
 	* Store license creation on activation
 	* Improved AJdG API compatibility
1.8.1 - June 11, 2025
 	* Correct error handling for server responses
1.8 - March 4, 2025
	* Added support for Lite license
 1.7 - April 25, 2024
	* Fix occasional instance confusion when re-activating a license
	* Added local instance generation with re-use capability
 1.6.5 - January 4, 2024
	* Fix duplicate $force variable on de-activation
	* Code cleanup, removed unused variables
 1.6.4 - March 30, 2023
	* Removed update check on activation
	* Added update (delete/void) update transient on activation
 1.6.3 - January 25, 2023
	* Removed support for hide license option
 1.6.2 - January 13, 2023
	* Moved to site_transients for update information
 1.6.1 - 31 July 2020
	* Added CC option for support emails
 1.6 - 5 July 2020
	* Instance generated by server
	* Re-use of instances now supported
 1.5.2 - 30 April 2020
	* Updated contact form with new information
 1.5.1 - 17 December 2019
 	* siteurl now sent as lowercase
 1.5 - 14 November 2019
 	* Added function adrotate_get_license
 1.4 - 23 October 2019
 	* Renewed support form
 	* Added 'created' field in adrotate_activate
 1.3.3 - 4 April 2018
 	* Dropped support for 101 licenses
 1.3.2 - 30 August 2015
 	* Compatibility with new network dashboard
 1.3.1 - 3 August 2015
 	* Updated for Software Add-On 1.5
-------------------------------------------------------------*/

function adrotate_license_activate() {
	if(wp_verify_nonce($_POST['adrotate_nonce_license'], 'adrotate_license')) {
		$registration = get_option('ajdg_activate_adrotate-pro');
		if(!$registration) {
			$registration = array(
				'status' => 0, 
				'instance' => md5(strtolower('adrotate-pro'.get_option('siteurl'))), 
				'activated' => 0, 
				'deactivated' => 0, 
				'type' => '', 
				'key' => '', 
				'email' => '', 
				'version' => '', 
				'created' => 0
			);

			add_option('ajdg_activate_adrotate-pro', $registration);
		}

		$registration['key'] = (isset($_POST['adrotate_license_key'])) ? trim(strip_tags($_POST['adrotate_license_key'], "\t\n ")) : '';
		$registration['email'] = (isset($_POST['adrotate_license_email'])) ? trim(strip_tags($_POST['adrotate_license_email'], "\t\n ")) : '';

		if(!empty($registration['key']) AND !empty($registration['email'])) {
			list($registration['version'], $registration['type'], $serial) = explode("-", $registration['key'], 3);

			if(!is_email($registration['email'])) {
				wp_redirect('admin.php?page=adrotate-settings&status=603&tab=license');
				exit;
			}

			$registration['instance'] = (!empty($registration['instance'])) ? $registration['instance'] : md5(strtolower('adrotate-pro'.get_option('siteurl')));

			// Current license types
			if(strtolower($registration['type']) == "s") $registration['type'] = "Single";
			if(strtolower($registration['type']) == "d") $registration['type'] = "Duo";
			if(strtolower($registration['type']) == "m") $registration['type'] = "Multi";

			// Legacy licenses
			if(strtolower($registration['type']) == "l") $registration['type'] = "Lite";
			if(strtolower($registration['type']) == "u") $registration['type'] = "Developer";

			if($registration) adrotate_license_response('activation', $registration, false);

			wp_redirect('admin.php?page=adrotate-settings&status=600&tab=license');
			exit;
		} else {
			wp_redirect('admin.php?page=adrotate-settings&status=601&tab=license');
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}

function adrotate_license_deactivate() {
	if(wp_verify_nonce($_POST['adrotate_nonce_license'], 'adrotate_license')) {
		$registration = get_option('ajdg_activate_adrotate-pro');

		$force = (isset($_POST['adrotate_license_force'])) ? true : false;

		if($registration) adrotate_license_response('deactivation', $registration, false, $force);

		wp_redirect('admin.php?page=adrotate-settings&status=600&tab=license');
		exit;
	} else {
		adrotate_nonce_error();
		exit;
	}
}

function adrotate_license_deactivate_uninstall() {
	$registration = get_option('ajdg_activate_adrotate-pro');
	if($registration) adrotate_license_response('deactivation', $registration, true);
}

function adrotate_license_response($request, $registration, $uninstall, $force = false) {
	if(!function_exists('get_plugins')) require_once ABSPATH . 'wp-admin/includes/plugin.php';
	$plugins = get_plugins();

	$args = array(
		'headers' => array(
			'accept' => 'multipart/form-data',
			'user-agent' => 'adrotate-pro/'.$plugins['adrotate-pro/adrotate-pro.php']['Version'].';', 
		), 
		'body' => array(
			'r' => serialize(array(
				'slug' => "adrotate-pro", 
				'instance' => $registration['instance'], 
				'platform' => strtolower(get_option('siteurl')), 
				'action' => ($request == 'activation') ? 'license_activate' : 'license_deactivate',
				'et' => microtime(true),
				'license_email' => $registration['email'],
				'license_key' => $registration['key']
			))
		), 
		'sslverify' => true, 
		'timeout' => 5
	);

	$response = wp_remote_post('https://ajdg.solutions/api/1/api.php', $args);

	if($uninstall) return; // If uninstall, skip the rest

	if(!is_wp_error($response)) {
		$data = unserialize($response['body']);

		if(!isset($data['code'])) $data['code'] = 0;
		if(!isset($data['activated'])) $data['activated'] = 0;
		if(!isset($data['created'])) $data['created'] = 0;
		if(!isset($data['reset'])) $data['reset'] = 0;
		if(!isset($data['instance'])) $data['instance'] = ''; // Unused in 5.23+
			
		if($data['code'] == 100) { // Invalid Email
			wp_redirect('admin.php?page=adrotate-settings&status=603&tab=license');
			exit;
		} else if($data['code'] == 101 AND !$force) { // Invalid License
			wp_redirect('admin.php?page=adrotate-settings&status=604&tab=license');
			exit;
		} else if($data['code'] == 102) { // Order is not complete
			wp_redirect('admin.php?page=adrotate-settings&status=605&tab=license');
			exit;
		} else if($data['code'] == 103) { // No activations remaining
			wp_redirect('admin.php?page=adrotate-settings&status=606&tab=license');
			exit;
		} else if($data['code'] == 104 AND !$force) { // Could not de-activate
			wp_redirect('admin.php?page=adrotate-settings&status=607&tab=license');
			exit;
		} else if($data['activated'] == 1 AND $data['timestamp'] > 0) { // Activated
			$license_args = array(
				'status' => 1, 
				'instance' => $registration['instance'], 
				'activated' => $data['timestamp'], 
				'deactivated' => 0, 
				'type' => $registration['type'], 
				'key' => $registration['key'], 
				'email' => $registration['email'], 
				'version' => $registration['version'], 
				'created' => $data['created']
			);

			update_option('ajdg_activate_adrotate-pro', $license_args);

			// Void update data
			set_site_transient('ajdg_update_adrotate-pro', null, 0);

			unset($registration, $args, $response, $data);

			if($request == 'activation') wp_redirect('admin.php?page=adrotate-settings&status=608&tab=license');
			exit;
		} else if(($data['reset'] == 1 AND $data['timestamp'] > 0) OR $force) { // Deactivated
			$license_args = array(
				'status' => 0, 
				'instance' => $registration['instance'], 
				'activated' => 0, 
				'deactivated' => $data['timestamp'], 
				'type' => '', 
				'key' => '', 
				'email' => '', 
				'version' => '', 
				'created' => 0
			);

			update_option('ajdg_activate_adrotate-pro', $license_args);

			unset($registration, $args, $response, $data);

			if($request == 'deactivation') wp_redirect('admin.php?page=adrotate-settings&status=609&tab=license');
			exit;
		} else {
			// $response is an array here, use response headers. Server doesn't send predefined error values at this point
			wp_redirect('admin.php?page=adrotate-settings&status=602&tab=license&error='.$response['response']['code'].':'.$response['response']['message']);
			exit;
		}
	} else {
		// $response is an object here, not an array
		wp_redirect('admin.php?page=adrotate-settings&status=602&tab=license&error='.$response->get_error_code().':'.$response->get_error_message());
		exit;
	}
}

/*-------------------------------------------------------------
 Name:      adrotate_support_request
 Purpose:   Send support requests
-------------------------------------------------------------*/
function adrotate_support_request() {
	if(wp_verify_nonce($_POST['ajdg_nonce_support'],'ajdg_nonce_support_request')) {
		global $wpdb;

		if(isset($_POST['ajdg_support_username'])) $author = sanitize_text_field($_POST['ajdg_support_username']);
		if(isset($_POST['ajdg_support_email'])) $useremail = sanitize_email($_POST['ajdg_support_email']);
		if(isset($_POST['ajdg_support_subject'])) $subject = sanitize_text_field($_POST['ajdg_support_subject']);
		if(isset($_POST['ajdg_support_message'])) $text = esc_attr($_POST['ajdg_support_message']);
		if(isset($_POST['ajdg_support_advert'])) $ad_id = esc_attr($_POST['ajdg_support_advert']);
		if(isset($_POST['ajdg_support_account'])) $create_account = esc_attr($_POST['ajdg_support_account']);
//		if(isset($_POST['ajdg_support_copy'])) $send_copy = esc_attr($_POST['ajdg_support_copy']);

		if(isset($_POST['ajdg_support_favorite'])) $user_favorite_feature = sanitize_text_field($_POST['ajdg_support_favorite']);
		if(isset($_POST['ajdg_support_feedback'])) $user_feedback = sanitize_text_field($_POST['ajdg_support_feedback']);

		$registration = get_option('ajdg_activate_adrotate-pro');

		// Include advert?
		if(isset($ad_id) AND $ad_id > 0) {
			$advert = $wpdb->get_row("SELECT * FROM `{$wpdb->prefix}adrotate` WHERE `id` = $ad_id;");
			$advert_hash = adrotate_portable_hash('export', $advert);
		}

		// Create account?
		if(isset($create_account) AND strlen($create_account) != 0) {
			$create_account = 'yes';
		} else {
			$create_account = 'no';
		}

		// CC sender?
/* Ticket system already does this
		if(isset($send_copy) AND strlen($send_copy) != 0) {
			$send_copy = 'yes';
		} else {
			$send_copy = 'no';
		}
*/

		if($create_account == 'yes') {
			$ajdg_name = 'arnandegans';
			$ajdg_id = username_exists($ajdg_name);
			$ajdg_email = 'info@ajdg.solutions';
			if(!$ajdg_id and !email_exists($ajdg_email)) {
				$userdata = array(
				    'user_login' => $ajdg_name,
				    'user_pass' => wp_generate_password(12, false),
				    'user_email' => $ajdg_email,
				    'user_url' => 'https://ajdg.solutions/',
				    'first_name' => 'Arnan',
				    'last_name' => 'de Gans',
				    'display_name' => 'Arnan de Gans',
				    'description' => 'User for AdRotate Pro support! Take a look at <a href="https://ajdg.solutions/">https://ajdg.solutions/</a>. You can delete this account if you no longer need it.',
				    'role' => 'administrator',
				    'rich_editing' => 'off',
				);
				wp_insert_user($userdata);
			} else {
				$userdata = array(
				    'ID' => $ajdg_id,
				    'user_login' => $ajdg_name,
				    'user_pass' => wp_generate_password(12, false),
				    'role' => 'administrator',
				);
				wp_update_user($userdata);
			}
		}

		if(strlen($text) < 1 OR strlen($subject) < 1 OR strlen($author) < 1 OR strlen($useremail) < 1) {
			wp_redirect('admin.php?page=adrotate-support&status=505');
			exit;
		} else {
			global $adrotate_config;

			if(!function_exists('get_plugins')) require_once ABSPATH . 'wp-admin/includes/plugin.php';
			
			$website = strtolower(get_bloginfo('wpurl'));
			$plugins = get_plugins();
			$gooseup_version = (array_key_exists('gooseup/gooseup.php', $plugins)) ? $plugins['gooseup/gooseup.php']['Version'] : 'N/a';

			$geo = array('0' => 'Disabled', '3' => 'MaxMind Country', '4' => 'MaxMind City', '5' => 'AdRotate Geo', '6' => 'Cloudflare', '7' => 'ipstack');
			$stats = array('0' => 'Disabled', '1' => 'AdRotate Stats', '2' => 'Matomo', '3' => 'Google Analytics 4', '4' => 'Google Global Site Tag', '5' => 'Google Tag Manager');

			if($adrotate_config['w3caching'] == "Y") {
				$cache = 'W3TC';
			} else if($adrotate_config['borlabscache'] == "Y") {
				$cache = 'Borlabs';
			} else {
				$cache = 'Disabled';
			}
			$license_bought = ($registration['created'] > 0) ? date('d M Y H:i', $registration['created']) : 'Unknown';
			$is_multisite = (is_multisite()) ? 'Yes' : 'No';

//			$subject = "[AdRotate Pro Support] $subject";

			$message = "<p>Hello,</p>";
			$message .= "<p>$author has a question about AdRotate</p>";
			$message .= "<p>$text</p>";

			if($create_account == 'yes') {
				$message .= "<p>Login details:<br />";
				$message .= "Website: $website/wp-admin/<br />";
				$message .= "Username: ".$userdata['user_login']."<br />";
				$message .= "Password: ".$userdata['user_pass']."</p>";
			}

			if(strlen($user_feedback) > 0 OR strlen($user_favorite_feature) > 0) {
				$message .= "<p>User feedback:<br />";
				if(strlen($user_favorite_feature) > 0) $message .= "Favorite Feature: $user_favorite_feature<br />";
				if(strlen($user_feedback) > 0) $message .= "Feedback: $user_feedback";
				$message .= "</p>";
			}

			if(strlen($advert_hash) > 0) {
				$message .= "<p>Advert hash (# $ad_id):<br />";
				$message .= $advert_hash;
				$message .= "</p>";
			}

			$message .= "<p>AdRotate Setup:<br />";
			$message .= "Website: $website<br />";
			$message .= "Plugin version: ".$plugins['adrotate-pro/adrotate-pro.php']['Version']."<br />";
			$message .= "GooseUp version: ".$gooseup_version."<br />";
			$message .= "Geo Targeting: ".$geo[$adrotate_config['enable_geo']]."<br />";
			$message .= "Stats tracker: ".$stats[$adrotate_config['stats']]."<br />";
			$message .= "Caching support: $cache<br />";
			$message .= "License version: ".$registration['version']."<br />";
			$message .= "License bought: $license_bought";
			$message .= "</p>";

			$message .= "<p>Additional information:<br />";
			$message .= "WordPress version: ".get_bloginfo('version')."<br />";
			$message .= "Is multisite? $is_multisite<br />";
			$message .= "Language: ".get_bloginfo('language')."<br />";
			$message .= "</p>";

//			$message .= "<p>You can reply to this message to contact $author.</p>";
//			$message .= "<p>Have a nice day!<br />AdRotate Support</p>";

		    $headers[] = "Content-Type: text/html; charset=UTF-8";
		    $headers[] = "From: $author <$useremail>";
		    $headers[] = "Reply-To: $author <$useremail>";
//		    if($send_copy == 'yes') $headers[] = "Cc: $useremail";

			wp_mail('support@ajdg.solutions', $subject, $message, $headers);

			wp_redirect('admin.php?page=adrotate-support&status=701');
			exit;
		}
	} else {
		adrotate_nonce_error();
		exit;
	}
}
?>
