<?php

namespace Directorist_Booking\App\Http\Controllers;

use WP_REST_Response;

class BookingController extends Controller {
    /**
     * Get confirmed bookings for current user's listings
     *
     * 
     */
    public function get_confirmed_bookings() {
        global $wpdb;

        $current_user_id = get_current_user_id();

        if ( ! $current_user_id ) {
            return [
                'status_code' => 401,
                'data'        => [
                    'success' => false,
                    'message' => __( 'User not logged in.', 'directorist-booking' ),
                ],
            ];
        }

        $bookings_table = $wpdb->prefix . 'directorist_booking';
        $posts_table    = $wpdb->prefix . 'posts';
        $postmeta_table = $wpdb->prefix . 'postmeta';
        $users_table    = $wpdb->prefix . 'users';

        $query = $wpdb->prepare(
            "SELECT 
                b.ID as booking_id,
                b.bookings_author,
                b.listing_id,
                b.date_start,
                b.date_end,
                b.comment,
                b.price,
                p.post_title as listing_title,
                p.post_author as listing_owner_id,
                pm.meta_value as booking_type_meta,
                u.display_name as booking_author_name
            FROM {$bookings_table} b
            INNER JOIN {$posts_table} p ON b.listing_id = p.ID
            LEFT JOIN {$postmeta_table} pm ON b.listing_id = pm.post_id AND pm.meta_key = '_booking_type'
            LEFT JOIN {$users_table} u ON b.bookings_author = u.ID
            WHERE b.status IN ('confirmed', 'paid')
            AND p.post_author = %d
            ORDER BY b.created DESC",
            $current_user_id
        );

        $results = $wpdb->get_results( $query );

        if ( $wpdb->last_error ) {
            return [
                'status_code' => 500,
                'data'        => [
                    'success' => false,
                    'message' => __( 'Database error occurred.', 'directorist-booking' ),
                    'error'   => $wpdb->last_error,
                ],
            ];
        }

        $bookings = [];

        if ( $results ) {
            foreach ( $results as $row ) {
                $booking_type = get_post_meta( (int) $row->listing_id, '_bdb_booking_type', true );

                // Parse comment data to extract structured information
                $comment_data = $this->parse_comment_data( $row->comment );

                $bookings[] = [
                    'id'                  => (int) $row->booking_id,
                    'booking_author_name' => $row->booking_author_name ?: '',
                    'listing_title'       => $row->listing_title,
                    'date_start'          => $row->date_start,
                    'date_end'            => $row->date_end,
                    'comment'             => $row->comment, // Keep original for backward compatibility
                    'booking_type'        => $booking_type,
                    'price'               => $row->price ? (float) $row->price : null,
                    // Add parsed comment fields
                    'guest_email'         => $comment_data['email'] ?? '',
                    'guest_phone'         => $comment_data['phone'] ?? '',
                    'guest_adults'        => $comment_data['adults'] ?? '',
                    'location'            => $comment_data['location'] ?? '',
                    'message'             => $comment_data['message'] ?? '',
                    'first_name'          => $comment_data['first_name'] ?? '',
                    'last_name'           => $comment_data['last_name'] ?? '',
                ];
            }
        }

        return [
            'status_code' => 200,
            'data'        => [
                'success'  => true,
                'count'    => count( $bookings ),
                'bookings' => $bookings,
            ],
        ];
    }

    /**
     * Parse comment data from various formats (JSON, serialized, etc.)
     *
     * @param string $comment Raw comment data from database
     * @return array Parsed comment data
     */
    private function parse_comment_data( $comment ) {
        $comment_data = [];

        if ( empty( $comment ) ) {
            return $comment_data;
        }

        // Try to parse as JSON first
        $json_data = json_decode( $comment, true );
        if ( json_last_error() === JSON_ERROR_NONE && is_array( $json_data ) ) {
            return $json_data;
        }

        // Try to parse as WordPress serialized data
        if ( is_serialized( $comment ) ) {
            $unserialized = maybe_unserialize( $comment );
            if ( is_array( $unserialized ) ) {
                return $unserialized;
            }
        }

        // Try to extract data from other formats
        if ( is_string( $comment ) ) {
            // Check if it contains structured data patterns
            if ( strpos( $comment, 'email' ) !== false || strpos( $comment, 'phone' ) !== false ) {
                // This might be a custom format - you can add specific parsing logic here
                error_log( 'BookingController: Comment contains structured data but is not JSON/serialized: ' . $comment );
            }
        }

        return $comment_data;
    }

    /**
     * Search booking events with advanced filtering
     *
     * @return array Search results
     */
    public function search_booking_events() {
        global $wpdb;

        $current_user_id = get_current_user_id();

        if ( ! $current_user_id ) {
            return [
                'status_code' => 401,
                'data'        => [
                    'success' => false,
                    'message' => __( 'User not logged in.', 'directorist-booking' ),
                ],
            ];
        }

        // Get search parameters
        $search_term  = isset( $_POST['search_term'] ) ? sanitize_text_field( wp_unslash( $_POST['search_term'] ) ) : '';
        $booking_type = isset( $_POST['booking_type'] ) ? sanitize_text_field( wp_unslash( $_POST['booking_type'] ) ) : '';
        $limit        = isset( $_POST['limit'] ) ? intval( $_POST['limit'] ) : 100;
        $offset       = isset( $_POST['offset'] ) ? intval( $_POST['offset'] ) : 0;

        $bookings_table = $wpdb->prefix . 'directorist_booking';
        $posts_table    = $wpdb->prefix . 'posts';
        $postmeta_table = $wpdb->prefix . 'postmeta';
        $users_table    = $wpdb->prefix . 'users';

        // Build search conditions
        $search_conditions = [];
        $search_params     = [$current_user_id];

        // Add search term conditions
        if ( ! empty( $search_term ) ) {
            $search_conditions[] = "(
				p.post_title LIKE %s OR 
				u.display_name LIKE %s OR 
				b.comment LIKE %s
			)";
            $search_like         = '%' . $wpdb->esc_like( $search_term ) . '%';
            $search_params[]     = $search_like;
            $search_params[]     = $search_like;
            $search_params[]     = $search_like;
        }

        // Add booking type condition
        if ( ! empty( $booking_type ) ) {
            $search_conditions[] = "pm.meta_value = %s";
            $search_params[]     = $booking_type;
        }

        // Build WHERE clause
        $where_clause = "b.status IN ('confirmed', 'paid') AND p.post_author = %d";
        if ( ! empty( $search_conditions ) ) {
            $where_clause .= " AND " . implode( ' AND ', $search_conditions );
        }

        $query = $wpdb->prepare(
            "SELECT 
				b.ID as booking_id,
				b.bookings_author,
				b.listing_id,
				b.date_start,
				b.date_end,
				b.comment,
				b.price,
				p.post_title as listing_title,
				p.post_author as listing_owner_id,
				pm.meta_value as booking_type_meta,
				u.display_name as booking_author_name
			FROM {$bookings_table} b
			INNER JOIN {$posts_table} p ON b.listing_id = p.ID
			LEFT JOIN {$postmeta_table} pm ON b.listing_id = pm.post_id AND pm.meta_key = '_booking_type'
			LEFT JOIN {$users_table} u ON b.bookings_author = u.ID
			WHERE {$where_clause}
			ORDER BY b.created DESC
			LIMIT %d OFFSET %d",
            array_merge( $search_params, [$limit, $offset] )
        );

        $results = $wpdb->get_results( $query );

        if ( $wpdb->last_error ) {
            return [
                'status_code' => 500,
                'data'        => [
                    'success' => false,
                    'message' => __( 'Database error occurred.', 'directorist-booking' ),
                    'error'   => $wpdb->last_error,
                ],
            ];
        }

        $bookings = [];

        if ( $results ) {
            foreach ( $results as $row ) {
                $booking_type_meta = get_post_meta( (int) $row->listing_id, '_bdb_booking_type', true );

                // Parse comment data to extract structured information
                $comment_data = $this->parse_comment_data( $row->comment );

                $bookings[] = [
                    'id'                  => (int) $row->booking_id,
                    'booking_author_name' => $row->booking_author_name ?: '',
                    'listing_title'       => $row->listing_title,
                    'date_start'          => $row->date_start,
                    'date_end'            => $row->date_end,
                    'comment'             => $row->comment, // Keep original for backward compatibility
                    'booking_type'        => $booking_type_meta,
                    'price'               => $row->price ? (float) $row->price : null,
                    // Add parsed comment fields
                    'guest_email'         => $comment_data['email'] ?? '',
                    'guest_phone'         => $comment_data['phone'] ?? '',
                    'guest_adults'        => $comment_data['adults'] ?? '',
                    'location'            => $comment_data['location'] ?? '',
                    'message'             => $comment_data['message'] ?? '',
                    'first_name'          => $comment_data['first_name'] ?? '',
                    'last_name'           => $comment_data['last_name'] ?? '',
                ];
            }
        }

        // Get total count for pagination
        $count_query = $wpdb->prepare(
            "SELECT COUNT(*) as total
			FROM {$bookings_table} b
			INNER JOIN {$posts_table} p ON b.listing_id = p.ID
			LEFT JOIN {$postmeta_table} pm ON b.listing_id = pm.post_id AND pm.meta_key = '_booking_type'
			LEFT JOIN {$users_table} u ON b.bookings_author = u.ID
			WHERE {$where_clause}",
            $search_params
        );

        $total_count = $wpdb->get_var( $count_query );

        return [
            'status_code' => 200,
            'data'        => [
                'success'      => true,
                'count'        => count( $bookings ),
                'total'        => (int) $total_count,
                'bookings'     => $bookings,
                'search_term'  => $search_term,
                'booking_type' => $booking_type,
            ],
        ];
    }
}