<?php

namespace Directorist_Booking\App\Providers;

defined( "ABSPATH" ) || exit;

use Directorist_Booking\WpMVC\Contracts\Provider;
use Directorist_Booking\WpMVC\Exceptions\Exception;
use Directorist_Booking\WpMVC\View\View;
use Directorist_Booking\App\Providers\Database;

class Dashboard implements Provider {
    public function boot() {
        add_action( "atbdp_tab_after_favorite_listings", [$this, "atbdp_tab_after_favorite_listings"] );
        add_action( "atbdp_tab_after_favorite_listings", [$this, "atbdp_tab_after_my_bookings"] );
        add_action( "atbdp_tab_content_after_favorite", [$this, "atbdp_tab_content_after_favorite"] );
        add_action( "atbdp_tab_content_after_favorite", [$this, "atbdp_booking_tab_content_after_my_bookings"] );

        // non legacy template
        add_action( 'directorist_after_dashboard_navigation', [ $this, 'non_legacy_add_dashboard_nav_link' ] );
        add_action( 'directorist_after_dashboard_navigation', [ $this, 'non_legacy_all_add_dashboard_nav_link' ] );
        add_action( 'directorist_after_dashboard_navigation', [ $this, 'dashboard_calender_booking_nav_link' ] );
        add_action( 'directorist_after_dashboard_contents', [ $this, 'non_legacy_add_dashboard_nav_content' ] );

        add_action( 'directorist_after_dashboard_contents', [ $this, 'dashboard_calender_booking_nav_content' ] );
        add_action( 'directorist_after_dashboard_contents', [ $this, 'non_legacy_all_add_dashboard_nav_content' ] );

        add_action( 'wp_footer', [ $this, 'atpp_plan_change_modal' ] );
        
        // Register AJAX handler for advanced booking filter
        add_action( 'wp_ajax_bdb_advanced_filter_bookings', [ $this, 'handle_advanced_filter_bookings' ] );
        
        // Register AJAX handler for booking calendar search
        add_action( 'wp_ajax_bdb_search_booking_events', [ $this, 'handle_search_booking_events' ] );
    }

    public function atpp_plan_change_modal() {

        echo View::get( 'user-refund-form' );
    }

    public function dashboard_calender_booking_nav_link() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        $user_id        = get_current_user_id();
        $user_type      = (string) get_user_meta( $user_id, '_user_type', true );
        
        if ( empty( $enable_booking ) || ( ! empty( $user_type ) && 'general' == $user_type ) ) return;

        // Get current tab from URL for active state
        $current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : '';
        $is_active   = ( $current_tab === 'booking_calender' ) ? ' active' : '';

        $html = '<li class="directorist-tab__nav__item' . $is_active . '"><a href="#" class="directorist-booking-nav-link directorist-tab__nav__link" id="booking_calender_tab" target="booking_calender"><span class="directorist_menuItem-text"><span class="directorist_menuItem-icon">' . directorist_icon( 'las la-calendar', false ) . '</span>' . esc_html__( 'Booking Calendar', 'directorist-booking' ) . '</span></a></li>';
        echo apply_filters( 'directorist_user_dashboard_booking_calender_tab', $html );
    }

    public function dashboard_calender_booking_nav_content() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        wp_enqueue_style( 'bdb-daterangepicker' );
        wp_enqueue_script( 'bdb-main' );
        wp_enqueue_script( 'bdb-dashboard' );
        wp_enqueue_script( 'bdb-moment' );
        wp_enqueue_script( 'bdb-daterangepicker' );

        // Get current tab from URL for visibility
        $current_tab   = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : '';
        $display_style = ( $current_tab === 'booking_calender' ) ? 'block' : 'none';
        
        // Calendar content rendered outside all_bookings wrapper
        $html = '<div class="directorist-tab__pane" id="booking_calender" style="display: ' . $display_style . ';">';
        echo apply_filters( 'atbdp_user_dashboard_booking_calender_content_wrapper', $html );
        
        echo View::get( 'booking-calender' );
        
        echo wp_kses_post( '</div>' );
    }

    public function non_legacy_all_add_dashboard_nav_link() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        $user_id        = get_current_user_id();
        $user_type      = (string) get_user_meta( $user_id, '_user_type', true );
        
        if ( empty( $enable_booking ) || ( ! empty( $user_type ) && 'general' == $user_type ) ) return;

        // Get current tab from URL for active state
        $current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : '';
        $is_active   = ( $current_tab === 'all_bookings' );

        // Get total booking count
        $total_bookings = bdb_count_bookings( $user_id );

        $html  = '<li class="directorist-tab__nav__item' . ( $is_active ? ' active' : '' ) . '">';
        $html .= '<a href="#" class="directorist-booking-nav-link directorist-tab__nav__link" target="all_bookings"><span class="directorist_menuItem-text"><span class="directorist_menuItem-icon">' . directorist_icon( 'las la-clipboard-list', false ) . '</span>' . __( "All Bookings", "directorist-booking" );
        
        // Add booking count badge if there are bookings
        if ( $total_bookings > 0 ) {
            $html .= '<span class="directorist_menuItem-badge">' . $total_bookings . '</span>';
        }
        
        $html .= '</span></a>';
        $html .= '</li>';

        echo apply_filters( 'atbdp_user_dashboard_all_bookings_tab', $html );
    }

    public function non_legacy_all_add_dashboard_nav_content() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        wp_enqueue_style( 'bdb-daterangepicker' );
        wp_enqueue_script( 'bdb-main' );
        wp_enqueue_script( 'bdb-dashboard' );
        // wp_enqueue_script( 'bdb-navigation-state' );
        wp_enqueue_script( 'bdb-moment' );
        wp_enqueue_script( 'bdb-daterangepicker' );
        
        // Get current tab from URL for visibility
        $current_tab   = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : '';
        $display_style = ( $current_tab === 'all_bookings' ) ? 'block' : 'none';
        
        // Create a wrapper div for all booking tabs
        $html = '<div class="directorist-tab__pane" id="all_bookings" style="display: ' . $display_style . ';">';
        echo apply_filters( 'atbdp_user_dashboard_all_bookings_content_wrapper', $html );
        
        $html = '<div class="directorist-all-bookings-wrapper">';
        echo $html;
        
        // Determine which tab should be active based on status
        $current_status = isset( $_GET['status'] ) ? sanitize_text_field( $_GET['status'] ) : 'confirmed';
        
        // Map status values to tab IDs
        $status_tab_map = [
            'confirmed' => 'all_booking_approved',
            'waiting'   => 'all_booking_waiting',
            'cancelled' => 'all_booking_cancelled',
            'refund'    => 'all_booking_refund'
        ];
        
        // Determine which tab should be active based on status
        $active_tab = 'all_booking_approved'; // default - always show approved as active
        if ( $current_status && isset( $status_tab_map[ $current_status ] ) ) {
            $active_tab = $status_tab_map[ $current_status ];
        }
        
        // Add the tab navigation
        $this->render_booking_tab_navigation( $current_tab, $active_tab );
        
        // Approved tab content - show when this tab is active
        $display_style = ( $active_tab === 'all_booking_approved' ) ? 'block' : 'none';
        $html          = '<div class="directorist-tab__pane" id="all_booking_approved" style="display: ' . $display_style . ';">';
        echo apply_filters( 'atbdp_user_dashboard_approved_bookings_content_wrapper', $html );
        
        $listings       = $this->get_agent_listings( '' );
        $args           = [
            'owner_id' => get_current_user_id(),
            'type'     => 'reservation',
        ];
        $limit          = 5;
        $pages          = '';
        $_GET['status'] = 'approved';
        
        if ( isset( $_GET['status'] ) ) {
            $booking_max    = bdb_count_bookings( get_current_user_id(), $_GET['status'] );
            $pages          = ceil( $booking_max / $limit );
            $args['status'] = $_GET['status'];
        } else {
            $booking_max = bdb_count_bookings( get_current_user_id() );
            $pages       = ceil( $booking_max / $limit );
        }
        
            $bookings = Database::get_newest_bookings( $args, $limit );
        $data         = [
            'message'  => '',
            'bookings' => $bookings,
            'listings' => $listings,
            'pages'    => $pages
        ];

        if ( is_user_logged_in() ) {
            echo View::get(
                'all-bookings', [ 
                    'data'           => $data,
                    'tab_context'    => 'all_bookings',
                    'status_context' => 'approved'
                ] 
            );
        }
            echo wp_kses_post( '</div>' );

        // Pending tab content - show when this tab is active
        $display_style = ( $active_tab === 'all_booking_waiting' ) ? 'block' : 'none';
        $html          = '<div class="directorist-tab__pane" id="all_booking_waiting" style="display: ' . $display_style . ';">';
            echo apply_filters( 'atbdp_user_dashboard_pending_bookings_content_wrapper', $html );
        
            $_GET['status'] = 'waiting';
        if ( isset( $_GET['status'] ) ) {
            $booking_max    = bdb_count_bookings( get_current_user_id(), $_GET['status'] );
            $pages          = ceil( $booking_max / $limit );
            $args['status'] = $_GET['status'];
        } else {
            $booking_max = bdb_count_bookings( get_current_user_id() );
            $pages       = ceil( $booking_max / $limit );
        }
        
            $bookings = Database::get_newest_bookings( $args, $limit );
        $data         = [
            'message'  => '',
            'bookings' => $bookings,
            'listings' => $listings,
            'pages'    => $pages
        ];

        if ( atbdp_logged_in_user() ) {
            echo View::get(
                'all-bookings', [ 
                    'data'           => $data,
                    'tab_context'    => 'all_bookings',
                    'status_context' => 'waiting'
                ] 
            );
        }
            echo wp_kses_post( '</div>' );

        // Cancelled tab content - show when this tab is active
        $display_style = ( $active_tab === 'all_booking_cancelled' ) ? 'block' : 'none';
        $html          = '<div class="directorist-tab__pane" id="all_booking_cancelled" style="display: ' . $display_style . ';">';
            echo apply_filters( 'atbdp_user_dashboard_cancelled_bookings_content_wrapper', $html );
        
            $_GET['status'] = 'cancelled';
        if ( isset( $_GET['status'] ) ) {
            $booking_max    = bdb_count_bookings( get_current_user_id(), $_GET['status'] );
            $pages          = ceil( $booking_max / $limit );
            $args['status'] = $_GET['status'];
        } else {
            $booking_max = bdb_count_bookings( get_current_user_id() );
            $pages       = ceil( $booking_max / $limit );
        }
        
            $bookings = Database::get_newest_bookings( $args, $limit );
        $data         = [
            'message'  => '',
            'bookings' => $bookings,
            'listings' => $listings,
            'pages'    => $pages
        ];

        if ( atbdp_logged_in_user() ) {
            echo View::get(
                'all-bookings', [ 
                    'data'           => $data,
                    'tab_context'    => 'all_bookings',
                    'status_context' => 'cancelled'
                ] 
            );
        }
            echo wp_kses_post( '</div>' );

        // Refund request tab content - show when this tab is active
        $display_style = ( $active_tab === 'all_booking_refund' ) ? 'block' : 'none';
            $html      = '<div class="directorist-tab__pane" id="all_booking_refund" style="display: ' . $display_style . ';">';
            echo apply_filters( 'atbdp_user_dashboard_refund_bookings_content_wrapper', $html );
            
            $_GET['status'] = 'refund';
        if ( isset( $_GET['status'] ) ) {
            $booking_max    = bdb_count_bookings( get_current_user_id(), $_GET['status'] );
            $pages          = ceil( $booking_max / $limit );
            $args['status'] = $_GET['status'];
        } else {
            $booking_max = bdb_count_bookings( get_current_user_id() );
            $pages       = ceil( $booking_max / $limit );
        }

            $bookings = Database::get_newest_bookings( $args, $limit );
            $data     = [
                'message'  => '',
                'bookings' => $bookings,
                'listings' => $listings,
                'pages'    => $pages
            ];

            if ( is_user_logged_in() ) {
                echo View::get( 'refund-policy/refund-request', [ 'data' => $data ] );
            }
        
            echo wp_kses_post( '</div>' ); // Close wrapper
            echo wp_kses_post( '</div>' ); // Close tab pane
            echo wp_kses_post( '</div>' ); // Close tab pane
    }

    private function render_booking_tab_navigation( $current_tab, $active_tab ) {
        
        ?>
        <div class="directorist-all-bookings-tabs">
            <div class="directorist-tab-nav">
                <!-- Approved tab -->
                <div class="directorist-tab-nav__item<?php echo ( $active_tab === 'all_booking_approved' ) ? ' active' : ''; ?>" data-tab="all_booking_approved">
                    <a href="#" class="directorist-tab-nav__link" data-tab="all_booking_approved">
                        <span class="directorist-tab-nav__icon">
                            <i class="directorist-icon-mask" aria-hidden="true" style="--directorist-icon: url(<?php echo home_url(); ?>/wp-content/plugins/directorist/assets/icons/line-awesome/svgs/check-circle-solid.svg)"></i>
                        </span>
                        <span class="directorist-tab-nav__text"><?php _e( "Approved", "directorist-booking" ); ?></span>
                    </a>
                </div>

                <!-- Pending tab -->
                <div class="directorist-tab-nav__item<?php echo ( $active_tab === 'all_booking_waiting' ) ? ' active' : ''; ?>" data-tab="all_booking_waiting">
                    <a href="#" class="directorist-tab-nav__link" data-tab="all_booking_waiting">
                        <span class="directorist-tab-nav__icon">
                            <i class="directorist-icon-mask" aria-hidden="true" style="--directorist-icon: url(<?php echo home_url(); ?>/wp-content/plugins/directorist/assets/icons/line-awesome/svgs/clock-solid.svg)"></i>
                        </span>
                        <span class="directorist-tab-nav__text"><?php _e( "Pending", "directorist-booking" ); ?></span>
                        <?php 
                        $waiting_count = bdb_count_bookings( get_current_user_id(), 'waiting' );
                        if ( $waiting_count > 0 ) : ?>
                            <span class="directorist-tab-nav__badge"><?php echo $waiting_count; ?></span>
                        <?php endif; ?>
                    </a>
                </div>

                <!-- Cancelled tab -->
                <div class="directorist-tab-nav__item<?php echo ( $active_tab === 'all_booking_cancelled' ) ? ' active' : ''; ?>" data-tab="all_booking_cancelled">
                    <a href="#" class="directorist-tab-nav__link" data-tab="all_booking_cancelled">
                        <span class="directorist-tab-nav__icon">
                            <i class="directorist-icon-mask" aria-hidden="true" style="--directorist-icon: url(<?php echo home_url(); ?>/wp-content/plugins/directorist/assets/icons/line-awesome/svgs/times-circle-solid.svg)"></i>
                        </span>
                        <span class="directorist-tab-nav__text"><?php _e( "Cancelled", "directorist-booking" ); ?></span>
                    </a>
                </div>

                <!-- Refund Request tab -->
                <div class="directorist-tab-nav__item<?php echo ( $active_tab === 'all_booking_refund' ) ? ' active' : ''; ?>" data-tab="all_booking_refund">
                    <a href="#" class="directorist-tab-nav__link" data-tab="all_booking_refund">
                        <span class="directorist-tab-nav__icon">
                            <i class="directorist-icon-mask" aria-hidden="true" style="--directorist-icon: url(<?php echo home_url(); ?>/wp-content/plugins/directorist/assets/icons/line-awesome/svgs/dollar-sign-solid.svg)"></i>
                        </span>
                        <span class="directorist-tab-nav__text"><?php _e( "Refund Request", "directorist-booking" ); ?></span>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    private function render_my_bookings_tab_navigation( $current_tab, $active_tab ) {
        
        ?>
        <div class="directorist-all-bookings-tabs">
            <div class="directorist-tab-nav">
                <!-- Approved tab -->
                <div class="directorist-tab-nav__item<?php echo ( $active_tab === 'my_booking_approved' ) ? ' active' : ''; ?>" data-tab="my_booking_approved">
                    <a href="#" class="directorist-tab-nav__link" data-tab="my_booking_approved">
                        <span class="directorist-tab-nav__icon">
                            <i class="directorist-icon-mask" aria-hidden="true" style="--directorist-icon: url(<?php echo home_url(); ?>/wp-content/plugins/directorist/assets/icons/line-awesome/svgs/check-circle-solid.svg)"></i>
                        </span>
                        <span class="directorist-tab-nav__text"><?php _e( "Approved", "directorist-booking" ); ?></span>
                    </a>
                </div>

                <!-- Pending tab -->
                <div class="directorist-tab-nav__item<?php echo ( $active_tab === 'my_booking_waiting' ) ? ' active' : ''; ?>" data-tab="my_booking_waiting">
                    <a href="#" class="directorist-tab-nav__link" data-tab="my_booking_waiting">
                        <span class="directorist-tab-nav__icon">
                            <i class="directorist-icon-mask" aria-hidden="true" style="--directorist-icon: url(<?php echo home_url(); ?>/wp-content/plugins/directorist/assets/icons/line-awesome/svgs/clock-solid.svg)"></i>
                        </span>
                        <span class="directorist-tab-nav__text"><?php _e( "Pending", "directorist-booking" ); ?></span>
                        <?php 
                        $waiting_count = bdb_count_my_bookings_by_status( get_current_user_id(), 'waiting' );
                        if ( $waiting_count > 0 ) : ?>
                            <span class="directorist-tab-nav__badge"><?php echo $waiting_count; ?></span>
                        <?php endif; ?>
                    </a>
                </div>

                <!-- Cancelled tab -->
                <div class="directorist-tab-nav__item<?php echo ( $active_tab === 'my_booking_cancelled' ) ? ' active' : ''; ?>" data-tab="my_booking_cancelled">
                    <a href="#" class="directorist-tab-nav__link" data-tab="my_booking_cancelled">
                        <span class="directorist-tab-nav__icon">
                            <i class="directorist-icon-mask" aria-hidden="true" style="--directorist-icon: url(<?php echo home_url(); ?>/wp-content/plugins/directorist/assets/icons/line-awesome/svgs/times-circle-solid.svg)"></i>
                        </span>
                        <span class="directorist-tab-nav__text"><?php _e( "Cancelled", "directorist-booking" ); ?></span>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    public function non_legacy_add_dashboard_nav_link() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        // Get current tab from URL for active state
        $current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : '';
        $is_active   = ( $current_tab === 'my_booking' ) ? ' active' : '';

        $html = '<li class="directorist-tab__nav__item' . $is_active . '"><a href="#" class="directorist-booking-nav-link directorist-tab__nav__link" id="my_bookings_tab" target="my_booking"><span class="directorist_menuItem-text"><span class="directorist_menuItem-icon">' . directorist_icon( 'las la-calendar-check', false ) . '</span>' . esc_html__( 'My Bookings', 'directorist-booking' ) . '</span></a></li>';
        echo apply_filters( 'atbdp_user_dashboard_booking_tab', $html );
    }

    public function non_legacy_add_dashboard_nav_content() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        wp_enqueue_style( 'bdb-daterangepicker' );
        wp_enqueue_script( 'bdb-main' );
        wp_enqueue_script( 'bdb-dashboard' );
        // wp_enqueue_script( 'bdb-navigation-state' );
        wp_enqueue_script( 'bdb-moment' );
        wp_enqueue_script( 'bdb-daterangepicker' );

        // Get current tab from URL for visibility
        $current_tab   = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : '';
        $display_style = ( $current_tab === 'my_booking' || empty( $current_tab ) ) ? 'block' : 'none';
        
        // Create a wrapper div for my booking tabs
        $html = '<div class="directorist-tab__pane" id="my_booking" style="display: ' . $display_style . ';">';
        echo apply_filters( 'atbdp_user_dashboard_booking_content_wrapper', $html );
        
        $html = '<div class="directorist-all-bookings-wrapper">';
        echo $html;
        
        // Determine which tab should be active based on status
        $current_status = isset( $_GET['status'] ) ? sanitize_text_field( $_GET['status'] ) : 'confirmed';
        
        // Map status values to tab IDs for my bookings
        $status_tab_map = [
            'confirmed' => 'my_booking_approved',
            'waiting'   => 'my_booking_waiting',
            'cancelled' => 'my_booking_cancelled'
        ];
        
        // Determine which tab should be active based on status
        $active_tab = 'my_booking_approved'; // default - always show approved as active
        if ( $current_status && isset( $status_tab_map[ $current_status ] ) ) {
            $active_tab = $status_tab_map[ $current_status ];
        }
        
        // Add the tab navigation
        $this->render_my_bookings_tab_navigation( $current_tab, $active_tab );
        
        // Approved tab content - show when this tab is active
        $display_style = ( $active_tab === 'my_booking_approved' ) ? 'block' : 'none';
        $html          = '<div class="directorist-tab__pane" id="my_booking_approved" style="display: ' . $display_style . ';">';
        echo $html;

        $args  = [
            'bookings_author' => get_current_user_id(),
            'type'            => 'reservation',
            'status'          => 'approved'
        ];
        $limit = 5;

        $bookings    = Database::get_newest_bookings( $args, $limit );
        $booking_max = bdb_count_my_bookings_by_status( get_current_user_id(), 'confirmed' );
        $pages       = ceil( $booking_max / $limit );
        $data        = [
            'message'  => '',
            'type'     => 'user_booking',
            'bookings' => $bookings,
            'pages'    => $pages
        ]; ?>

        <div id="my_bookings_area">
            <?php 
            echo View::get(
                'all-bookings', [
                    'data'           => $data,
                    'tab_context'    => 'my_booking',
                    'status_context' => 'approved'
                ] 
            );
            ?>
        </div>

        <?php
        echo wp_kses_post( '</div>' ); // Close approved tab pane

        // Pending tab content - show when this tab is active
        $display_style = ( $active_tab === 'my_booking_waiting' ) ? 'block' : 'none';
        $html          = '<div class="directorist-tab__pane" id="my_booking_waiting" style="display: ' . $display_style . ';">';
        echo $html;

        $args = [
            'bookings_author' => get_current_user_id(),
            'type'            => 'reservation',
            'status'          => 'waiting'
        ];

        $bookings    = Database::get_newest_bookings( $args, $limit );
        $booking_max = bdb_count_my_bookings_by_status( get_current_user_id(), 'waiting' );
        $pages       = ceil( $booking_max / $limit );
        $data        = [
            'message'  => '',
            'type'     => 'user_booking',
            'bookings' => $bookings,
            'pages'    => $pages
        ]; ?>

        <div id="my_bookings_area">
            <?php 
            echo View::get(
                'all-bookings', [
                    'data'           => $data,
                    'tab_context'    => 'my_booking',
                    'status_context' => 'waiting'
                ] 
            );
            ?>
        </div>

        <?php
        echo wp_kses_post( '</div>' ); // Close waiting tab pane

        // Cancelled tab content - show when this tab is active
        $display_style = ( $active_tab === 'my_booking_cancelled' ) ? 'block' : 'none';
        $html          = '<div class="directorist-tab__pane" id="my_booking_cancelled" style="display: ' . $display_style . ';">';
        echo $html;

        $args = [
            'bookings_author' => get_current_user_id(),
            'type'            => 'reservation',
            'status'          => 'cancelled'
        ];

        $bookings    = Database::get_newest_bookings( $args, $limit );
        $booking_max = bdb_count_my_bookings_by_status( get_current_user_id(), 'cancelled' );
        $pages       = ceil( $booking_max / $limit );
        $data        = [
            'message'  => '',
            'type'     => 'user_booking',
            'bookings' => $bookings,
            'pages'    => $pages
        ]; ?>

        <div id="my_bookings_area">
            <?php 
            echo View::get(
                'all-bookings', [
                    'data'           => $data,
                    'tab_context'    => 'my_booking',
                    'status_context' => 'cancelled'
                ] 
            );
            ?>
        </div>

        <?php
        echo wp_kses_post( '</div>' ); // Close cancelled tab pane
        
        echo wp_kses_post( '</div>' ); // Close wrapper
        echo wp_kses_post( '</div>' ); // Close tab pane
    }

    // added the tab on dashboard
    public function atbdp_tab_after_favorite_listings() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        $html = '<li class="atbdp_tab_nav--content-link"><a href="#" class="atbd_tn_link" id="my_bookings_tab" target="my_booking"> ' . esc_html__( 'My Bookings', 'directorist-booking' ) . '</a></li>';
        echo apply_filters( 'atbdp_user_dashboard_booking_tab', $html );
    }

    // add all bookings tab on dashboard
    public function atbdp_tab_after_my_bookings() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        $html  = '<li class="atbdp_tab_nav--content-link atbdp_tab_nav--has-child atbdp_all_booking_nav">';
        $html .= '<a href="#" class="atbdp_all_booking_nav-link directorist-tab__nav__link atbd-dash-nav-dropdown">' . __( "All Bookings", "directorist-booking" ) . ' ' . directorist_icon( 'fas fa-angle-down', false ) . '</a>';
        $html .= '<ul class="atbd-dashboard-nav">';
        $html .= '<li><a href="#" class="atbd_tn_link" target="booking_approved">' . __( "Approved", "directorist-booking" ) . '</a></li>';
        $html .= '<li><a href="#" class="atbd_tn_link" target="booking_waiting">' . __( "Pending", "directorist-booking" ) . '</a></li>';
        $html .= '<li><a href="#" class="atbd_tn_link" target="booking_cancelled">' . __( "Cancelled", "directorist-booking" ) . '</a></li>';
        $html .= '</ul>';
        $html .= '</li>';

        echo apply_filters( 'atbdp_user_dashboard_all_bookings_tab', $html );
    }

    // content of my booking tab
    public function atbdp_tab_content_after_favorite() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        wp_enqueue_style( 'bdb-daterangepicker' );
        wp_enqueue_script( 'bdb-main' );
        wp_enqueue_script( 'bdb-dashboard' );
        wp_enqueue_script( 'bdb-moment' );
        wp_enqueue_script( 'bdb-daterangepicker' );

        $html = '<div class="atbd_tab_inner" id="my_booking">';
        echo apply_filters( 'atbdp_user_dashboard_booking_content_wrapper', $html );

        $args  = [
            'bookings_author' => get_current_user_id(),
            'type'            => 'reservation'
        ];
        $limit = 5;

        $bookings    = Database::get_newest_bookings( $args, $limit );
        $booking_max = bdb_count_my_bookings( get_current_user_id() );
        $pages       = ceil( $booking_max / $limit );
        $data        = [
            'message'  => '',
            'type'     => 'user_booking',
            'bookings' => $bookings,
            'pages'    => $pages
        ]; ?>

        <div id="my_bookings_area">
            <?php 
                //include BDB_TEMPLATES_DIR . '/all-bookings.php'; 
                echo View::get(
                    'all-bookings', [
                        'data' => $data,
                    ] 
                );
            ?>
        </div>

        <div class="test"></div>

        <?php
        echo wp_kses_post( '</div>' );
    }

    public function atbdp_booking_tab_content_after_my_bookings() {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        wp_enqueue_style( 'bdb-daterangepicker' );
        wp_enqueue_script( 'bdb-main' );
        wp_enqueue_script( 'bdb-dashboard' );
        wp_enqueue_script( 'bdb-moment' );
        wp_enqueue_script( 'bdb-daterangepicker' );
        // approved tab content
        $html = '<div class="atbd_tab_inner" id="booking_approved">';
        echo apply_filters( 'atbdp_user_dashboard_approved_bookings_content_wrapper', $html );
            $listings       = $this->get_agent_listings( '' );
            $args           = [
                'owner_id' => get_current_user_id(),
                'type'     => 'reservation',
            ];
            $limit          = 10;
            $pages          = '';
            $_GET['status'] = 'approved';
            if ( isset( $_GET['status'] ) ) {
                $booking_max    = bdb_count_bookings( get_current_user_id(), $_GET['status'] );
                $pages          = ceil( $booking_max / $limit );
                $args['status'] = $_GET['status'];
            } else {
                $booking_max = bdb_count_bookings( get_current_user_id() );
                $pages       = ceil( $booking_max / $limit );
            }
            $bookings = Database::get_newest_bookings( $args, $limit );
            $data     = [
                'message'  => '',
                'bookings' => $bookings,
                'listings' => $listings,
                'pages'    => $pages
            ];

            if ( atbdp_logged_in_user() ) {
                //include BDB_TEMPLATES_DIR . '/all-bookings.php';
                echo View::get(
                    'all-bookings', [
                        'data' => $data,
                    ] 
                );
            }
            echo wp_kses_post( '</div>' );

            //pending tab
            $html = '<div class="atbd_tab_inner" id="booking_waiting">';
            echo apply_filters( 'atbdp_user_dashboard_pending_bookings_content_wrapper', $html );
            $listings       = $this->get_agent_listings( '' );
            $args           = [
                'owner_id' => get_current_user_id(),
                'type'     => 'reservation',
            ];
            $limit          = 10;
            $pages          = '';
            $_GET['status'] = 'waiting';
            if ( isset( $_GET['status'] ) ) {
                $booking_max    = bdb_count_bookings( get_current_user_id(), $_GET['status'] );
                $pages          = ceil( $booking_max / $limit );
                $args['status'] = $_GET['status'];
            } else {
                $booking_max = bdb_count_bookings( get_current_user_id() );
                $pages       = ceil( $booking_max / $limit );
            }
            $bookings = Database::get_newest_bookings( $args, $limit );
            $data     = [
                'message'  => '',
                'bookings' => $bookings,
                'listings' => $listings,
                'pages'    => $pages
            ];

            if ( atbdp_logged_in_user() ) {
                //include BDB_TEMPLATES_DIR . '/all-bookings.php';
                echo View::get(
                    'all-bookings', [
                        'data' => $data,
                    ] 
                );
            }
            echo wp_kses_post( '</div>' );

            //cancelled tab
            $html = '<div class="atbd_tab_inner" id="booking_cancelled">';
            echo apply_filters( 'atbdp_user_dashboard_cancelled_bookings_content_wrapper', $html );
            $listings       = $this->get_agent_listings( '' );
            $args           = [
                'owner_id' => get_current_user_id(),
                'type'     => 'reservation',
            ];
            $limit          = 10;
            $pages          = '';
            $_GET['status'] = 'cancelled';
            if ( isset( $_GET['status'] ) ) {
                $booking_max    = bdb_count_bookings( get_current_user_id(), $_GET['status'] );
                $pages          = ceil( $booking_max / $limit );
                $args['status'] = $_GET['status'];
            } else {
                $booking_max = bdb_count_bookings( get_current_user_id() );
                $pages       = ceil( $booking_max / $limit );
            }
            $bookings = Database::get_newest_bookings( $args, $limit );
            $data     = [
                'message'  => '',
                'bookings' => $bookings,
                'listings' => $listings,
                'pages'    => $pages
            ];

            if ( atbdp_logged_in_user() ) {
                //include BDB_TEMPLATES_DIR . '/all-bookings.php';
                echo View::get(
                    'all-bookings', [
                        'data' => $data,
                    ] 
                );
            }
            echo wp_kses_post( '</div>' );
    }

    /**
     * Function to get ids added by the user/agent
     * @return array array of listing ids
     */
    public function get_agent_listings( $status ) {
        $enable_booking = get_directorist_option( 'enable_booking', 1 );
        if ( empty( $enable_booking ) ) return;

        $current_user = wp_get_current_user();

        switch ( $status ) {
            case 'pending':
                $post_status = ['pending_payment', 'draft', 'pending'];
                break;

            case 'active':
                $post_status = ['publish'];
                break;

            case 'expired':
                $post_status = ['expired'];
                break;

            default:
                $post_status = ['publish', 'pending_payment', 'expired', 'draft', 'pending'];
                break;
        }

        return get_posts(
            [
                'author'         => $current_user->ID,
                'fields'         => 'ids', // Only get post IDs
                'posts_per_page' => -1,
                'post_type'      => ATBDP_POST_TYPE,
                'post_status'    => $post_status,
            ]
        );
    }

    /**
     * Handle advanced booking filter AJAX request
     */
    public function handle_advanced_filter_bookings() {
        // Verify nonce
        if ( ! wp_verify_nonce( $_POST['nonce'], directorist_get_nonce_key() ) ) {
            wp_send_json_error(
                [
                    'message' => __( 'Security check failed', 'directorist-booking' )
                ]
            );
        }

        $filters        = isset( $_POST['filters'] ) ? $_POST['filters'] : [];
        $current_tab    = isset( $_POST['current_tab'] ) ? sanitize_text_field( $_POST['current_tab'] ) : 'all_bookings';
        $current_status = isset( $_POST['current_status'] ) ? sanitize_text_field( $_POST['current_status'] ) : '';

        try {
            // Start timing for performance monitoring
            $start_time = microtime( true );
            
            // Build query arguments based on filters
            $args = $this->build_filter_args( $filters, $current_tab, $current_status );
            
            // Get filtered bookings with optimized query
            $limit    = 5;
            $offset   = 0;
            $bookings = Database::get_newest_bookings( $args, $limit, $offset );

            // Get total count for pagination
            $total_count  = $this->get_filtered_bookings_count( $args );
            $pages        = ceil( $total_count / $limit );
            $current_page = isset( $_POST['page'] ) ? absint( $_POST['page'] ) : 1;
            
            // Prepare response data
            $data = [
                'bookings'       => $bookings,
                'pages'          => $pages,
                'total_count'    => $total_count,
                'bookings_html'  => $this->render_bookings_html( $bookings, $current_tab ),
                'pagination'     => bdb_ajax_pagination( $pages, $current_page ), // Simplified pagination
                'message'        => sprintf( __( 'Found %d bookings', 'directorist-booking' ), $total_count ),
                'execution_time' => round( ( microtime( true ) - $start_time ) * 1000, 2 ) . 'ms'
            ];

            wp_send_json_success( $data );

        } catch ( Exception $e ) {
            // Log the error for debugging
            wp_send_json_error(
                [
                    'message' => __( 'Failed to filter bookings. Please try again.', 'directorist-booking' ),
                    'debug'   => $e->getMessage()
                ]
            );
        }
    }

    /**
     * Build query arguments based on filters
     */
    private function build_filter_args( $filters, $current_tab, $current_status ) {
        
        // Determine which bookings to show based on tab
        if ( $current_tab === 'my_booking' || $current_tab === 'my_bookings' ) {
            // Show bookings made by current user (as customer)
            $args = [
                'bookings_author' => get_current_user_id(),
                'type'            => 'reservation'
            ];
        } else {
            // Show bookings for listings owned by current user (as owner)
            $args = [
                'owner_id' => get_current_user_id(),
                'type'     => 'reservation'
            ];
        }

        // Handle search filter
        if ( ! empty( $filters['search'] ) ) {
            $args['search'] = sanitize_text_field( $filters['search'] );
        }

        // Handle status filters (keep for backward compatibility)
        if ( ! empty( $filters['status'] ) && is_array( $filters['status'] ) ) {
            if ( count( $filters['status'] ) === 1 ) {
                $args['status'] = $filters['status'][0];
            } else {
                // Multiple statuses - we'll handle this in the query
                $args['status'] = $filters['status'];
            }
        } elseif ( $current_status ) {
            $args['status'] = $current_status;
        }

        // Handle date filters
        if ( ! empty( $filters['date_from'] ) || ! empty( $filters['date_to'] ) ) {
            $args['date_from'] = ! empty( $filters['date_from'] ) ? sanitize_text_field( $filters['date_from'] ) : '';
            $args['date_to']   = ! empty( $filters['date_to'] ) ? sanitize_text_field( $filters['date_to'] ) : '';
            
        }

        // Handle sort order
        if ( ! empty( $filters['sort_by'] ) ) {
            $args['sort_by'] = sanitize_text_field( $filters['sort_by'] );
        }
        
        return $args;
    }

    /**
     * Get count of filtered bookings
     */
    private function get_filtered_bookings_count( $args ) {
        global $wpdb;
        
        $where_conditions = ['NOT comment = "owner reservations"'];
        
        // Handle status filter
        if ( isset( $args['status'] ) ) {
            if ( is_array( $args['status'] ) ) {
                $status_placeholders = implode( ',', array_fill( 0, count( $args['status'] ), '%s' ) );
                $where_conditions[]  = $wpdb->prepare( "status IN ($status_placeholders)", $args['status'] );
            } else {
                $where_conditions[] = $wpdb->prepare( 'status = %s', $args['status'] );
            }
        }

        // Handle date filters - filter by booking date (date_start) not request date (created)
        // Extract only date part from datetime fields for proper comparison
        if ( ! empty( $args['date_from'] ) ) {
            $where_conditions[] = $wpdb->prepare( 'DATE(date_start) >= %s', $args['date_from'] );
        }
        if ( ! empty( $args['date_to'] ) ) {
            $where_conditions[] = $wpdb->prepare( 'DATE(date_start) <= %s', $args['date_to'] );
        }

        // Handle listing filter
        if ( ! empty( $args['listing_id'] ) ) {
            $where_conditions[] = $wpdb->prepare( 'listing_id = %d', $args['listing_id'] );
        }

        $where_clause = implode( ' AND ', $where_conditions );
        
        $query = "SELECT COUNT(*) FROM {$wpdb->prefix}directorist_booking WHERE $where_clause";
        
        return $wpdb->get_var( $query );
    }

    /**
     * Render bookings HTML
     */
    private function render_bookings_html( $bookings, $current_tab ) {
        if ( empty( $bookings ) ) {
            return '<p class="no-bookings-found">' . __( 'No bookings found matching your criteria.', 'directorist-booking' ) . '</p>';
        }

        $html = '<div class="directorist-booking-card-container">';
        
        foreach ( $bookings as $booking ) {
            $booking['listing_title'] = get_the_title( $booking['listing_id'] );
            
            if ( $current_tab === 'my_booking' ) {
                $html .= View::get( 'dashboard/content-user-booking', ['value' => $booking] );
            } else {
                $html .= View::get( 'dashboard/content-booking', ['value' => $booking] );
            }
        }
        
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render pagination HTML
     */
    private function render_pagination( $pages, $current_page ) {
        if ( $pages <= 1 ) {
            return '';
        }

        $html = '<div class="directorist-pagination">';
        
        // Previous button
        if ( $current_page > 1 ) {
            $html .= '<a href="#" class="page-link prev-page" data-page="' . ( $current_page - 1 ) . '">' . __( 'Previous', 'directorist-booking' ) . '</a>';
        }
        
        // Page numbers
        for ( $i = 1; $i <= $pages; $i++ ) {
            $active_class = ( $i === $current_page ) ? ' active' : '';
            $html        .= '<a href="#" class="page-link' . $active_class . '" data-page="' . $i . '">' . $i . '</a>';
        }
        
        // Next button
        if ( $current_page < $pages ) {
            $html .= '<a href="#" class="page-link next-page" data-page="' . ( $current_page + 1 ) . '">' . __( 'Next', 'directorist-booking' ) . '</a>';
        }
        
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Handle AJAX request for booking calendar search
     */
    public function handle_search_booking_events() {
        // Verify nonce for security
        if ( ! wp_verify_nonce( $_POST['nonce'], 'directorist_nonce' ) ) {
            wp_die( 'Security check failed' );
        }

        // Check if user is logged in
        if ( ! is_user_logged_in() ) {
            wp_send_json_error(
                [
                    'message' => __( 'You must be logged in to search bookings.', 'directorist-booking' )
                ]
            );
        }

        // Use the BookingController to handle the search
        $controller = new \Directorist_Booking\App\Http\Controllers\BookingController();
        $result     = $controller->search_booking_events();

        // Send JSON response
        if ( $result['status_code'] === 200 ) {
            wp_send_json_success( $result['data'] );
        } else {
            wp_send_json_error( $result['data'] );
        }
    }
}