<?php

namespace Directorist_Booking\App\Providers;

defined( "ABSPATH" ) || exit;

use Directorist_Booking\WpMVC\Contracts\Provider;

class GoogleCalendar implements Provider {

    const OAUTH_TOKEN_URL       = 'https://oauth2.googleapis.com/token';
    const CALENDAR_API_URL      = 'https://www.googleapis.com/calendar/v3';
    const CALENDAR_LIST_API_URL = 'https://www.googleapis.com/calendar/v3/users/me/calendarList';

    public function boot() {
        add_action( 'bdb_booking_status_changed', [$this, 'handle_booking_status_change'], 10, 3 );
        
        add_action( 'wp_ajax_bdb_save_google_credentials', [$this, 'save_credentials'] );
        add_action( 'wp_ajax_bdb_initiate_google_oauth', [$this, 'initiate_oauth'] );
        add_action( 'wp_ajax_bdb_google_oauth_callback', [$this, 'handle_oauth_callback'] );
        add_action( 'wp_ajax_bdb_disconnect_google_calendar', [$this, 'disconnect_calendar'] );
        add_action( 'wp_ajax_bdb_get_google_calendars', [$this, 'get_calendars'] );
        add_action( 'wp_ajax_bdb_save_calendar_settings', [$this, 'save_calendar_settings'] );
        add_action( 'wp_ajax_bdb_get_calendar_settings', [$this, 'get_calendar_settings'] );
        add_action( 'wp_ajax_bdb_get_connection_status', [$this, 'get_connection_status'] );
        add_action( 'wp_ajax_bdb_get_saved_credentials', [$this, 'get_saved_credentials'] );
        
        add_action( 'wp_enqueue_scripts', [$this, 'enqueue_scripts'] );
    }

    public function enqueue_scripts() {
        if ( ! is_user_logged_in() ) {
            return;
        }
    
        // Localize to front-admin (common)
        wp_localize_script(
            'bdb-front-admin', 'bdb_google_calendar', [
                'ajax_url'     => admin_url( 'admin-ajax.php' ),
                'nonce'        => wp_create_nonce( 'bdb_google_calendar_nonce' ),
                'redirect_uri' => admin_url( 'admin-ajax.php?action=bdb_google_oauth_callback' )
            ]
        );
    
        // Also localize to dashboard script
        wp_localize_script(
            'bdb-dashboard', 'bdb_google_calendar', [
                'ajax_url'     => admin_url( 'admin-ajax.php' ),
                'nonce'        => wp_create_nonce( 'bdb_google_calendar_nonce' ),
                'redirect_uri' => admin_url( 'admin-ajax.php?action=bdb_google_oauth_callback' )
            ]
        );
    }

    public function get_connection_status() {
        check_ajax_referer( 'bdb_google_calendar_nonce', 'nonce' );
        
        $user_id = get_current_user_id();
        if ( ! $user_id ) {
            wp_send_json_error( ['message' => __( 'User not logged in', 'directorist-booking' )] );
        }

        $access_token = get_user_meta( $user_id, 'bdb_google_access_token', true );
        $expires_at   = get_user_meta( $user_id, 'bdb_google_token_expires_at', true );
        $calendar_id  = get_user_meta( $user_id, 'bdb_google_calendar_id', true );
        
        $is_connected = ! empty( $access_token ) && ! empty( $expires_at ) && time() < $expires_at;

        wp_send_json_success(
            [
                'connected'       => $is_connected,
                'calendar_id'     => $calendar_id,
                'has_credentials' => ! empty( get_user_meta( $user_id, 'bdb_google_client_id', true ) )
            ]
        );
    }

    public function get_saved_credentials() {
        check_ajax_referer( 'bdb_google_calendar_nonce', 'nonce' );
        
        $user_id = get_current_user_id();
        if ( ! $user_id ) {
            wp_send_json_error( ['message' => __( 'User not logged in', 'directorist-booking' )] );
        }

        $client_id  = get_user_meta( $user_id, 'bdb_google_client_id', true );
        $has_secret = ! empty( get_user_meta( $user_id, 'bdb_google_client_secret', true ) );

        wp_send_json_success(
            [
                'client_id'  => $client_id,
                'has_secret' => $has_secret
            ]
        );
    }

    public function save_credentials() {
        check_ajax_referer( 'bdb_google_calendar_nonce', 'nonce' );
        
        $user_id = get_current_user_id();
        if ( ! $user_id ) {
            wp_send_json_error( ['message' => __( 'User not logged in', 'directorist-booking' )] );
        }

        $client_id     = isset( $_POST['client_id'] ) ? sanitize_text_field( wp_unslash( $_POST['client_id'] ) ) : '';
        $client_secret = isset( $_POST['client_secret'] ) ? sanitize_text_field( wp_unslash( $_POST['client_secret'] ) ) : '';

        if ( empty( $client_id ) || empty( $client_secret ) ) {
            wp_send_json_error( ['message' => __( 'Client ID and Secret are required', 'directorist-booking' )] );
        }

        update_user_meta( $user_id, 'bdb_google_client_id', $client_id );
        update_user_meta( $user_id, 'bdb_google_client_secret', $client_secret );

        wp_send_json_success(
            [
                'message' => __( 'Credentials saved successfully', 'directorist-booking' )
            ]
        );
    }

    public function initiate_oauth() {
        check_ajax_referer( 'bdb_google_calendar_nonce', 'nonce' );
        
        $user_id = get_current_user_id();
        if ( ! $user_id ) {
            wp_send_json_error( ['message' => __( 'User not logged in', 'directorist-booking' )] );
        }

        $client_id = get_user_meta( $user_id, 'bdb_google_client_id', true );
        
        if ( empty( $client_id ) ) {
            wp_send_json_error( ['message' => __( 'Please save credentials first', 'directorist-booking' )] );
        }

        $state = wp_create_nonce( 'bdb_google_oauth_' . $user_id );
        update_user_meta( $user_id, 'bdb_google_oauth_state', $state );

        $redirect_uri = admin_url( 'admin-ajax.php?action=bdb_google_oauth_callback' );

        $auth_url = 'https://accounts.google.com/o/oauth2/v2/auth?' . http_build_query(
            [
                'client_id'     => $client_id,
                'redirect_uri'  => $redirect_uri,
                'response_type' => 'code',
                'scope'         => 'https://www.googleapis.com/auth/calendar.events https://www.googleapis.com/auth/calendar.readonly',
                'access_type'   => 'offline',
                'prompt'        => 'consent',
                'state'         => $state,
            ]
        );

        wp_send_json_success( ['auth_url' => $auth_url] );
    }

    public function handle_oauth_callback() {
        if ( ! isset( $_GET['code'] ) || ! isset( $_GET['state'] ) ) {
            wp_die( __( 'Invalid OAuth callback', 'directorist-booking' ) );
        }

        $code  = sanitize_text_field( wp_unslash( $_GET['code'] ) );
        $state = sanitize_text_field( wp_unslash( $_GET['state'] ) );

        $users = get_users(
            [
                'meta_key'   => 'bdb_google_oauth_state',
                'meta_value' => $state,
                'number'     => 1
            ]
        );

        if ( empty( $users ) ) {
            wp_die( __( 'Invalid state parameter', 'directorist-booking' ) );
        }

        $user_id = $users[0]->ID;
        
        if ( ! wp_verify_nonce( $state, 'bdb_google_oauth_' . $user_id ) ) {
            wp_die( __( 'Invalid state parameter', 'directorist-booking' ) );
        }

        $client_id     = get_user_meta( $user_id, 'bdb_google_client_id', true );
        $client_secret = get_user_meta( $user_id, 'bdb_google_client_secret', true );
        $redirect_uri  = admin_url( 'admin-ajax.php?action=bdb_google_oauth_callback' );

        $response = wp_remote_post(
            self::OAUTH_TOKEN_URL, [
                'body' => [
                    'code'          => $code,
                    'client_id'     => $client_id,
                    'client_secret' => $client_secret,
                    'redirect_uri'  => $redirect_uri,
                    'grant_type'    => 'authorization_code',
                ],
            ]
        );

        if ( is_wp_error( $response ) ) {
            wp_die( __( 'Failed to exchange code for token', 'directorist-booking' ) );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        if ( $response_code !== 200 ) {
            wp_die( __( 'Failed to get access token - HTTP ' . $response_code, 'directorist-booking' ) );
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['access_token'] ) ) {
            update_user_meta( $user_id, 'bdb_google_access_token', $body['access_token'] );
            update_user_meta( $user_id, 'bdb_google_token_expires_at', time() + ( $body['expires_in'] ?? 3600 ) );
            
            if ( isset( $body['refresh_token'] ) ) {
                update_user_meta( $user_id, 'bdb_google_refresh_token', $body['refresh_token'] );
            }
            
            delete_user_meta( $user_id, 'bdb_google_oauth_state' );

            $this->render_oauth_success_response();
        }

        wp_die( __( 'Failed to save tokens', 'directorist-booking' ) );
    }

    private function render_oauth_success_response() {
        $redirect_url = home_url( '/dashboard/?tab=booking_calender&google_calendar=connected' );

        nocache_headers();
        status_header( 200 );
        header( 'Content-Type: text/html; charset=' . get_option( 'blog_charset' ), true );

        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="<?php echo esc_attr( get_option( 'blog_charset', 'utf-8' ) ); ?>">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <title><?php esc_html_e( 'Google Calendar Connected', 'directorist-booking' ); ?></title>
            <script>
                (function () {
                    var redirectUrl = <?php echo wp_json_encode( $redirect_url ); ?>;

                    function getOrigin() {
                        if (window.location.origin) {
                            return window.location.origin;
                        }
                        return window.location.protocol + '//' + window.location.host;
                    }

                    function notifyParent() {
                        if (window.opener && !window.opener.closed) {
                            try {
                                window.opener.postMessage('google_calendar_connected', getOrigin());
                            } catch (err) {
                                try {
                                    window.opener.postMessage('google_calendar_connected', '*');
                                } catch (err2) {}
                            }
                        }
                    }

                    notifyParent();

                    setTimeout(function () {
                        try {
                            window.close();
                        } catch (err) {}

                        if (!window.closed) {
                            window.location.replace(redirectUrl);
                        }
                    }, 500);
                })();
            </script>
            <style>
                body {
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
                    margin: 0;
                    padding: 40px 16px;
                    text-align: center;
                    color: #1a1a1a;
                    background-color: #f7f7f7;
                }
                .gc-success {
                    max-width: 420px;
                    margin: 0 auto;
                    background: #ffffff;
                    border-radius: 12px;
                    padding: 32px 24px;
                    box-shadow: 0 12px 30px rgba(26, 26, 26, 0.12);
                }
                .gc-success__title {
                    font-size: 20px;
                    font-weight: 600;
                    margin-bottom: 12px;
                }
                .gc-success__subtitle {
                    font-size: 14px;
                    color: #4a5568;
                }
            </style>
        </head>
        <body>
            <div class="gc-success" role="status" aria-live="polite">
                <div class="gc-success__title"><?php esc_html_e( 'Google Calendar connected', 'directorist-booking' ); ?></div>
                <div class="gc-success__subtitle"><?php esc_html_e( 'This window will close automatically. You can close it manually if it stays open.', 'directorist-booking' ); ?></div>
            </div>
        </body>
        </html>
        <?php
        exit;
    }

    private function get_access_token( $user_id ) {
        $access_token = get_user_meta( $user_id, 'bdb_google_access_token', true );
        $expires_at   = get_user_meta( $user_id, 'bdb_google_token_expires_at', true );

        if ( $access_token && $expires_at && time() < $expires_at ) {
            return $access_token;
        }

        $refresh_token = get_user_meta( $user_id, 'bdb_google_refresh_token', true );
        if ( ! $refresh_token ) {
            return null;
        }

        $client_id     = get_user_meta( $user_id, 'bdb_google_client_id', true );
        $client_secret = get_user_meta( $user_id, 'bdb_google_client_secret', true );

        $response = wp_remote_post(
            self::OAUTH_TOKEN_URL, [
                'body' => [
                    'client_id'     => $client_id,
                    'client_secret' => $client_secret,
                    'refresh_token' => $refresh_token,
                    'grant_type'    => 'refresh_token',
                ],
            ]
        );

        if ( is_wp_error( $response ) ) {
            error_log( 'Google Calendar: Token refresh failed - ' . $response->get_error_message() );
            return null;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        if ( $response_code !== 200 ) {
            error_log( 'Google Calendar: Token refresh failed - HTTP ' . $response_code );
            return null;
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['access_token'] ) ) {
            $access_token = $body['access_token'];
            $expires_in   = $body['expires_in'] ?? 3600;

            update_user_meta( $user_id, 'bdb_google_access_token', $access_token );
            update_user_meta( $user_id, 'bdb_google_token_expires_at', time() + $expires_in );

            return $access_token;
        }

        return null;
    }

    public function get_calendars() {
        check_ajax_referer( 'bdb_google_calendar_nonce', 'nonce' );
        
        $user_id = get_current_user_id();
        if ( ! $user_id ) {
            wp_send_json_error( ['message' => __( 'User not logged in', 'directorist-booking' )] );
        }

        $access_token = $this->get_access_token( $user_id );
        if ( ! $access_token ) {
            wp_send_json_error( ['message' => __( 'Not connected to Google Calendar', 'directorist-booking' )] );
        }

        $response = wp_remote_get(
            self::CALENDAR_LIST_API_URL, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $access_token,
                ],
                'timeout' => 30,
            ]
        );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( ['message' => __( 'Failed to fetch calendars', 'directorist-booking' )] );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        if ( $response_code !== 200 ) {
            wp_send_json_error( ['message' => __( 'Failed to fetch calendars', 'directorist-booking' )] );
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        
        if ( isset( $body['items'] ) ) {
            wp_send_json_success( ['calendars' => $body['items']] );
        }

        wp_send_json_error( ['message' => __( 'No calendars found', 'directorist-booking' )] );
    }

    public function save_calendar_settings() {
        check_ajax_referer( 'bdb_google_calendar_nonce', 'nonce' );
        
        $user_id = get_current_user_id();
        if ( ! $user_id ) {
            wp_send_json_error( ['message' => __( 'User not logged in', 'directorist-booking' )] );
        }

        $calendar_id        = isset( $_POST['calendar_id'] ) ? sanitize_text_field( wp_unslash( $_POST['calendar_id'] ) ) : 'primary';
        $conflict_calendars = isset( $_POST['conflict_calendars'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['conflict_calendars'] ) ) : [];
        update_user_meta( $user_id, 'bdb_google_calendar_id', $calendar_id );
        update_user_meta( $user_id, 'bdb_google_conflict_calendars', $conflict_calendars );

        wp_send_json_success(
            [
                'message' => __( 'Settings saved successfully', 'directorist-booking' )
            ]
        );
    }

    public function get_calendar_settings() {
        check_ajax_referer( 'bdb_google_calendar_nonce', 'nonce' );
        
        $user_id = get_current_user_id();
        if ( ! $user_id ) {
            wp_send_json_error( ['message' => __( 'User not logged in', 'directorist-booking' )] );
        }

        $calendar_id        = get_user_meta( $user_id, 'bdb_google_calendar_id', true );
        $conflict_calendars = get_user_meta( $user_id, 'bdb_google_conflict_calendars', true );

        wp_send_json_success(
            [
                'calendar_id'        => $calendar_id ?: 'primary',
                'conflict_calendars' => $conflict_calendars ?: []
            ]
        );
    }

    private function calendars_have_conflicts( $access_token, $start, $end, $conflict_calendars, $timezone ) {
        $conflict_calendars = array_filter( (array) $conflict_calendars );
        if ( empty( $conflict_calendars ) ) {
            return false;
        }
    
        if ( isset( $start['dateTime'] ) ) {
            $time_min = $start['dateTime'];
            $time_max = $end['dateTime'] ?? $start['dateTime'];
        } else {
            $time_min = wp_date( 'c', strtotime( $start['date'] . ' 00:00:00 ' . $timezone ) );
            $time_max = wp_date( 'c', strtotime( $end['date'] . ' 23:59:59 ' . $timezone ) );
        }
    
        $response = wp_remote_post(
            self::CALENDAR_API_URL . '/freeBusy',
            [
                'headers' => [
                    'Authorization' => 'Bearer ' . $access_token,
                    'Content-Type'  => 'application/json',
                ],
                'body'    => wp_json_encode(
                    [
                        'timeMin'  => $time_min,
                        'timeMax'  => $time_max,
                        'timeZone' => $timezone,
                        'items'    => array_map(
                            static function ( $calendar_id ) {
                                return [ 'id' => $calendar_id ];
                            },
                            $conflict_calendars
                        ),
                    ]
                ),
                'timeout' => 15,
            ]
        );
    
        if ( is_wp_error( $response ) ) {
            error_log( 'Google Calendar: freeBusy request failed - ' . $response->get_error_message() );
            return false;
        }
    
        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( empty( $body['calendars'] ) ) {
            return false;
        }
    
        foreach ( $body['calendars'] as $calendar ) {
            if ( ! empty( $calendar['busy'] ) ) {
                return true;
            }
        }
    
        return false;
    }

    public function handle_booking_status_change( $booking_id, $new_status, $old_status = null ) {
        global $wpdb;

        $booking = $wpdb->get_row(
            $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}directorist_booking WHERE ID = %d", $booking_id ),
            ARRAY_A
        );

        if ( ! $booking ) {
            return;
        }

        $listing_owner_id = get_post_field( 'post_author', $booking['listing_id'] );
        if ( ! $listing_owner_id ) {
            return;
        }

        $access_token = $this->get_access_token( $listing_owner_id );
        if ( ! $access_token ) {
            return;
        }

        if ( $new_status === 'confirmed' ) {
            $this->add_to_calendar( $booking, $listing_owner_id );
        } elseif ( $new_status === 'cancelled' && ! empty( $booking['google_calendar_event_id'] ) ) {
            $this->delete_from_calendar( $booking['google_calendar_event_id'], $listing_owner_id );
        }
    }

    private function add_to_calendar( $booking, $user_id ) {
        $access_token = $this->get_access_token( $user_id );
        if ( ! $access_token ) {
            return;
        }

        global $wpdb;

        $lock_token = '__gc_sync_' . wp_generate_uuid4();
        $updated    = $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$wpdb->prefix}directorist_booking
                SET google_calendar_event_id = %s
                WHERE ID = %d
                AND (google_calendar_event_id IS NULL OR google_calendar_event_id = '')",
                $lock_token,
                $booking['ID']
            )
        );

        if ( 0 === $updated ) {
            return; // another process already claimed this booking
        }

        $booking_type = get_post_meta( $booking['listing_id'], '_bdb_booking_type', true );

        if ( $booking_type === 'event' ) {
            return;
        }

        $listing_title = get_the_title( $booking['listing_id'] );
        $comment_data  = json_decode( $booking['comment'], true );
        $user_email    = $comment_data['email'] ?? '';
        $user_name     = $comment_data['first_name'] ?? '';
        $user_phone    = $comment_data['phone'] ?? '';

        $description = sprintf(
            "Booking by: %s\nEmail: %s\nPhone: %s\nPrice: %s",
            $user_name,
            $user_email,
            $user_phone,
            atbdp_format_payment_amount( $booking['price'] )
        );

        $local_timezone = wp_timezone_string();

        if ( $booking_type === 'rent' ) {
            $event_data = [
                'summary'     => $listing_title . ' - Booking',
                'description' => $description,
                'start'       => [
                    'date' => date( 'Y-m-d', strtotime( $booking['date_start'] ) ),
                ],
                'end'         => [
                    'date' => date( 'Y-m-d', strtotime( $booking['date_end'] . ' +1 day' ) ),
                ],
                'attendees'   => [
                    ['email' => $user_email, 'displayName' => $user_name]
                ],
            ];
        } else {
            $date_start = $booking['date_start'];
            $date_end   = $booking['date_end'];
            $has_time   = ( strlen( $date_start ) > 10 && strpos( $date_start, ':' ) !== false );

            if ( ! $has_time && isset( $comment_data['hour'] ) ) {
                $hour_range = $comment_data['hour'];
                if ( strpos( $hour_range, ' - ' ) !== false ) {
                    $times      = explode( ' - ', $hour_range );
                    $start_time = date( 'H:i:s', strtotime( $times[0] ) );
                    $end_time   = date( 'H:i:s', strtotime( $times[1] ) );
                    $date_start = $booking['date_start'] . ' ' . $start_time;
                    $date_end   = $booking['date_end'] . ' ' . $end_time;
                    $has_time   = true;
                }
            }

            if ( $has_time ) {
                preg_match( '/(\d{4}-\d{2}-\d{2})\s+(\d{2}:\d{2}:\d{2})/', $date_start, $start_matches );
                preg_match( '/(\d{4}-\d{2}-\d{2})\s+(\d{2}:\d{2}:\d{2})/', $date_end, $end_matches );

                if ( ! empty( $start_matches ) && ! empty( $end_matches ) ) {
                    $event_data = [
                        'summary'     => $listing_title . ' - Booking',
                        'description' => $description,
                        'start'       => [
                            'dateTime' => $start_matches[1] . 'T' . $start_matches[2],
                            'timeZone' => $local_timezone,
                        ],
                        'end'         => [
                            'dateTime' => $end_matches[1] . 'T' . $end_matches[2],
                            'timeZone' => $local_timezone,
                        ],
                        'attendees'   => [
                            ['email' => $user_email, 'displayName' => $user_name]
                        ],
                    ];
                } else {
                    $has_time = false;
                }
            }
            
            if ( ! $has_time ) {
                $event_data = [
                    'summary'     => $listing_title . ' - Booking',
                    'description' => $description,
                    'start'       => [
                        'date' => date( 'Y-m-d', strtotime( $booking['date_start'] ) ),
                    ],
                    'end'         => [
                        'date' => date( 'Y-m-d', strtotime( $booking['date_end'] . ' +1 day' ) ),
                    ],
                    'attendees'   => [
                        ['email' => $user_email, 'displayName' => $user_name]
                    ],
                ];
            }
        }

        $conflict_calendars = get_user_meta( $user_id, 'bdb_google_conflict_calendars', true );

        if ( $this->calendars_have_conflicts( $access_token, $event_data['start'], $event_data['end'], $conflict_calendars, $local_timezone ) ) {
            $wpdb->update(
                $wpdb->prefix . 'directorist_booking',
                ['google_calendar_event_id' => null],
                ['ID' => $booking['ID']],
                ['%s'],
                ['%d']
            );
            error_log( 'Google Calendar: conflict detected, booking skipped for ID ' . $booking['ID'] );
            return;
        }

        $calendar_id = rawurlencode( get_user_meta( $user_id, 'bdb_google_calendar_id', true ) ?: 'primary' );
        $url         = self::CALENDAR_API_URL . "/calendars/{$calendar_id}/events";
        $response    = wp_remote_post(
            $url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $access_token,
                    'Content-Type'  => 'application/json',
                ],
                'body'    => wp_json_encode( $event_data ),
                'timeout' => 30,
            ]
        );

        if ( is_wp_error( $response ) ) {
            $wpdb->update(
                $wpdb->prefix . 'directorist_booking',
                ['google_calendar_event_id' => null],
                ['ID' => $booking['ID']],
                ['%s'],
                ['%d']
            );
            error_log( 'Google Calendar: Failed to create event - ' . $response->get_error_message() );
            return;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        if ( $response_code !== 200 && $response_code !== 201 ) {
            $wpdb->update(
                $wpdb->prefix . 'directorist_booking',
                ['google_calendar_event_id' => null],
                ['ID' => $booking['ID']],
                ['%s'],
                ['%d']
            );
            return;
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['id'] ) ) {
            $wpdb->update(
                $wpdb->prefix . 'directorist_booking',
                ['google_calendar_event_id' => $body['id']],
                ['ID' => $booking['ID']],
                ['%s'],
                ['%d']
            );
        }
    }

    private function delete_from_calendar( $event_id, $user_id ) {
        $access_token = $this->get_access_token( $user_id );
        if ( ! $access_token ) {
            return;
        }

        $calendar_id = rawurlencode( get_user_meta( $user_id, 'bdb_google_calendar_id', true ) ?: 'primary' );
        $event_id    = rawurlencode( $event_id );
        $url         = self::CALENDAR_API_URL . "/calendars/{$calendar_id}/events/{$event_id}";

        wp_remote_request(
            $url, [
                'method'  => 'DELETE',
                'headers' => [
                    'Authorization' => 'Bearer ' . $access_token,
                ],
                'timeout' => 30,
            ]
        );
    }

    public function disconnect_calendar() {
        check_ajax_referer( 'bdb_google_calendar_nonce', 'nonce' );
        
        $user_id = get_current_user_id();
        if ( ! $user_id ) {
            wp_send_json_error( ['message' => __( 'User not logged in', 'directorist-booking' )] );
        }

        delete_user_meta( $user_id, 'bdb_google_access_token' );
        delete_user_meta( $user_id, 'bdb_google_refresh_token' );
        delete_user_meta( $user_id, 'bdb_google_token_expires_at' );
        delete_user_meta( $user_id, 'bdb_google_calendar_id' );
        delete_user_meta( $user_id, 'bdb_google_client_id' );
        delete_user_meta( $user_id, 'bdb_google_client_secret' );

        wp_send_json_success( ['message' => __( 'Disconnected successfully', 'directorist-booking' )] );
    }
}