(function ($) {
  // Update section classes based on checkbox states
  function initializeBHSections() {
    document
      .querySelectorAll(".directorist-bh-dayzone__single")
      .forEach((dayZone) => {
        const switchCheckbox = dayZone.querySelector(
          '.directorist-bh-dayzone__single__switch input[type="checkbox"]'
        );
        const hourSelectionCheckbox = dayZone.querySelector(
          '.directorist-bh-dayzone__single__hour-selection input[type="checkbox"]'
        );
        if (switchCheckbox)
          dayZone.classList.toggle(
            "directorist-enable-hour-time-selection",
            switchCheckbox.checked
          );

        // ✅ Trigger a change event if checked
        if (switchCheckbox.checked) {
          switchCheckbox.dispatchEvent(new Event("change", { bubbles: true }));
        }

        if (hourSelectionCheckbox)
          dayZone.classList.toggle(
            "directorist-full-time",
            hourSelectionCheckbox.checked
          );
      });
  }

  // Toggle visibility of BH sections based on status and type
  function toggleBusinessHourSections() {
    const isEnabled = $("#enable_directorist_bh_listing").is(":checked");
    const isCustomHour = $("input[name='directorist_bh_option']").is(
      ":checked"
    );

    // BH Type Toggle
    $(".directorist-bh-section--type")
      .toggleClass("directorist-bh-show", isEnabled)
      .toggleClass("directorist-bh-hide", !isEnabled);

    // BH Timezone Toggle
    $(".directorist-bh-section--timezone")
      .toggleClass("directorist-bh-show", isEnabled)
      .toggleClass("directorist-bh-hide", !isEnabled);

    // BH Selection Toggle
    $(".directorist-bh-section--selection")
      .toggleClass("directorist-bh-show", isEnabled && isCustomHour)
      .toggleClass("directorist-bh-hide", !(isEnabled && isCustomHour));
  }

  // Handle Next Day Label Update
  let isUpdatingNextDay = false;

  function handleBHNextDayLabel($startSelect, $endSelect) {
    if (isUpdatingNextDay) return;

    $selected = $endSelect.find("option:selected");

    const startTime = $startSelect.val();
    const endTime = $endSelect.val();

    const timeToMinutes = (t) => {
      if (!t) return 0;
      const [h, m] = t.split(":").map(Number);
      return h * 60 + m;
    };

    const startTimeCount = timeToMinutes(startTime);
    const endTimeCount = timeToMinutes(endTime);

    if (!startTime || !endTime) return;

    // Check if crossing midnight in two scenarios:
    // 1. PM to AM crossing (e.g., 22:00 to 02:00)
    const isCrossingMidnight = startTimeCount >= 720 && endTimeCount < 720;

    // 2. Early morning crossing (e.g., 01:15 to 00:30)
    const isCrossingToNextDay = endTimeCount < startTimeCount;

    const dayMatch = $endSelect.attr("name")?.match(/\[([a-z]+)\]/i);
    const day = dayMatch?.[1] || "";

    const days = [
      "sunday",
      "monday",
      "tuesday",
      "wednesday",
      "thursday",
      "friday",
      "saturday",
    ];
    const nextDay = days[(days.indexOf(day) + 1) % 7];
    // const suffix = `(${nextDay.slice(0, 3)})`;
    const suffix = nextDay;

    if ((isCrossingMidnight || isCrossingToNextDay) && nextDay) {
      if (!$selected.data("original-text")) {
        $selected.data("original-text", $selected.text());
      }

      const originalText = $selected.data("original-text");
      // const updatedText = `${originalText} (${suffix})`;
      // $selected.text(updatedText);
      $selected.attr("data-next-day", "true");
      $selected.attr("data-next-day-suffix", suffix);

      isUpdatingNextDay = true;
      $endSelect.trigger("change"); // Now won't recurse
      isUpdatingNextDay = false;

      setTimeout(() => {
        const rendered = $endSelect
          .next(".select2")
          .find(".select2-selection__rendered");

        const mainText = originalText;
        const spanHtml = `<span class="next-day-suffix">${suffix}</span>`;
        rendered.html(`${mainText} ${spanHtml}`);
      }, 0);
    } else {
      const originalText = $selected.data("original-text") || $selected.text();
      $selected.text(originalText);
      $selected.removeAttr("data-next-day data-next-day-suffix");

      setTimeout(() => {
        const rendered = $endSelect
          .next(".select2")
          .find(".select2-selection__rendered");

        rendered.html(originalText);
      }, 0);
    }
  }

  // Initialize next day label update for all existing items
  function initializeBHLabelUpdate() {
    $(".directorist-bh-dayzone__single__choice__item").each(function () {
      const $container = $(this);
      const $startSelect = $container.find(
        ".directorist-selects--start select"
      );
      const $endSelect = $container.find(".directorist-selects--close select");

      // Update next day label
      handleBHNextDayLabel($startSelect, $endSelect);
    });
  }

  // Handle start time change event
  function handleStartTimeChange(e) {
    const $startSelect = $(e.currentTarget);
    const $endSelect = $startSelect
      .closest(".directorist-bh-dayzone__single__choice__item")
      .find(".directorist-selects--close select");

    handleBHNextDayLabel($startSelect, $endSelect);
  }

  // Handle end time change event
  function handleEndTimeChange(e) {
    const $endSelect = $(e.currentTarget);
    const $startSelect = $endSelect
      .closest(".directorist-bh-dayzone__single__choice__item")
      .find(".directorist-selects--start select");

    handleBHNextDayLabel($startSelect, $endSelect);
  }

  // Toggle 24/7 hour selection
  $("body").on(
    "change",
    ".directorist-bh-dayzone__single__hour-selection .directorist-checkbox input",
    function () {
      if ($(this).prop("checked")) {
        $(this)
          .closest(".directorist-bh-dayzone__single__hour-selection")
          .siblings(".directorist-bh-dayzone__single__choice")
          .parent(".directorist-bh-dayzone__single")
          .addClass("directorist-full-time");
      } else {
        $(this)
          .closest(".directorist-bh-dayzone__single__hour-selection")
          .siblings(".directorist-bh-dayzone__single__choice")
          .parent(".directorist-bh-dayzone__single")
          .removeClass("directorist-full-time");
      }
    }
  );

  // Each Day Switch
  $("body").on("change", ".directorist-switch-input", function () {
    const $wrapper = $(this).closest(".directorist-bh-dayzone__single");

    if (this.checked) {
      $wrapper.addClass("directorist-enable-hour-time-selection");

      $($wrapper)
        .find("select")
        .each(function () {
          const $select = $(this);
          if (!$select.hasClass("select2-hidden-accessible")) {
            const placeholder =
              $select.closest(".directorist-selects--start").length > 0
                ? "Select Starting Time"
                : $select.closest(".directorist-selects--close").length > 0
                ? "Select Closing Time"
                : "Select an option";

            $select.select2({
              placeholder,
              allowClear: true,
              width: "100%",
              dropdownCssClass: "dbh-select2-dropdown",
              containerCssClass: "dbh-select2-container",
            });
          }
        });
    } else {
      $wrapper.removeClass("directorist-enable-hour-time-selection");

      // ✅ Check if the wrapper has the "directorist-full-time" class
      if ($wrapper.hasClass("directorist-full-time")) {
        $wrapper.removeClass("directorist-full-time");

        // ✅ Uncheck all related hour-selection checkboxes
        $wrapper
          .find(
            '.directorist-bh-dayzone__single__hour-selection input[type="checkbox"]'
          )
          .prop("checked", false);
      }
    }
  });

  // Add/Clone single choice item
  $("body").on(
    "click",
    ".directorist-bh-dayzone__single__choice__add",
    function (e) {
      e.preventDefault();

      const $choiceWrapper = $(this).closest(
        ".directorist-bh-dayzone__single__choice"
      );
      const $originalItem = $choiceWrapper
        .find(".directorist-bh-dayzone__single__choice__item")
        .last();

      const $originalClose = $originalItem.find(
        ".directorist-selects--close select"
      );

      // Clone and clean
      const $clonedItem = $originalItem
        .clone()
        .removeClass("directorist-not-removable");

      $clonedItem.find("select").each(function () {
        const $select = $(this);

        if ($select.hasClass("select2-hidden-accessible")) {
          $select.select2("destroy");
        }

        $select
          .removeClass("select2-hidden-accessible")
          .removeAttr("data-select2-id tabindex aria-hidden")
          .siblings(".select2")
          .remove();

        $select.closest(".directorist-selects").removeAttr("data-select2-id");
        $select
          .closest(".directorist-bh-dayzone__single__choice__slot")
          .removeAttr("data-select2-id");

        $select.val(null);
      });

      $clonedItem.removeAttr("data-select2-id");
      $choiceWrapper.removeAttr("data-select2-id");

      // Insert clone
      $originalItem.after($clonedItem);

      // Reindex names
      $choiceWrapper
        .find(".directorist-bh-dayzone__single__choice__item")
        .each(function (index) {
          const $startSelect = $(this).find(
            ".directorist-selects--start select"
          );
          const $closeSelect = $(this).find(
            ".directorist-selects--close select"
          );

          if ($startSelect.length) {
            $startSelect.attr(
              "name",
              $startSelect
                .attr("name")
                .replace(/\[start]\[\d+]/, `[start][${index}]`)
            );
          }

          if ($closeSelect.length) {
            $closeSelect.attr(
              "name",
              $closeSelect
                .attr("name")
                .replace(/\[close]\[\d+]/, `[close][${index}]`)
            );
          }
        });

      $choiceWrapper
        .find(".directorist-bh-dayzone__single__choice__item")
        .removeClass("directorist-not-removable");

      // Re-trigger select logic on original close select to re-render label
      setTimeout(() => {
        $originalClose.trigger("change").trigger("select2:select");
      }, 0);
    }
  );

  // Remove single choice item
  $("body").on(
    "click",
    ".directorist-bh-dayzone__single__choice__remove",
    function (e) {
      e.preventDefault();

      const $choiceWrapper = $(this).closest(
        ".directorist-bh-dayzone__single__choice"
      );

      // Remove the clicked item only if more than 1 item exists
      const $items = $choiceWrapper.find(
        ".directorist-bh-dayzone__single__choice__item"
      );
      if ($items.length > 1) {
        $(this)
          .closest(".directorist-bh-dayzone__single__choice__item")
          .remove();
      }

      // Recalculate remaining items
      const $remainingItems = $choiceWrapper.find(
        ".directorist-bh-dayzone__single__choice__item"
      );
      $remainingItems.toggleClass(
        "directorist-not-removable",
        $remainingItems.length <= 1
      );
    }
  );

  // Event: Business Hour Enable/Disable
  $("body").on(
    "change",
    "#enable_directorist_bh_listing",
    toggleBusinessHourSections
  );

  // Event: Business Hour Type Change
  $("body").on(
    "change",
    'input[type="checkbox"].directorist-247-alternative',
    function () {
      // Uncheck all other checkboxes in the same group
      $('input[type="checkbox"].directorist-247-alternative')
        .not(this)
        .prop("checked", false);

      // Handle "always" and "custom" logic
      switch (this.value) {
        case "always":
          $(this)
            .closest(".directorist-bh-wrap")
            .find(".directorist-bh-section--selection")
            .removeClass("directorist-bh-show")
            .addClass("directorist-bh-hide");

          $(this)
            .closest(".directorist-bh-section")
            .addClass("bh-always-active");

          $('input[name="directorist_bh_option"]').prop("checked", false);

          break;

        case "custom":
          $(this)
            .closest(".directorist-bh-wrap")
            .find(".directorist-bh-section--selection")
            .removeClass("directorist-bh-hide")
            .addClass("directorist-bh-show");

          $(this)
            .closest(".directorist-bh-section")
            .removeClass("bh-always-active");

          $('input[name="enable247hour"]').prop("checked", false);

          break;
      }
    }
  );

  // Event: Update next day label on start time change
  $("body")
    .on("change", ".directorist-selects--start select", handleStartTimeChange)
    .on(
      "select2:select",
      ".directorist-selects--start select",
      handleStartTimeChange
    );

  // Event: Update next day label on end time change
  $("body")
    .on("change", ".directorist-selects--close select", handleEndTimeChange)
    .on(
      "select2:select",
      ".directorist-selects--close select",
      handleEndTimeChange
    );

  // Initial Load
  $(window).on("load", function () {
    // Update classes on initial load
    const observer = new MutationObserver(initializeBHSections);
    observer.observe(document.body, { childList: true, subtree: true });

    initializeBHSections();

    // Delay a bit to ensure Select2 init is complete
    setTimeout(() => {
      // Initialize Select2 on all selects
      initializeBHLabelUpdate();

      // Initial timezone dropdown
      if ($("#dbh-select-timezone").length) {
        $("#dbh-select-timezone").select2({
          placeholder: "Select Timezone",
          dropdownCssClass: "dbh-select2-dropdown",
          containerCssClass: "dbh-select2-container",
        });
      }
    }, 1000); // 1000ms delay; adjust as needed
  });

  /**
   * Business Hours Copy Dropdown Functionality
   * Handles the copy dropdown for copying business hours from one day to others
   */
  function initializeCopyDropdown() {
    // Toggle dropdown visibility on copy button click
    $("body").on(
      "click",
      ".directorist-bh-dayzone__single__action__btn--copy",
      function (e) {
        e.preventDefault();
        e.stopPropagation();

        const $dropdown = $(".directorist-bh-dayzone__copy-dropdown");
        const isVisible = $dropdown.hasClass(
          "directorist-bh-dayzone__copy-dropdown--show"
        );

        // Toggle current dropdown
        if (!isVisible) {
          this.classList.add("active");
          // Close any other open dropdowns first
          $(".directorist-bh-dayzone__copy-dropdown").removeClass(
            "directorist-bh-dayzone__copy-dropdown--show"
          );

          // Store current button reference for scroll updates
          currentButton = $(this);
          currentDay = $(this).closest(".directorist-bh-dayzone__single");
          const currentDayValue = currentDay.data("day");

          // Disable the current day's checkbox in dropdown
          disableCurrentDayInDropdown($dropdown, currentDayValue);

          // Position dropdown relative to the clicked button with a small delay
          setTimeout(() => {
            positionDropdownRelativeToButton($dropdown, $(this));
          }, 10);

          $dropdown.addClass("directorist-bh-dayzone__copy-dropdown--show");
          // Initialize button states when dropdown opens
          updateButtonStates($dropdown);
        } else {
          // Close dropdown when clicked again (toggle behavior)
          closeDropdown($dropdown);
        }
      }
    );

    // Close dropdown when clicking outside
    $(document).on("click", function (e) {
      if (
        !$(e.target).closest(
          ".directorist-bh-dayzone__copy-dropdown, .directorist-bh-dayzone__single__action__btn--copy"
        ).length
      ) {
        const $dropdown = $(".directorist-bh-dayzone__copy-dropdown");
        closeDropdown($dropdown);
      }
    });

    // Update dropdown position on scroll to keep it aligned with button
    let currentButton = null;
    $(window).on("scroll resize", function () {
      const $dropdown = $(".directorist-bh-dayzone__copy-dropdown");
      if (
        $dropdown.hasClass("directorist-bh-dayzone__copy-dropdown--show") &&
        currentButton
      ) {
        positionDropdownRelativeToButton($dropdown, currentButton);
      }
    });

    // Select All / Unselect All functionality
    $("body").on("change", "#directorist-bh-copy-select-all", function () {
      const $dropdown = $(".directorist-bh-dayzone__copy-dropdown");

      // Get enabled checkboxes
      const $enabledCheckboxes = $dropdown
        .find(".directorist-bh-dayzone__copy-dropdown__checkbox")
        .not(this)
        .filter(function () {
          const $item = $(this).closest(
            ".directorist-bh-dayzone__copy-dropdown__item"
          );
          return !$item.hasClass("disabled");
        });

      const checkedCount = $enabledCheckboxes.filter(":checked").length;
      const totalCount = $enabledCheckboxes.length;
      const allSelected = checkedCount === totalCount;

      // Determine action based on current selection state
      // If all are selected, unselect all; otherwise, select all
      const shouldSelect = !allSelected;

      // Check/uncheck all day checkboxes (excluding disabled items)
      $enabledCheckboxes.each(function () {
        $(this).prop("checked", shouldSelect);
      });

      // Update Select All checkbox state to match the action
      $(this).prop("checked", shouldSelect);

      // Update button states
      updateButtonStates($dropdown);

      // Update Select All state after changing individual checkboxes
      updateSelectAllState($dropdown);
    });

    // Individual checkbox change - update Select All state and button states
    $("body").on(
      "change",
      ".directorist-bh-dayzone__copy-dropdown__checkbox:not(#directorist-bh-copy-select-all)",
      function () {
        const $dropdown = $(".directorist-bh-dayzone__copy-dropdown");
        updateSelectAllState($dropdown);
        updateButtonStates($dropdown);
      }
    );

    // Reset All functionality
    $("body").on(
      "click",
      ".directorist-bh-dayzone__copy-dropdown__btn--reset",
      function (e) {
        e.preventDefault();
        const $dropdown = $(".directorist-bh-dayzone__copy-dropdown");

        // Uncheck all checkboxes (excluding disabled items)
        $dropdown
          .find(".directorist-bh-dayzone__copy-dropdown__checkbox")
          .each(function () {
            const $item = $(this).closest(
              ".directorist-bh-dayzone__copy-dropdown__item"
            );
            // Only affect checkboxes that are not in disabled items
            if (!$item.hasClass("disabled")) {
              $(this).prop("checked", false);
            }
          });

        // Update button states
        updateButtonStates($dropdown);
      }
    );

    // Apply functionality - copy business hours to selected days
    $("body").on(
      "click",
      ".directorist-bh-dayzone__copy-dropdown__btn--apply",
      function (e) {
        e.preventDefault();
        const $dropdown = $(".directorist-bh-dayzone__copy-dropdown");

        // Get selected days
        const selectedDays = [];
        $dropdown
          .find(
            ".directorist-bh-dayzone__copy-dropdown__checkbox:checked:not(#directorist-bh-copy-select-all)"
          )
          .each(function () {
            selectedDays.push($(this).val());
          });

        // Copy business hours data to selected days
        copyBusinessHoursToSelectedDays(selectedDays);

        // Close dropdown
        closeDropdown($dropdown);

        // Reset checkboxes
        $dropdown
          .find(".directorist-bh-dayzone__copy-dropdown__checkbox")
          .prop("checked", false);
      }
    );
  }

  /**
   * Update Select All checkbox state based on individual selections
   * @param {jQuery} $dropdown - The dropdown element
   */
  function updateSelectAllState($dropdown) {
    const $allCheckboxes = $dropdown.find(
      ".directorist-bh-dayzone__copy-dropdown__checkbox:not(#directorist-bh-copy-select-all)"
    );
    const $selectAll = $dropdown.find("#directorist-bh-copy-select-all");
    const $selectAllLabel = $dropdown.find(
      'label[for="directorist-bh-copy-select-all"]'
    );

    // Filter out disabled items when counting
    const $enabledCheckboxes = $allCheckboxes.filter(function () {
      const $item = $(this).closest(
        ".directorist-bh-dayzone__copy-dropdown__item"
      );
      return !$item.hasClass("disabled");
    });

    const checkedCount = $enabledCheckboxes.filter(":checked").length;
    const totalCount = $enabledCheckboxes.length;

    // Update Select All state based on individual selections
    if (checkedCount === 0) {
      $selectAll.prop("indeterminate", false).prop("checked", false);
      // Update label text to "Select All"
      $selectAllLabel.text("Select All");
    } else if (checkedCount === totalCount) {
      $selectAll.prop("indeterminate", false).prop("checked", true);
      // Update label text to "Unselect All"
      $selectAllLabel.text("Unselect All");
    } else {
      $selectAll.prop("indeterminate", true);
      // Update label text to "Select All" for indeterminate state
      $selectAllLabel.text("Select All");
    }
  }

  /**
   * Update button states (enabled/disabled) based on checkbox selections
   * @param {jQuery} $dropdown - The dropdown element
   */
  function updateButtonStates($dropdown) {
    const $allCheckboxes = $dropdown.find(
      ".directorist-bh-dayzone__copy-dropdown__checkbox:not(#directorist-bh-copy-select-all)"
    );
    const $resetBtn = $dropdown.find(
      ".directorist-bh-dayzone__copy-dropdown__btn--reset"
    );
    const $applyBtn = $dropdown.find(
      ".directorist-bh-dayzone__copy-dropdown__btn--apply"
    );

    // Filter out disabled items when counting
    const $enabledCheckboxes = $allCheckboxes.filter(function () {
      const $item = $(this).closest(
        ".directorist-bh-dayzone__copy-dropdown__item"
      );
      return !$item.hasClass("disabled");
    });

    const checkedCount = $enabledCheckboxes.filter(":checked").length;
    const totalCount = $enabledCheckboxes.length;

    // Reset All button: disabled when no checkboxes are selected
    $resetBtn.prop("disabled", checkedCount === 0);

    // Apply button: disabled when no checkboxes are selected
    $applyBtn.prop("disabled", checkedCount === 0);

    // Update Select All label text based on current state
    const $selectAllLabel = $dropdown.find(
      'label[for="directorist-bh-copy-select-all"]'
    );
    if (checkedCount === totalCount && totalCount > 0) {
      $selectAllLabel.text("Unselect All");
    } else {
      $selectAllLabel.text("Select All");
    }
  }

  /**
   * Clear dropdown inline styles and close it
   * @param {jQuery} $dropdown - The dropdown element
   */
  function closeDropdown($dropdown) {
    $dropdown.removeClass("directorist-bh-dayzone__copy-dropdown--show");
    // Clear dynamically added inline styles
    $dropdown.css({
      display: "",
      position: "",
      top: "",
      left: "",
      "z-index": "",
    });
    // Remove disabled class from all dropdown items when closing
    $dropdown
      .find(".directorist-bh-dayzone__copy-dropdown__item")
      .removeClass("disabled");
    // Clear all selected checkboxes when closing
    $dropdown
      .find(".directorist-bh-dayzone__copy-dropdown__checkbox")
      .prop("checked", false);

    // Remove active class from current button
    $(".directorist-bh-dayzone__single__action__btn").removeClass("active");
    // Clear current button reference when closing
    currentButton = null;
  }

  /**
   * Copy business hours data from source day to selected target days
   * @param {Array} selectedDays - Array of selected day values (e.g., ['bdbh[monday]', 'bdbh[tuesday]'])
   */
  function copyBusinessHoursToSelectedDays(selectedDays) {
    if (!currentDay || selectedDays.length === 0) {
      console.warn("No source day or selected days found");
      return;
    }

    const sourceDayValue = currentDay.data("day");

    // Extract day name from source (e.g., 'bdbh[monday]' -> 'monday')
    const sourceDayName = sourceDayValue.replace("bdbh[", "").replace("]", "");

    // Copy to each selected day
    selectedDays.forEach(function (targetDayValue) {
      const $targetDay = $(
        `.directorist-bh-dayzone__single[data-day="${targetDayValue}"]`
      );

      if ($targetDay.length) {
        // Extract day name from target (e.g., 'bdbh[tuesday]' -> 'tuesday')
        const targetDayName = targetDayValue
          .replace("bdbh[", "")
          .replace("]", "");

        copyDayData(currentDay, $targetDay, sourceDayName, targetDayName);
      } else {
        console.warn(`Target day not found: ${targetDayValue}`);
      }
    });
  }

  /**
   * Copy all business hours data from source day to target day
   * @param {jQuery} $sourceDay - Source day element
   * @param {jQuery} $targetDay - Target day element
   * @param {String} sourceDayName - Source day name (e.g., 'monday')
   * @param {String} targetDayName - Target day name (e.g., 'tuesday')
   */
  function copyDayData($sourceDay, $targetDay, sourceDayName, targetDayName) {
    // 1. Copy enable/disable state
    const $sourceEnable = $sourceDay.find(
      `input[name="bdbh[${sourceDayName}][enable]"]`
    );
    const $targetEnable = $targetDay.find(
      `input[name="bdbh[${targetDayName}][enable]"]`
    );
    if ($sourceEnable.length && $targetEnable.length) {
      $targetEnable.prop("checked", $sourceEnable.prop("checked"));
      // Trigger change event to update UI
      $targetEnable.trigger("change");
    }

    // 2. Copy 24/7 hour selection state
    const $source247 = $sourceDay.find(
      `input[name="bdbh[${sourceDayName}][remain_close]"]`
    );
    const $target247 = $targetDay.find(
      `input[name="bdbh[${targetDayName}][remain_close]"]`
    );
    if ($source247.length && $target247.length) {
      $target247.prop("checked", $source247.prop("checked"));
      // Trigger change event to update UI
      $target247.trigger("change");
    }

    // 3. Copy time slots
    copyTimeSlots($sourceDay, $targetDay, sourceDayName, targetDayName);

    // 4. Update UI classes and states
    updateDayUIState($targetDay, $sourceDay);
  }

  /**
   * Copy time slots from source to target day
   * @param {jQuery} $sourceDay - Source day element
   * @param {jQuery} $targetDay - Target day element
   * @param {String} sourceDayName - Source day name
   * @param {String} targetDayName - Target day name
   */
  function copyTimeSlots($sourceDay, $targetDay, sourceDayName, targetDayName) {
    const $sourceChoice = $sourceDay.find(
      ".directorist-bh-dayzone__single__choice"
    );
    const $targetChoice = $targetDay.find(
      ".directorist-bh-dayzone__single__choice"
    );

    if (!$sourceChoice.length || !$targetChoice.length) {
      console.warn("Choice elements not found");
      return;
    }

    // Get the wrapper containers
    const $sourceWrapper = $sourceChoice.find(
      ".directorist-bh-dayzone__single__choice__wrapper"
    );
    const $targetWrapper = $targetChoice.find(
      ".directorist-bh-dayzone__single__choice__wrapper"
    );

    if (!$sourceWrapper.length || !$targetWrapper.length) {
      console.warn("Choice wrappers not found", {
        sourceWrapper: $sourceWrapper.length,
        targetWrapper: $targetWrapper.length,
      });
      return;
    }

    // Note: We assume target wrapper always has at least one item
    // This is guaranteed by the WordPress template structure

    // Get all source time slot items
    const $sourceItems = $sourceWrapper.find(
      ".directorist-bh-dayzone__single__choice__item"
    );

    // Clear existing target items (except the first one which is not removable)
    const $targetItems = $targetWrapper.find(
      ".directorist-bh-dayzone__single__choice__item"
    );

    // Clear existing target items but always keep the first one
    if ($targetItems.length > 1) {
      // Remove all items except the first one (which should be non-removable)
      $targetItems.slice(1).remove();
    }

    // Always clear values of the first item to prepare for copying
    const $firstItem = $targetItems.first();
    if ($firstItem.length) {
      $firstItem.find("select").val(null);
      $firstItem.find('input[type="checkbox"]').prop("checked", false);
    }

    // Copy each source item to target
    $sourceItems.each(function (index) {
      const $sourceItem = $(this);
      let $targetItem;

      if (index === 0) {
        // Use the existing first item for the first slot
        $targetItem = $targetWrapper
          .find(".directorist-bh-dayzone__single__choice__item")
          .first();
      } else {
        // Clone the first item for additional slots
        $targetItem = $targetWrapper
          .find(".directorist-bh-dayzone__single__choice__item")
          .first()
          .clone();
        $targetItem.removeClass("directorist-not-removable");

        // Clean up any existing Select2 instances from the clone
        $targetItem.find("select").each(function () {
          const $select = $(this);
          if ($select.hasClass("select2-hidden-accessible")) {
            $select.select2("destroy");
          }
          // Remove any Select2 related attributes and elements
          $select
            .removeClass("select2-hidden-accessible")
            .removeAttr("data-select2-id tabindex aria-hidden")
            .siblings(".select2")
            .remove();
          $select.closest(".directorist-selects").removeAttr("data-select2-id");
          $select
            .closest(".directorist-bh-dayzone__single__choice__slot")
            .removeAttr("data-select2-id");
        });

        // Append to the choice wrapper container
        $targetWrapper.append($targetItem);
      }

      // Update name attributes for the new index first
      $targetItem.find("select").each(function () {
        const $select = $(this);
        const currentName = $select.attr("name");
        if (currentName) {
          const newName = currentName.replace(/\[\d+\]/, `[${index}]`);
          $select.attr("name", newName);
        }
      });

      // Copy start time
      const $sourceStart = $sourceItem.find(
        `select[name="bdbh[${sourceDayName}][start][${index}]"]`
      );
      const $targetStart = $targetItem.find(
        `select[name="bdbh[${targetDayName}][start][${index}]"]`
      );
      if ($sourceStart.length && $targetStart.length) {
        $targetStart.val($sourceStart.val());
        // Reinitialize Select2 if needed
        if ($targetStart.hasClass("select2-hidden-accessible")) {
          $targetStart.select2("destroy");
        }
        $targetStart.select2({
          placeholder: "Select Starting Time",
          allowClear: true,
          width: "100%",
          dropdownCssClass: "dbh-select2-dropdown",
          containerCssClass: "dbh-select2-container",
        });
      }

      // Copy end time
      const $sourceEnd = $sourceItem.find(
        `select[name="bdbh[${sourceDayName}][close][${index}]"]`
      );
      const $targetEnd = $targetItem.find(
        `select[name="bdbh[${targetDayName}][close][${index}]"]`
      );
      if ($sourceEnd.length && $targetEnd.length) {
        $targetEnd.val($sourceEnd.val());
        // Reinitialize Select2 if needed
        if ($targetEnd.hasClass("select2-hidden-accessible")) {
          $targetEnd.select2("destroy");
        }
        $targetEnd.select2({
          placeholder: "Select Closing Time",
          allowClear: true,
          width: "100%",
          dropdownCssClass: "dbh-select2-dropdown",
          containerCssClass: "dbh-select2-container",
        });
      }
    });

    // Update remove button states
    const $finalTargetItems = $targetWrapper.find(
      ".directorist-bh-dayzone__single__choice__item"
    );
    $finalTargetItems.toggleClass(
      "directorist-not-removable",
      $finalTargetItems.length <= 1
    );
  }

  /**
   * Update target day UI state to match source day
   * @param {jQuery} $targetDay - Target day element
   * @param {jQuery} $sourceDay - Source day element
   */
  function updateDayUIState($targetDay, $sourceDay) {
    // Copy CSS classes
    const sourceClasses = $sourceDay.attr("class").split(" ");
    const relevantClasses = sourceClasses.filter(
      (cls) =>
        cls.includes("directorist-enable-hour-time-selection") ||
        cls.includes("directorist-full-time")
    );

    // Remove existing classes
    $targetDay.removeClass(
      "directorist-enable-hour-time-selection directorist-full-time"
    );

    // Add relevant classes
    relevantClasses.forEach((cls) => {
      $targetDay.addClass(cls);
    });

    // Trigger change events to update UI
    $targetDay.find('input[type="checkbox"]').each(function () {
      $(this).trigger("change");
    });

    // Update next day labels for time selects
    setTimeout(() => {
      $targetDay
        .find(".directorist-bh-dayzone__single__choice__item")
        .each(function () {
          const $item = $(this);
          const $startSelect = $item.find(".directorist-selects--start select");
          const $endSelect = $item.find(".directorist-selects--close select");

          if ($startSelect.length && $endSelect.length) {
            handleBHNextDayLabel($startSelect, $endSelect);
          }
        });
    }, 100);
  }

  /**
   * Disable the current day's checkbox in the dropdown
   * @param {jQuery} $dropdown - The dropdown element
   * @param {String} currentDayValue - The current day's data-day value (e.g., 'bdbh[monday]')
   */
  function disableCurrentDayInDropdown($dropdown, currentDayValue) {
    // First, remove disabled class from all dropdown items
    $dropdown
      .find(".directorist-bh-dayzone__copy-dropdown__item")
      .removeClass("disabled");

    // Find the checkbox with matching value and disable its parent item
    const $matchingCheckbox = $dropdown.find(
      `input[value="${currentDayValue}"]`
    );
    if ($matchingCheckbox.length) {
      const $parentItem = $matchingCheckbox.closest(
        ".directorist-bh-dayzone__copy-dropdown__item"
      );
      $parentItem.addClass("disabled");
    } else {
      console.warn(`Checkbox not found for current day: ${currentDayValue}`);
    }
  }

  /**
   * Position dropdown relative to the clicked button
   * @param {jQuery} $dropdown - The dropdown element
   * @param {jQuery} $button - The button element
   */
  function positionDropdownRelativeToButton($dropdown, $button) {
    // Get button position relative to viewport using getBoundingClientRect for more accuracy
    const buttonRect = $button[0].getBoundingClientRect();

    // Get dropdown dimensions
    const dropdownWidth = 172; // Fixed width from CSS
    const dropdownHeight = 200; // Approximate height

    // Calculate position using viewport coordinates
    let left = buttonRect.right - dropdownWidth; // Align to right edge
    let top = buttonRect.bottom + 4; // 4px margin from button

    // Ensure dropdown doesn't go off-screen horizontally
    if (left < 10) {
      left = 10; // 10px margin from left edge
    } else if (left + dropdownWidth > window.innerWidth - 10) {
      left = window.innerWidth - dropdownWidth - 10; // 10px margin from right edge
    }

    // Ensure dropdown doesn't go off-screen vertically
    if (top + dropdownHeight > window.innerHeight - 10) {
      // Position above the button instead
      top = buttonRect.top - dropdownHeight - 4;
    }

    // Apply position
    $dropdown.css({
      display: "block",
      position: "fixed",
      top: top + "px",
      left: left + "px",
      "z-index": 1000,
    });
  }

  // Initialize copy dropdown on page load
  $(document).ready(function () {
    initializeCopyDropdown();
  });
})(jQuery);
