<?php
// Prohibit direct script loading.
defined( 'ABSPATH' ) || die( 'No direct script access allowed!' );

if ( ! function_exists( 'atbdp_get_option' ) ) {

    /**
     * It retrieves an option from the database if it exists and returns false if it is not exist.
     * It is a custom function to get the data of custom setting page
     * @param string $name The name of the option we would like to get. Eg. map_api_key
     * @param string $group The name of the group where the option is saved. eg. general_settings
     * @param mixed $default    Default value for the option key if the option does not have value then default will be returned
     * @return mixed    It returns the value of the $name option if it exists in the option $group in the database, false otherwise.
     */
    function atbdp_get_option( $name, $group, $default = false ) {
        // at first get the group of options from the database.
        // then check if the data exists in the array and if it exists then return it
        // if not, then return false
        if ( empty( $name ) || empty( $group ) ) {
            if ( ! empty( $default ) ) {
                return $default;
            }

            return false;
        } // vail if either $name or option $group is empty
        $options_array = (array) get_option( $group );
        if ( array_key_exists( $name, $options_array ) ) {
            return $options_array[$name];
        } else {
            if ( ! empty( $default ) ) {
                return $default;
            }

            return false;
        }
    }
}

if ( ! function_exists( 'atbdp_sanitize_array' ) ) {
    /**
     * It sanitize a multi-dimensional array
     * @param array &$array The array of the data to sanitize
     * @return mixed
     */
    function atbdp_sanitize_array( &$array ) {

        foreach ( $array as &$value ) {

            if ( ! is_array( $value ) ) {

                // sanitize if value is not an array
                $value = sanitize_text_field( $value );

            } else {

                // go inside this function again
                atbdp_sanitize_array( $value );
            }

        }

        return $array;

    }
}

function atbdp_hoursRange( $lower = 0, $upper = 86400, $step = 3600, $format = '' ) {
    $times = [];
    foreach ( range( $lower, $upper, $step ) as $increment ) {
        $increment                  = gmdate( 'H:i', $increment );
        list( $hour, $minutes )       = explode( ':', $increment );
        $date                       = new DateTime( $hour . ':' . $minutes );
        $times[(string) $increment] = $date->format( $format );
    }
    return $times;
}

function atbdp_get_old_hours( $old ) {
    $times = atbdp_hoursRange( 0, 86400, 60 * 15, 'g:i a' );
    foreach ( $times as $key => $time ) {
        if ( $old == $key ) {
            return $key;
        } elseif ( $old == $time ) {
            return $time;
        }
    }
}

function atbdp_hours( $name, $value = '', $type = '' ) {
    $intval      = (int) apply_filters( 'directorist_business_hours_interval', 15 );
    $times       = atbdp_hoursRange( 0, 86400, 60 * $intval, 'g:i a' );
    $type        = $type == 'open' ? __( 'Select Starting Time', 'directorist-business-hours' ) : __( 'Select Closing Time', 'directorist-business-hours' );
    $time_format = get_directorist_option( 'atbh_time_format', '12' );
    $html        = '';
    $html .= '<select name="' . $name . '">';
    $html .= '<option value="">' . $type . '</option>';
    foreach ( $times as $key => $time ) {
        if ( '24' == $time_format ) {
            $time = $key;
        } else {
            $time = $time;
        }
        $html .= sprintf( '<option value="%s" %s>%s</option>', $key, selected( $value, $key, false ), $time );
    }
    $html .= '</select>';
    return $html;
}

function atbdp_hoursss() {
    $times = atbdp_hoursRange( 0, 86400, 60 * 15, 'g:i a' );
    $html  = '';
    $html .= '<ul class="dbh-default-times">';
    $time_format = get_directorist_option( 'atbh_time_format', '12' );
    foreach ( $times as $key => $time ) {
        if ( '24' == $time_format ) {
            $time = $key;
        } else {
            $time = $time;
        }
        $html .= sprintf( '<li><a data-time="%s" href="">%s</a></li>', $key, $time );
    }
    $html .= '</ul>';
    return $html;
}

if( ! function_exists( 'dbh_is_open' ) ) {
    function dbh_is_open( $time_open, $time_close, $timezone ) {
        $dt           = new DateTime( 'now', new DateTimezone( $timezone ) );
        $current_time = $dt->format( 'g:i a' );
        $time_now     = DateTime::createFromFormat( 'H:i a', $current_time );

		if ($time_close >= $time_open) {

			return $time_open <= $time_now && $time_now < $time_close;

		} else {

			return !($time_close <= $time_now && $time_now < $time_open);
		}
    }
}



function atbdp_time_calculation( $time, $day, $business_hours = [], $timezone = '' ) {
	
    $_start_times = $time['start'] ? $time['start'] : '';
    $_close_time  = $time['close'] ? $time['close'] : '';
    $dt           = new DateTime( 'now', new DateTimezone( $timezone ) );
    $current_time = $dt->format( 'g:i a' );
    $time_now     = DateTime::createFromFormat( 'H:i a', $current_time );

    if ( $_start_times ) {
        $business_times = [];
        foreach ( $_start_times as $index => $_start_time ) {
            $business_times[] = [
                'start' => $_start_time,
                'close' => $_close_time[$index],
            ];
        }
        foreach ( $business_times as $business_time ) {


            $start_time = date( 'h:i a', strtotime( esc_attr( $business_time['start'] ) ) );
            $close_time = date( 'h:i a', strtotime( esc_attr( $business_time['close'] ) ) );
            $time_start = DateTime::createFromFormat( 'H:i a', $start_time );
            $time_end   = DateTime::createFromFormat( 'H:i a', $close_time );


            $is_open = dbh_is_open( $start_time, $close_time, $timezone );

            if( $is_open ) {
                $_day = date( 'D' );
            }else{
                $_day = 'cls';
            }
           
            /*
             * time start as am (12.01 am to 11.58 am)
             * lets calculate time
             * is start time is smaller than current time and grater than close time
             */
            if (  ( $time_now >= $time_start ) && ( $time_now < $time_end ) ) {
                $_day = date( 'D' );
            }

            if ( empty( $time['enable'] ) ) {
                $_day = '';
            }
        }
    }

    //e_var_dump($time['remain_close']);
    $open_all_day = ( ! empty( $time['remain_close'] ) && 'open' === $time['remain_close'] ) ? 1 : '';
    $remain_close = ( ! empty( $time['remain_close'] ) && (  ( 'on' === $time['remain_close'] ) || ( '1' === $business_hours[$day]['remain_close'] ) ) ) ? 1 : '';
    if ( 1 == $open_all_day ) {
        $_day = date( 'D' );
    }
    if ( $remain_close ) {
        $_day = '';
    }
    return $_day;

}


if ( ! function_exists( 'atbdp_time_calculationn' ) ) {

    function atbdp_time_calculationn( $time, $day, $business_hours = [], $timezone = '' ) {

        $_day         = '';
        $interval     = DateTime::createFromFormat( 'H:i a', '11:59 am' );
        $dt           = new DateTime( 'now', new DateTimezone( $timezone ) );
        $current_time = $dt->format( 'g:i a' );
        $time_now     = DateTime::createFromFormat( 'H:i a', $current_time );

        $_start_time  = ! empty( $business_hours[$day]['start'] ) ? $business_hours[$day]['start'] : '';
        $_start_times = is_array( $_start_time ) ? $_start_time : [ $_start_time ];
        $_close_time  = ! empty( $business_hours[$day]['close'] ) ? $business_hours[$day]['close'] : '';
        $_close_time  = is_array( $_close_time ) ? $_close_time : [ $_close_time ];

        if ( $_start_times ) {
            $business_times = [];
            foreach ( $_start_times as $index => $_start_time ) {
                $business_times[] = [
                    'start' => $_start_time,
                    'close' => $_close_time[$index],
                ];
            }
            foreach ( $business_times as $business_time ) {


                $start_time = date( 'h:i a', strtotime( esc_attr( $business_time['start'] ) ) );
                $close_time = date( 'h:i a', strtotime( esc_attr( $business_time['close'] ) ) );
                $time_start = DateTime::createFromFormat( 'H:i a', $start_time );
                $time_end   = DateTime::createFromFormat( 'H:i a', $close_time );


                $is_open = dbh_is_open( $start_time, $close_time, $timezone );

                if( $is_open ) {
                    $_day = date( 'D' );
                }else{
                    $_day = 'cls';
                }
               
                /*
                 * time start as am (12.01 am to 11.58 am)
                 * lets calculate time
                 * is start time is smaller than current time and grater than close time
                 */
                if (  ( $time_now >= $time_start ) && ( $time_now < $time_end ) ) {
                    $_day = date( 'D' );
                }

                if ( empty( $business_hours[$day]['enable'] ) ) {
                    $_day = '';
                }
            }
        }

        //e_var_dump($business_hours[$day]['remain_close']);
        $open_all_day = ( ! empty( $business_hours[$day]['remain_close'] ) && 'open' === $business_hours[$day]['remain_close'] ) ? 1 : '';
        $remain_close = ( ! empty( $business_hours[$day]['remain_close'] ) && (  ( 'on' === $business_hours[$day]['remain_close'] ) || ( '1' === $business_hours[$day]['remain_close'] ) ) ) ? 1 : '';
        if ( 1 == $open_all_day ) {
            $_day = date( 'D' );
        }
        if ( $remain_close ) {
            $_day = '';
        }
        return $_day;
    }
}

if ( ! function_exists( 'atbdp_hours_badge' ) ) {
    function atbdp_hours_badge( $business_hours, $time, $day, $timezone ) {
        $open_close  = '';
        $open_       = get_directorist_option( 'open_badge_text', __( 'Open', 'directorist-business-hours' ) );
        $close_      = get_directorist_option( 'close_badge_text', 'Closed', 'directorist-business-hours' );
        $time        = atbdp_time_calculation( $time, $day, $business_hours, $timezone );
        $open_class  = directorist_legacy_mode() ? 'atbd_badge atbdp_info_list atbd_badge_open' : 'directorist-badge directorist-info-item directorist-badge-open';
        $close_class = directorist_legacy_mode() ? 'atbd_badge atbdp_info_list atbd_badge_close' : 'directorist-badge directorist-info-item directorist-badge-close';
        $_day        = date( 'D' );
        if ( 'cls' === $time ) {
            $open_close = '<span class="' . $close_class . '">' . $close_ . '</span>';
        } elseif ( $_day === $time ) {
            $open_close = '<span class="' . $open_class . '">' . $open_ . '</span>';
        }
        return $open_close;
    }
}

if ( ! function_exists( 'business_open_close_status' ) ) {
    function business_open_close_status( $business_hours, $time, $day, $timezone ) {
        $open_close = false;
        $time       = atbdp_time_calculation( $time, $day, $business_hours, $timezone );
        $_day       = date( 'D' );
        if ( 'cls' === $time ) {
            $open_close = false;
        } elseif ( $_day === $time ) {
            $open_close = true;
        }
        var_dump($open_close);
        return $open_close;
    }
}

function directorist_business_hours_get_template( $template_file, $args = [] ) {
    if ( is_array( $args ) ) {
        extract( $args );
    }

    $theme_template  = '/directorist-business-hours/' . $template_file . '.php';
    $plugin_template = BDBH_TEMPLATES_DIR . $template_file . '.php';

    if ( file_exists( get_stylesheet_directory() . $theme_template ) ) {
        $file = get_stylesheet_directory() . $theme_template;
    } elseif ( file_exists( get_template_directory() . $theme_template ) ) {
        $file = get_template_directory() . $theme_template;
    } else {
        $file = $plugin_template;
    }

    do_action( 'directorist_bh_before_template_loaded', $template_file, $file, $args );

    if ( file_exists( $file ) ) {
        include $file;
    }
}

if ( ! function_exists( 'directorist_open_close_badge_class' ) ) {
    function directorist_open_close_badge_class() {
        return directorist_legacy_mode() ? 'atbd_badge atbd_badge_close' : 'directorist-badge directorist-badge-success directorist-badge-open';
    }
}


/**
 * @param   array     $business_time Multislot business time of a single day
 * @param   string    $day Business day
 * @return  boolean   Business open or closed
 * @since   2.3.0
 */

// if ( ! function_exists( 'directorist_business_open_by_hours' ) ) {
//     function directorist_business_open_by_hours( $time, $day ) {
//         $is_open = false;

//         if( strtolower(date("l")) != $day ) {
//             return false;
//         }

//         $current         = date('Y-m-d H:i:s');
//         $starting_times  = ! empty( $time['start'] ) ? $time['start'] : [];
//         $clasing_times   = ! empty( $time['close'] ) ? $time['close'] : [];

//         if ( is_array( $starting_times ) ) {
//             foreach ( $starting_times as $index => $starting_time ) {
                
//                 if ( ! $starting_time ) {
//                     continue;
//                 }

//                 $start  = date( 'Y-m-d H:i:s', strtotime( $starting_time ) );
//                 $close  = date( 'Y-m-d H:i:s', strtotime( $clasing_times[$index] ) );
                
//                 if( directorist_calculated_status( $current, $start, $close ) ) {
//                     $is_open = true;
//                 }

//             }
//         }

//         return $is_open;
//     }
// }


if ( ! function_exists( 'directorist_business_open_by_hours' ) ) {

    function directorist_business_open_by_hours( $time, $day, $timezone = '' ) {
        $is_open = false;
    
        $starting_times = ! empty( $time['start'] ) ? $time['start'] : [];
        $closing_times  = ! empty( $time['close'] ) ? $time['close'] : [];
    
        // Set timezone if provided
        if ( ! empty( $timezone ) ) {
            date_default_timezone_set( $timezone );
        }
        
        $current_ts = time();
    
        if ( is_array( $starting_times ) ) {
            foreach ( $starting_times as $index => $start_time ) {
                $close_time = isset( $closing_times[ $index ] ) ? $closing_times[ $index ] : '';
    
                if ( ! $start_time || ! $close_time ) continue;
    
                // Generate date base
                $date_for_day = ( strtolower( date("l") ) === $day ) ? date( 'Y-m-d' ) : date( 'Y-m-d', strtotime( '-1 day' ) );
    
                $start_ts = strtotime( "$date_for_day $start_time" );
                $end_ts   = strtotime( "$date_for_day $close_time" );
    
                // If closing is earlier than opening, treat as overnight
                if ( $end_ts <= $start_ts ) {
                    $end_ts = strtotime( '+1 day', $end_ts );
                }
    
                if ( $current_ts >= $start_ts && $current_ts <= $end_ts ) {
                    $is_open = true;
                    break;
                }
            }
        }
    
        return $is_open;
    }
}


/**
 * @param   object    $current Time time of defined timezone
 * @param   object    $start Time time of defined timezone
 * @param   object    $close Time time of defined timezone
 * @return  boolean   Business open or closed
 * @since   2.3.0
 */

if ( ! function_exists( 'directorist_calculated_status' ) ) {
    function directorist_calculated_status( $current, $start, $close ) {
        if( ( $start < $current && $close > $current) ){
            return true;
        }else{
            return false;
        }
    }
}

function directorist_calculated_statuss( $current, $start, $close ) {

    $f = DateTime::createFromFormat('!H:i', $start);
    $t = DateTime::createFromFormat('!H:i', $close);
    $i = DateTime::createFromFormat('!H:i', $current);
    if ($f > $t) $t->modify('+1 day');
    return ($f <= $i && $i <= $t) || ($f <= $i->modify('+1 day') && $i <= $t);
}

/**
 * It displays business hours in an unordered list
 * @param int $listing_id default get_the_ID()
 * @return void
 * @since   2.3.0
 */

// if ( ! function_exists( 'directorist_business_open_close_status' ) ) {

//     function directorist_business_open_close_status( $listing_id = '' ) {

//         if( ! $listing_id && is_singular( ATBDP_POST_TYPE ) ) {
//             $listing_id = get_the_ID();
//         }
        
//         $general_none   = get_directorist_option( 'timezone', 'America/New_York' );
//         $listing_zone   = get_post_meta( $listing_id, '_timezone', true );
//         $timezone       = $listing_zone ? $listing_zone : $general_none;

//         $bdbh           = get_post_meta( $listing_id, '_bdbh', true );
//         $for24_7        = get_post_meta( $listing_id, '_enable247hour', true );
//         $disable_hours  = get_post_meta( $listing_id, '_disable_bz_hour_listing', true );
//         $business_hours = ! empty( $bdbh ) ? atbdp_sanitize_array( $bdbh ) : array(); // arrays of days and times if exist
        
//         date_default_timezone_set( $timezone );
//         $today = strtolower(date("l"));
        
//         if( $for24_7 ) {
//             return true;
//         }

//         if( ! $business_hours ) {
//             return false;
//         }

//         if( $disable_hours ) {
//             return false;
//         }

//         $status = false;

//         foreach( $business_hours as $day => $time ) {

//             if ( $day != $today ) {
//                 continue;
//             }

//             if ( empty( $time['enable'] ) ) {
//                 $status = false;
//                 continue;
//             }

//             if ( ! empty( $time['remain_close'] ) && ( 'open' === $time['remain_close'] ) ) {
//                 $status = true;
//                 continue;
//             }

//             $is_open = directorist_business_open_by_hours( $time, $day );
//             $status  = true;

//             if ( empty( $is_open ) ) {
//                 $status = false;
//             }

//         }

//         return $status;
//     }
// }

if ( ! function_exists( 'directorist_business_open_close_status' ) ) {

    function directorist_business_open_close_status( $listing_id = '' ) {

        if ( ! $listing_id && is_singular( ATBDP_POST_TYPE ) ) {
            $listing_id = get_the_ID();
        }
    
        $general_none   = get_directorist_option( 'timezone', 'America/New_York' );
        $listing_zone   = get_post_meta( $listing_id, '_timezone', true );
        $timezone       = $listing_zone ? $listing_zone : $general_none;
    
        $bdbh           = get_post_meta( $listing_id, '_bdbh', true );
        $for24_7        = get_post_meta( $listing_id, '_enable247hour', true );
        $disable_hours  = get_post_meta( $listing_id, '_disable_bz_hour_listing', true );
        $business_hours = ! empty( $bdbh ) ? atbdp_sanitize_array( $bdbh ) : array();
    
        date_default_timezone_set( $timezone );
    
        if ( $for24_7 ) return true;
        if ( ! $business_hours || $disable_hours ) return false;
    
        $today     = strtolower( date("l") );
        $yesterday = strtolower( date("l", strtotime("-1 day")) );
    
        foreach ( [$today, $yesterday] as $day ) {
            if ( empty( $business_hours[ $day ] ) ) continue;
    
            $time = $business_hours[ $day ];
    
            if ( empty( $time['enable'] ) ) continue;
    
            // Check if remain_close is set to 'open' (All Day) - only for current day
            if ( $day === $today && ! empty( $time['remain_close'] ) && $time['remain_close'] === 'open' ) {
                return true;
            }
    
            if ( directorist_business_open_by_hours( $time, $day, $timezone ) ) {
                return true;
            }
        }
    
        return false;
    }
}


function is_day_open_now($day, $time, $timezone = '') {
    $starting_times = ! empty( $time['start'] ) ? $time['start'] : [];
    $closing_times  = ! empty( $time['close'] ) ? $time['close'] : [];
    $now = time();

    foreach ( $starting_times as $index => $start_time ) {
        $end_time = ! empty( $closing_times[$index] ) ? $closing_times[$index] : '';
        if ( ! $start_time || ! $end_time ) continue;

        // Get the base date for the slot
        $day_num = date('w', strtotime($day)); // 0 (Sun) - 6 (Sat)
        $today_num = date('w');
        $base_date = date('Y-m-d', strtotime("last $day", strtotime('tomorrow')));
        if ($today_num == $day_num) {
            $base_date = date('Y-m-d');
        } elseif (($today_num == 0 && $day_num == 6)) { // Special case for Sunday after Saturday
            $base_date = date('Y-m-d', strtotime('yesterday'));
        }

        $start_ts = strtotime("$base_date $start_time");
        $end_ts   = strtotime("$base_date $end_time");

        // Overnight slot
        if ( $end_ts <= $start_ts ) {
            $end_ts = strtotime('+1 day', $end_ts);
        }

        if ( $now >= $start_ts && $now <= $end_ts ) {
            return true;
        }
    }
    return false;
}


/**
 * It return the day string
 * @return string
 * @since  2.7.5
 */
if ( ! function_exists( 'directorist_day' ) ) {
    function directorist_day( $day ) {
        
        if( empty( $day ) ) {
            return '';
        }

        $days = [
            'saturday'  => __( 'Saturday', 'directorist-business-hours' ),
            'sunday'    => __( 'Sunday', 'directorist-business-hours' ),
            'monday'    => __( 'Monday', 'directorist-business-hours' ),
            'tuesday'   => __( 'Tuesday', 'directorist-business-hours' ),
            'wednesday' => __( 'Wednesday', 'directorist-business-hours' ),
            'thursday'  => __( 'Thursday', 'directorist-business-hours' ),
            'friday'    => __( 'Friday', 'directorist-business-hours' ),
        ];
        return array_key_exists( $day, $days ) ? $days[$day] : '';
    }
}


/**
 * It displays business hours in an unordered list
 * @return void
 * @since  2.3.0
 */

if ( ! function_exists( 'show_business_hours' ) ) {

    function show_business_hours( $listing_id = '' ) {


        if( ! $listing_id ) {
            $listing_id     = get_the_ID();
        }
        $general_none   = get_directorist_option( 'timezone', 'America/New_York' );
        $time_format    = get_directorist_option( 'atbh_time_format', '12' );
        $listing_zone   = get_post_meta( $listing_id, '_timezone', true );
        $timezone       = $listing_zone ? $listing_zone : $general_none;
        $bdbh           = get_post_meta( $listing_id, '_bdbh', true );
        $disabled       = get_post_meta( $listing_id, '_disable_bz_hour_listing', true );
        $bdbh_version   = get_post_meta( $listing_id, '_bdbh_version', true );
        $business_hours = !empty( $bdbh ) ? atbdp_sanitize_array( $bdbh ) : array(); // arrays of days and times if exist
        date_default_timezone_set( $timezone );
        $today          = strtolower(date("l"));
        $current        = date('Y-m-d H:i:s');


        if( ! $business_hours || empty( directorist_is_business_hour_enabled_for_listing( $listing_id ) ) ) {
            return;
        }

        $default_day_time = array(
            "start" => array(
                0 => ""
            ),
            "close" => array(
                0 => ""
            )
        );


        $update_business_hours = array(
            "monday" => ! empty( $business_hours["monday"] ) ? $business_hours["monday"] : $default_day_time,
            "tuesday" => ! empty( $business_hours["tuesday"] ) ? $business_hours["tuesday"] : $default_day_time,
            "wednesday" => ! empty( $business_hours["wednesday"] ) ? $business_hours["wednesday"] : $default_day_time,
            "thursday" => ! empty( $business_hours["thursday"] ) ? $business_hours["thursday"] : $default_day_time,
            "friday" => ! empty( $business_hours["friday"] ) ? $business_hours["friday"] : $default_day_time,
            "saturday" => ! empty( $business_hours["saturday"] ) ? $business_hours["saturday"] : $default_day_time,
            "sunday" => ! empty( $business_hours["sunday"] ) ? $business_hours["sunday"] : $default_day_time,
        );

        // After determining $today
        $prev_day        = strtolower( date( 'l', strtotime( 'yesterday' ) ) );
        $overnight_active = false;

        $prev_day_enabled = true;

        if ( ! empty( $update_business_hours[ $prev_day ] ) && empty( $update_business_hours[ $prev_day ]['enable'] ) ) {
            $prev_day_enabled = false;
        }

        if ( ! empty( $prev_day_enabled ) && ! empty( $update_business_hours[ $prev_day ]['start'] ) && ! empty( $update_business_hours[ $prev_day ]['close'] ) ) {
            foreach ( $update_business_hours[ $prev_day ]['start'] as $index => $start_time ) {
                $start_ts = strtotime( date( 'Y-m-d', strtotime( '-1 day' ) ) . " $start_time" );
                $close_ts = strtotime( date( 'Y-m-d', strtotime( '-1 day' ) ) . ' ' . $update_business_hours[ $prev_day ]['close'][ $index ] );
                $now_ts   = time();

                if ( $close_ts <= $start_ts ) {
                    $close_ts = strtotime( date( 'Y-m-d', strtotime( '-1 day' ) ) . ' ' . $update_business_hours[ $prev_day ]['close'][ $index ] . ' +1 day' );
                }

                if ( $now_ts >= $start_ts && $now_ts < $close_ts ) {
                    $today            = $prev_day;
                    $overnight_active = true;
                    break;
                }
            }
        }

        ?> <ul>
            <?php
        $open_now_day = '';
        foreach( $update_business_hours as $day => $time ) {
            $day_enabled = true;

            if ( ! empty( $bdbh_version ) && empty( $time['enable'] ) ) {
                $day_enabled = false;
            }

            // IMPORTANT: If overnight shift is active, skip setting $open_now_day
            if (!$overnight_active && $day_enabled && is_day_open_now($day, $time, $timezone)) {
                $open_now_day = $day;
            }

            if ( ! $time ) {
                continue;
            }
            $starting_times  = ! empty( $time['start'] ) ? $time['start'] : '';
            $clasing_times   = ! empty( $time['close'] ) ? $time['close'] : '';

            $is_open = directorist_business_open_close_status( $listing_id );

            // Only one day gets the "today" class
            if ( $open_now_day ) {
                $the_day_condition = ( $day == $open_now_day );
            } else {
                $the_day_condition = ( $day == $today );
            }
            $the_day    = $the_day_condition ? ' directorist-bh-schedule__item--today' : '';
            $open__now  = $is_open ? 'directorist-bh-schedule__item--open' : 'directorist-bh-schedule__item--closed';

            ?>
            <li class="directorist-bh-schedule__item directorist-bh-schedule__item--<?php echo esc_attr( ! $day_enabled ? 'closed': ''); ?> <?php echo esc_attr( $the_day ); ?>">

            <span class="directorist-business-day directorist-bh-schedule__day">
                <?php echo esc_html( directorist_day( $day ) ); ?>
                <?php if ( $the_day_condition ) : ?>
                    <span class="directorist-bh-schedule__day__badge directorist-bh-schedule__day__badge--today">Today</span>
                <?php endif; ?>
            </span>

            <?php if( ( empty( $bdbh_version ) ) && ( ! empty( $time['remain_close'] ) && (  ( 'on' === $time['remain_close'] ) || ( '1' === $time['remain_close'] ) ) ) ) { ?>

            <span class="directorist-business-hours directorist-bh-schedule__time">
                <span class="directorist-bh-schedule__item__badge directorist-bh-schedule__item__badge--closed">
                    <?php _e( 'Closed', 'directorist-business-hours' )?>
                </span>
            </span>

            <?php }elseif( ! empty( $bdbh_version ) && empty( $time['enable'] ) ) { ?>

            <span class="directorist-business-hours directorist-bh-schedule__time">
                <span class="directorist-bh-schedule__item__badge directorist-bh-schedule__item__badge--closed">
                    <?php _e( 'Closed', 'directorist-business-hours' )?>
                </span>
            </span>

            <?php }else{ ?>

            <div class="directorist-business-hours directorist-bh-schedule__time">

            <?php if ( ! empty( $time['remain_close'] ) && ( 'open' === $time['remain_close'] ) ) { ?>

            <span class="directorist-bh-schedule__item__badge directorist-bh-schedule__item__badge--open"><?php echo __( 'Open 24h', 'directorist-business-hours' ); ?></span>

            <?php }else{
            
            
            // time-slots
            foreach ( $starting_times as $index => $starting_time ) {
                

                if ( ! $starting_time ) {
                    continue;
                }

                $start  = date( 'h:i a', strtotime( $starting_time ) );
                $close  = date( 'h:i a', strtotime( $clasing_times[$index] ) );

                if ( '24' == $time_format ) {
                    $start = DateTime::createFromFormat( 'H:i a', $start )->format( 'H:i' );
                    $close = DateTime::createFromFormat( 'H:i a', $close )->format( 'H:i' );
                }

                $start_  = date( 'Y-m-d H:i:s', strtotime( $starting_time ) );
                $close_  = date( 'Y-m-d H:i:s', strtotime( $clasing_times[$index] ) );

                //print time for multi slot
                $open_slot = '';

                if( directorist_calculated_status( $current, $start_, $close_ ) ) {
                    $open_slot = 'directorist-bh-schedule__open_slot';
                }
                ?>
                <div class='directorist-time-single <?php echo esc_attr( $open_slot ); ?>'>
                    <span class="time directorist-start-time"><?php echo $start; ?></span> - <span class="time directorist-close-time"><?php echo $close; ?></span>
                    <?php
                    // Get 3-letter day abbreviation for the next day if slot is overnight
                    $start_ts = strtotime($starting_time);
                    $close_ts = strtotime($clasing_times[$index]);

                    if ($close_ts !== false && $start_ts !== false && $close_ts <= $start_ts) {
                        // Overnight slot, show next day's 3-letter abbreviation
                        $next_day = date('D', strtotime("$day +1 day"));
                        echo '<span class="directorist-bh-schedule__day__badge directorist-bh-schedule__day__badge--nextday">' . esc_html($next_day) . '</span>';
                    }
                    ?>
                </div>
                <?php
            } // end foreach of timeslot
            } // if not 24*7
            } // if not close show hours
            ?> </li> <?php
        } // end foreach of days
        ?> </ul> <?php

    }
}

/**
 * It displays business badges
 * @return void
 * @since  2.3.0
 */

if ( ! function_exists( 'directorist_show_open_close_badge' ) ) {

    function directorist_show_open_close_badge( $listing_id = '' ) {

        $open             = get_directorist_option('open_badge_text', 'Open' );
        $close            = get_directorist_option('close_badge_text', 'Closed' );
        $time_format      = get_directorist_option('atbh_time_format', '12' );

        $status = directorist_business_open_close_status( $listing_id );
        //$status = directorist_business_hours_status_times( $listing_id );
        
        if( ! empty( $status ) ) {
            ?>
            <span class="atbd_upper_badge__text atbd_upper_badge__text--open directorist-bh-module__status__text directorist-bh-module__status__text--open"><?php echo esc_html($open ); ?></span>

        <?php } else { ?>

            <span class="atbd_upper_badge__text atbd_upper_badge__text--close directorist-bh-module__status__text module__status__text--close"><?php echo esc_attr( $close ); ?></span>

        <?php }

    }
}

if ( ! function_exists( 'directorist_listings_show_open_close_badge' ) ) {

    function directorist_listings_show_open_close_badge( $listing_id = '' ) {

        $open = get_directorist_option('open_badge_text', 'Open' );
        $close = get_directorist_option('close_badge_text', 'Closed' );

        $status = directorist_business_open_close_status( $listing_id );

        if( $status ) { ?>
            <span class="directorist-badge directorist-info-item directorist-badge-open"><?php echo esc_attr( $open ); ?></span>
        <?php
        }else{ ?>
            <span class="directorist-badge directorist-info-item directorist-badge-close"><?php echo esc_attr( $close ); ?></span>
        <?php }


    }
}

function directorist_human_time_diff_short( $from, $to ) {
    $diff    = abs( $to - $from );
    $days    = floor( $diff / 86400 );
    $hours   = floor( ( $diff % 86400 ) / 3600 );
    $minutes = floor( ( $diff % 3600 ) / 60 );

    if ( $days > 0 ) {
        return $days . 'd';
    } elseif ( $hours > 0 ) {
        return $hours . 'h';
    } else {
        return $minutes . 'm';
    }
}


/**
 * Get current open/close status and next open/close times for a listing.
 * Returns:
 * [
 *   'is_open' => bool,
 *   'current_open' => timestamp|null,
 *   'current_close' => timestamp|null,
 *   'next_open' => timestamp|null,
 *   'next_close' => timestamp|null,
 *   'open_day' => string|null,
 *   'close_day' => string|null,
 * ]
 */
function directorist_business_hours_status_times( $listing_id = '' ) {
    if( ! $listing_id ) $listing_id = get_the_ID();

    $general_none   = get_directorist_option( 'timezone', 'America/New_York' );
    $listing_zone   = get_post_meta( $listing_id, '_timezone', true );
    $timezone       = $listing_zone ? $listing_zone : $general_none;
    $bdbh           = get_post_meta( $listing_id, '_bdbh', true );
    $enable247hour  = get_post_meta( $listing_id, '_enable247hour', true );
    $business_hours = !empty( $bdbh ) ? atbdp_sanitize_array( $bdbh ) : array();
    
    if ( ! empty( $enable247hour ) ) { 
        return [
            'always_open' => true,
        ];
    }

    if( empty( $business_hours ) ) return [
        'is_open' => false,
        'current_open' => null,
        'current_close' => null,
        'next_open' => null,
        'next_close' => null,
        'open_day' => null,
        'close_day' => null,
    ];

    date_default_timezone_set( $timezone );
    $now = time();
    $days = ['sunday','monday','tuesday','wednesday','thursday','friday','saturday'];
    $today_idx = date('w', $now);

    $current_open = null;
    $current_close = null;
    $open_day = null;
    $close_day = null;
    $is_open = false;
    
    // 1. Check if currently open (including overnight from yesterday)
    for ($offset = 0; $offset <= 1; $offset++) {
        $day_idx = ($today_idx - $offset + 7) % 7;
        $day = $days[$day_idx];
        
        if (empty($business_hours[$day]['enable'])) continue;
        
        // Check if remain_close is set to 'open' (All Day)
        if ( $offset === 0 && ! empty( $business_hours[$day]['remain_close'] ) && $business_hours[$day]['remain_close'] === 'open' ) {
            $is_open = true;
            $current_open = strtotime(date('Y-m-d 00:00:00', $now));
            $current_close = strtotime(date('Y-m-d 23:59:59', $now));
            $open_day = $day;
            $close_day = $day;
            break;
        }
        
        $starts = !empty($business_hours[$day]['start']) ? $business_hours[$day]['start'] : [];
        $closes = !empty($business_hours[$day]['close']) ? $business_hours[$day]['close'] : [];
        foreach ($starts as $idx => $start) {
            $close = isset($closes[$idx]) ? $closes[$idx] : '';
            if (!$start || !$close) continue;

            $slot_date = strtotime("-$offset day", strtotime(date('Y-m-d 00:00:00', $now)));
            $start_ts = strtotime(date('Y-m-d', $slot_date) . ' ' . $start);
            $close_ts = strtotime(date('Y-m-d', $slot_date) . ' ' . $close);
            if ($close_ts <= $start_ts) $close_ts = strtotime('+1 day', $close_ts);

            if ($now >= $start_ts && $now < $close_ts) {
                $is_open = true;
                $current_open = $start_ts;
                $current_close = $close_ts;
                $open_day = $day;
                $close_day = $days[($day_idx + ($close_ts > $start_ts ? 0 : 1)) % 7];
                break 2;
            }
        }
    }

    // 2. Find next open slot (if closed)
    $next_open = null;
    $next_close = null;
    if (!$is_open) {
        $soonest = null;
        for ($i = 0; $i < 7; $i++) {
            $day = $days[($today_idx + $i) % 7];
            
            if (empty($business_hours[$day]['enable'])) continue;
            
            // Check if remain_close is set to 'open' (All Day)
            if ( ! empty( $business_hours[$day]['remain_close'] ) && $business_hours[$day]['remain_close'] === 'open' ) {
                $slot_date = strtotime("+$i day", strtotime(date('Y-m-d 00:00:00', $now)));
                $start_ts = strtotime(date('Y-m-d 00:00:00', $slot_date));
                $close_ts = strtotime(date('Y-m-d 23:59:59', $slot_date));
                
                if (is_null($soonest) || $start_ts < $soonest['start_ts']) {
                    $soonest = [
                        'start_ts' => $start_ts,
                        'close_ts' => $close_ts,
                        'open_day' => $day,
                        'close_day' => $day,
                    ];
                }
                continue;
            }
            
            $starts = !empty($business_hours[$day]['start']) ? $business_hours[$day]['start'] : [];
            $closes = !empty($business_hours[$day]['close']) ? $business_hours[$day]['close'] : [];
            foreach ($starts as $idx => $start) {
                $close = isset($closes[$idx]) ? $closes[$idx] : '';
                if (!$start || !$close) continue;
                $slot_date = strtotime("+$i day", strtotime(date('Y-m-d 00:00:00', $now)));
                $start_ts = strtotime(date('Y-m-d', $slot_date) . ' ' . $start);
                $close_ts = strtotime(date('Y-m-d', $slot_date) . ' ' . $close);
                if ($close_ts <= $start_ts) $close_ts = strtotime('+1 day', $close_ts);

                if ($now < $start_ts && (is_null($soonest) || $start_ts < $soonest['start_ts'])) {
                    $soonest = [
                        'start_ts' => $start_ts,
                        'close_ts' => $close_ts,
                        'open_day' => $day,
                        'close_day' => $days[($today_idx + $i + ($close_ts > $start_ts ? 0 : 1)) % 7],
                    ];
                }
            }
        }
        if ($soonest) {
            $next_open = $soonest['start_ts'];
            $next_close = $soonest['close_ts'];
            $open_day = $soonest['open_day'];
            $close_day = $soonest['close_day'];
        }
    }

    return [
        'is_open' => $is_open,
        'current_open' => $current_open,
        'current_close' => $current_close,
        'next_open' => $next_open,
        'next_close' => $next_close,
        'open_day' => $open_day,
        'close_day' => $close_day,
    ];
}

function directorist_world_timezones() {
    return array(
       'America/Adak' => '(GMT-10:00) America/Adak (Hawaii-Aleutian Standard Time)',
       'America/Atka' => '(GMT-10:00) America/Atka (Hawaii-Aleutian Standard Time)',
       'America/Anchorage' => '(GMT-9:00) America/Anchorage (Alaska Standard Time)',
       'America/Juneau' => '(GMT-9:00) America/Juneau (Alaska Standard Time)',
       'America/Nome' => '(GMT-9:00) America/Nome (Alaska Standard Time)',
       'America/Yakutat' => '(GMT-9:00) America/Yakutat (Alaska Standard Time)',
       'America/Dawson' => '(GMT-8:00) America/Dawson (Pacific Standard Time)',
       'America/Ensenada' => '(GMT-8:00) America/Ensenada (Pacific Standard Time)',
       'America/Los_Angeles' => '(GMT-8:00) America/Los_Angeles (Pacific Standard Time)',
       'America/Tijuana' => '(GMT-8:00) America/Tijuana (Pacific Standard Time)',
       'America/Vancouver' => '(GMT-8:00) America/Vancouver (Pacific Standard Time)',
       'America/Whitehorse' => '(GMT-8:00) America/Whitehorse (Pacific Standard Time)',
       'Canada/Pacific' => '(GMT-8:00) Canada/Pacific (Pacific Standard Time)',
       'Canada/Yukon' => '(GMT-8:00) Canada/Yukon (Pacific Standard Time)',
       'Mexico/BajaNorte' => '(GMT-8:00) Mexico/BajaNorte (Pacific Standard Time)',
       'America/Boise' => '(GMT-7:00) America/Boise (Mountain Standard Time)',
       'America/Cambridge_Bay' => '(GMT-7:00) America/Cambridge_Bay (Mountain Standard Time)',
       'America/Chihuahua' => '(GMT-7:00) America/Chihuahua (Mountain Standard Time)',
       'America/Dawson_Creek' => '(GMT-7:00) America/Dawson_Creek (Mountain Standard Time)',
       'America/Denver' => '(GMT-7:00) America/Denver (Mountain Standard Time)',
       'America/Edmonton' => '(GMT-7:00) America/Edmonton (Mountain Standard Time)',
       'America/Hermosillo' => '(GMT-7:00) America/Hermosillo (Mountain Standard Time)',
       'America/Inuvik' => '(GMT-7:00) America/Inuvik (Mountain Standard Time)',
       'America/Mazatlan' => '(GMT-7:00) America/Mazatlan (Mountain Standard Time)',
       'America/Phoenix' => '(GMT-7:00) America/Phoenix (Mountain Standard Time)',
       'America/Shiprock' => '(GMT-7:00) America/Shiprock (Mountain Standard Time)',
       'America/Yellowknife' => '(GMT-7:00) America/Yellowknife (Mountain Standard Time)',
       'Canada/Mountain' => '(GMT-7:00) Canada/Mountain (Mountain Standard Time)',
       'Mexico/BajaSur' => '(GMT-7:00) Mexico/BajaSur (Mountain Standard Time)',
       'America/Belize' => '(GMT-6:00) America/Belize (Central Standard Time)',
       'America/Cancun' => '(GMT-6:00) America/Cancun (Central Standard Time)',
       'America/Chicago' => '(GMT-6:00) America/Chicago (Central Standard Time)',
       'America/Costa_Rica' => '(GMT-6:00) America/Costa_Rica (Central Standard Time)',
       'America/El_Salvador' => '(GMT-6:00) America/El_Salvador (Central Standard Time)',
       'America/Guatemala' => '(GMT-6:00) America/Guatemala (Central Standard Time)',
       'America/Knox_IN' => '(GMT-6:00) America/Knox_IN (Central Standard Time)',
       'America/Managua' => '(GMT-6:00) America/Managua (Central Standard Time)',
       'America/Menominee' => '(GMT-6:00) America/Menominee (Central Standard Time)',
       'America/Merida' => '(GMT-6:00) America/Merida (Central Standard Time)',
       'America/Mexico_City' => '(GMT-6:00) America/Mexico_City (Central Standard Time)',
       'America/Monterrey' => '(GMT-6:00) America/Monterrey (Central Standard Time)',
       'America/Rainy_River' => '(GMT-6:00) America/Rainy_River (Central Standard Time)',
       'America/Rankin_Inlet' => '(GMT-6:00) America/Rankin_Inlet (Central Standard Time)',
       'America/Regina' => '(GMT-6:00) America/Regina (Central Standard Time)',
       'America/Swift_Current' => '(GMT-6:00) America/Swift_Current (Central Standard Time)',
       'America/Tegucigalpa' => '(GMT-6:00) America/Tegucigalpa (Central Standard Time)',
       'America/Winnipeg' => '(GMT-6:00) America/Winnipeg (Central Standard Time)',
       'Canada/Central' => '(GMT-6:00) Canada/Central (Central Standard Time)',
       'Canada/East-Saskatchewan' => '(GMT-6:00) Canada/East-Saskatchewan (Central Standard Time)',
       'Canada/Saskatchewan' => '(GMT-6:00) Canada/Saskatchewan (Central Standard Time)',
       'Chile/EasterIsland' => '(GMT-6:00) Chile/EasterIsland (Easter Is. Time)',
       'Mexico/General' => '(GMT-6:00) Mexico/General (Central Standard Time)',
       'America/Atikokan' => '(GMT-5:00) America/Atikokan (Eastern Standard Time)',
       'America/Bogota' => '(GMT-5:00) America/Bogota (Colombia Time)',
       'America/Cayman' => '(GMT-5:00) America/Cayman (Eastern Standard Time)',
       'America/Coral_Harbour' => '(GMT-5:00) America/Coral_Harbour (Eastern Standard Time)',
       'America/Detroit' => '(GMT-5:00) America/Detroit (Eastern Standard Time)',
       'America/Fort_Wayne' => '(GMT-5:00) America/Fort_Wayne (Eastern Standard Time)',
       'America/Grand_Turk' => '(GMT-5:00) America/Grand_Turk (Eastern Standard Time)',
       'America/Guayaquil' => '(GMT-5:00) America/Guayaquil (Ecuador Time)',
       'America/Havana' => '(GMT-5:00) America/Havana (Cuba Standard Time)',
       'America/Indianapolis' => '(GMT-5:00) America/Indianapolis (Eastern Standard Time)',
       'America/Iqaluit' => '(GMT-5:00) America/Iqaluit (Eastern Standard Time)',
       'America/Jamaica' => '(GMT-5:00) America/Jamaica (Eastern Standard Time)',
       'America/Lima' => '(GMT-5:00) America/Lima (Peru Time)',
       'America/Louisville' => '(GMT-5:00) America/Louisville (Eastern Standard Time)',
       'America/Montreal' => '(GMT-5:00) America/Montreal (Eastern Standard Time)',
       'America/Nassau' => '(GMT-5:00) America/Nassau (Eastern Standard Time)',
       'America/New_York' => '(GMT-5:00) America/New_York (Eastern Standard Time)',
       'America/Nipigon' => '(GMT-5:00) America/Nipigon (Eastern Standard Time)',
       'America/Panama' => '(GMT-5:00) America/Panama (Eastern Standard Time)',
       'America/Pangnirtung' => '(GMT-5:00) America/Pangnirtung (Eastern Standard Time)',
       'America/Port-au-Prince' => '(GMT-5:00) America/Port-au-Prince (Eastern Standard Time)',
       'America/Resolute' => '(GMT-5:00) America/Resolute (Eastern Standard Time)',
       'America/Thunder_Bay' => '(GMT-5:00) America/Thunder_Bay (Eastern Standard Time)',
       'America/Toronto' => '(GMT-5:00) America/Toronto (Eastern Standard Time)',
       'Canada/Eastern' => '(GMT-5:00) Canada/Eastern (Eastern Standard Time)',
       'America/Caracas' => '(GMT-4:-30) America/Caracas (Venezuela Time)',
       'America/Anguilla' => '(GMT-4:00) America/Anguilla (Atlantic Standard Time)',
       'America/Antigua' => '(GMT-4:00) America/Antigua (Atlantic Standard Time)',
       'America/Aruba' => '(GMT-4:00) America/Aruba (Atlantic Standard Time)',
       'America/Asuncion' => '(GMT-4:00) America/Asuncion (Paraguay Time)',
       'America/Barbados' => '(GMT-4:00) America/Barbados (Atlantic Standard Time)',
       'America/Blanc-Sablon' => '(GMT-4:00) America/Blanc-Sablon (Atlantic Standard Time)',
       'America/Boa_Vista' => '(GMT-4:00) America/Boa_Vista (Amazon Time)',
       'America/Campo_Grande' => '(GMT-4:00) America/Campo_Grande (Amazon Time)',
       'America/Cuiaba' => '(GMT-4:00) America/Cuiaba (Amazon Time)',
       'America/Curacao' => '(GMT-4:00) America/Curacao (Atlantic Standard Time)',
       'America/Dominica' => '(GMT-4:00) America/Dominica (Atlantic Standard Time)',
       'America/Eirunepe' => '(GMT-4:00) America/Eirunepe (Amazon Time)',
       'America/Glace_Bay' => '(GMT-4:00) America/Glace_Bay (Atlantic Standard Time)',
       'America/Goose_Bay' => '(GMT-4:00) America/Goose_Bay (Atlantic Standard Time)',
       'America/Grenada' => '(GMT-4:00) America/Grenada (Atlantic Standard Time)',
       'America/Guadeloupe' => '(GMT-4:00) America/Guadeloupe (Atlantic Standard Time)',
       'America/Guyana' => '(GMT-4:00) America/Guyana (Guyana Time)',
       'America/Halifax' => '(GMT-4:00) America/Halifax (Atlantic Standard Time)',
       'America/La_Paz' => '(GMT-4:00) America/La_Paz (Bolivia Time)',
       'America/Manaus' => '(GMT-4:00) America/Manaus (Amazon Time)',
       'America/Marigot' => '(GMT-4:00) America/Marigot (Atlantic Standard Time)',
       'America/Martinique' => '(GMT-4:00) America/Martinique (Atlantic Standard Time)',
       'America/Moncton' => '(GMT-4:00) America/Moncton (Atlantic Standard Time)',
       'America/Montserrat' => '(GMT-4:00) America/Montserrat (Atlantic Standard Time)',
       'America/Port_of_Spain' => '(GMT-4:00) America/Port_of_Spain (Atlantic Standard Time)',
       'America/Porto_Acre' => '(GMT-4:00) America/Porto_Acre (Amazon Time)',
       'America/Porto_Velho' => '(GMT-4:00) America/Porto_Velho (Amazon Time)',
       'America/Puerto_Rico' => '(GMT-4:00) America/Puerto_Rico (Atlantic Standard Time)',
       'America/Rio_Branco' => '(GMT-4:00) America/Rio_Branco (Amazon Time)',
       'America/Santiago' => '(GMT-4:00) America/Santiago (Chile Time)',
       'America/Santo_Domingo' => '(GMT-4:00) America/Santo_Domingo (Atlantic Standard Time)',
       'America/St_Barthelemy' => '(GMT-4:00) America/St_Barthelemy (Atlantic Standard Time)',
       'America/St_Kitts' => '(GMT-4:00) America/St_Kitts (Atlantic Standard Time)',
       'America/St_Lucia' => '(GMT-4:00) America/St_Lucia (Atlantic Standard Time)',
       'America/St_Thomas' => '(GMT-4:00) America/St_Thomas (Atlantic Standard Time)',
       'America/St_Vincent' => '(GMT-4:00) America/St_Vincent (Atlantic Standard Time)',
       'America/Thule' => '(GMT-4:00) America/Thule (Atlantic Standard Time)',
       'America/Tortola' => '(GMT-4:00) America/Tortola (Atlantic Standard Time)',
       'America/Virgin' => '(GMT-4:00) America/Virgin (Atlantic Standard Time)',
       'Antarctica/Palmer' => '(GMT-4:00) Antarctica/Palmer (Chile Time)',
       'Atlantic/Bermuda' => '(GMT-4:00) Atlantic/Bermuda (Atlantic Standard Time)',
       'Atlantic/Stanley' => '(GMT-4:00) Atlantic/Stanley (Falkland Is. Time)',
       'Brazil/Acre' => '(GMT-4:00) Brazil/Acre (Amazon Time)',
       'Brazil/West' => '(GMT-4:00) Brazil/West (Amazon Time)',
       'Canada/Atlantic' => '(GMT-4:00) Canada/Atlantic (Atlantic Standard Time)',
       'Chile/Continental' => '(GMT-4:00) Chile/Continental (Chile Time)',
       'America/St_Johns' => '(GMT-3:-30) America/St_Johns (Newfoundland Standard Time)',
       'Canada/Newfoundland' => '(GMT-3:-30) Canada/Newfoundland (Newfoundland Standard Time)',
       'America/Araguaina' => '(GMT-3:00) America/Araguaina (Brasilia Time)',
       'America/Bahia' => '(GMT-3:00) America/Bahia (Brasilia Time)',
       'America/Belem' => '(GMT-3:00) America/Belem (Brasilia Time)',
       'America/Buenos_Aires' => '(GMT-3:00) America/Buenos_Aires (Argentine Time)',
       'America/Catamarca' => '(GMT-3:00) America/Catamarca (Argentine Time)',
       'America/Cayenne' => '(GMT-3:00) America/Cayenne (French Guiana Time)',
       'America/Cordoba' => '(GMT-3:00) America/Cordoba (Argentine Time)',
       'America/Fortaleza' => '(GMT-3:00) America/Fortaleza (Brasilia Time)',
       'America/Godthab' => '(GMT-3:00) America/Godthab (Western Greenland Time)',
       'America/Jujuy' => '(GMT-3:00) America/Jujuy (Argentine Time)',
       'America/Maceio' => '(GMT-3:00) America/Maceio (Brasilia Time)',
       'America/Mendoza' => '(GMT-3:00) America/Mendoza (Argentine Time)',
       'America/Miquelon' => '(GMT-3:00) America/Miquelon (Pierre & Miquelon Standard Time)',
       'America/Montevideo' => '(GMT-3:00) America/Montevideo (Uruguay Time)',
       'America/Paramaribo' => '(GMT-3:00) America/Paramaribo (Suriname Time)',
       'America/Recife' => '(GMT-3:00) America/Recife (Brasilia Time)',
       'America/Rosario' => '(GMT-3:00) America/Rosario (Argentine Time)',
       'America/Santarem' => '(GMT-3:00) America/Santarem (Brasilia Time)',
       'America/Sao_Paulo' => '(GMT-3:00) America/Sao_Paulo (Brasilia Time)',
       'Antarctica/Rothera' => '(GMT-3:00) Antarctica/Rothera (Rothera Time)',
       'Brazil/East' => '(GMT-3:00) Brazil/East (Brasilia Time)',
       'America/Noronha' => '(GMT-2:00) America/Noronha (Fernando de Noronha Time)',
       'Atlantic/South_Georgia' => '(GMT-2:00) Atlantic/South_Georgia (South Georgia Standard Time)',
       'Brazil/DeNoronha' => '(GMT-2:00) Brazil/DeNoronha (Fernando de Noronha Time)',
       'America/Scoresbysund' => '(GMT-1:00) America/Scoresbysund (Eastern Greenland Time)',
       'Atlantic/Azores' => '(GMT-1:00) Atlantic/Azores (Azores Time)',
       'Atlantic/Cape_Verde' => '(GMT-1:00) Atlantic/Cape_Verde (Cape Verde Time)',
       'Africa/Abidjan' => '(GMT+0:00) Africa/Abidjan (Greenwich Mean Time)',
       'Africa/Accra' => '(GMT+0:00) Africa/Accra (Ghana Mean Time)',
       'Africa/Bamako' => '(GMT+0:00) Africa/Bamako (Greenwich Mean Time)',
       'Africa/Banjul' => '(GMT+0:00) Africa/Banjul (Greenwich Mean Time)',
       'Africa/Bissau' => '(GMT+0:00) Africa/Bissau (Greenwich Mean Time)',
       'Africa/Casablanca' => '(GMT+0:00) Africa/Casablanca (Western European Time)',
       'Africa/Conakry' => '(GMT+0:00) Africa/Conakry (Greenwich Mean Time)',
       'Africa/Dakar' => '(GMT+0:00) Africa/Dakar (Greenwich Mean Time)',
       'Africa/El_Aaiun' => '(GMT+0:00) Africa/El_Aaiun (Western European Time)',
       'Africa/Freetown' => '(GMT+0:00) Africa/Freetown (Greenwich Mean Time)',
       'Africa/Lome' => '(GMT+0:00) Africa/Lome (Greenwich Mean Time)',
       'Africa/Monrovia' => '(GMT+0:00) Africa/Monrovia (Greenwich Mean Time)',
       'Africa/Nouakchott' => '(GMT+0:00) Africa/Nouakchott (Greenwich Mean Time)',
       'Africa/Ouagadougou' => '(GMT+0:00) Africa/Ouagadougou (Greenwich Mean Time)',
       'Africa/Sao_Tome' => '(GMT+0:00) Africa/Sao_Tome (Greenwich Mean Time)',
       'Africa/Timbuktu' => '(GMT+0:00) Africa/Timbuktu (Greenwich Mean Time)',
       'America/Danmarkshavn' => '(GMT+0:00) America/Danmarkshavn (Greenwich Mean Time)',
       'Atlantic/Canary' => '(GMT+0:00) Atlantic/Canary (Western European Time)',
       'Atlantic/Faeroe' => '(GMT+0:00) Atlantic/Faeroe (Western European Time)',
       'Atlantic/Faroe' => '(GMT+0:00) Atlantic/Faroe (Western European Time)',
       'Atlantic/Madeira' => '(GMT+0:00) Atlantic/Madeira (Western European Time)',
       'Atlantic/Reykjavik' => '(GMT+0:00) Atlantic/Reykjavik (Greenwich Mean Time)',
       'Atlantic/St_Helena' => '(GMT+0:00) Atlantic/St_Helena (Greenwich Mean Time)',
       'Europe/Belfast' => '(GMT+0:00) Europe/Belfast (Greenwich Mean Time)',
       'Europe/Dublin' => '(GMT+0:00) Europe/Dublin (Greenwich Mean Time)',
       'Europe/Guernsey' => '(GMT+0:00) Europe/Guernsey (Greenwich Mean Time)',
       'Europe/Isle_of_Man' => '(GMT+0:00) Europe/Isle_of_Man (Greenwich Mean Time)',
       'Europe/Jersey' => '(GMT+0:00) Europe/Jersey (Greenwich Mean Time)',
       'Europe/Lisbon' => '(GMT+0:00) Europe/Lisbon (Western European Time)',
       'Europe/London' => '(GMT+0:00) Europe/London (Greenwich Mean Time)',
       'Africa/Algiers' => '(GMT+1:00) Africa/Algiers (Central European Time)',
       'Africa/Bangui' => '(GMT+1:00) Africa/Bangui (Western African Time)',
       'Africa/Brazzaville' => '(GMT+1:00) Africa/Brazzaville (Western African Time)',
       'Africa/Ceuta' => '(GMT+1:00) Africa/Ceuta (Central European Time)',
       'Africa/Douala' => '(GMT+1:00) Africa/Douala (Western African Time)',
       'Africa/Kinshasa' => '(GMT+1:00) Africa/Kinshasa (Western African Time)',
       'Africa/Lagos' => '(GMT+1:00) Africa/Lagos (Western African Time)',
       'Africa/Libreville' => '(GMT+1:00) Africa/Libreville (Western African Time)',
       'Africa/Luanda' => '(GMT+1:00) Africa/Luanda (Western African Time)',
       'Africa/Malabo' => '(GMT+1:00) Africa/Malabo (Western African Time)',
       'Africa/Ndjamena' => '(GMT+1:00) Africa/Ndjamena (Western African Time)',
       'Africa/Niamey' => '(GMT+1:00) Africa/Niamey (Western African Time)',
       'Africa/Porto-Novo' => '(GMT+1:00) Africa/Porto-Novo (Western African Time)',
       'Africa/Tunis' => '(GMT+1:00) Africa/Tunis (Central European Time)',
       'Africa/Windhoek' => '(GMT+1:00) Africa/Windhoek (Western African Time)',
       'Arctic/Longyearbyen' => '(GMT+1:00) Arctic/Longyearbyen (Central European Time)',
       'Atlantic/Jan_Mayen' => '(GMT+1:00) Atlantic/Jan_Mayen (Central European Time)',
       'Europe/Amsterdam' => '(GMT+1:00) Europe/Amsterdam (Central European Time)',
       'Europe/Andorra' => '(GMT+1:00) Europe/Andorra (Central European Time)',
       'Europe/Belgrade' => '(GMT+1:00) Europe/Belgrade (Central European Time)',
       'Europe/Berlin' => '(GMT+1:00) Europe/Berlin (Central European Time)',
       'Europe/Bratislava' => '(GMT+1:00) Europe/Bratislava (Central European Time)',
       'Europe/Brussels' => '(GMT+1:00) Europe/Brussels (Central European Time)',
       'Europe/Budapest' => '(GMT+1:00) Europe/Budapest (Central European Time)',
       'Europe/Copenhagen' => '(GMT+1:00) Europe/Copenhagen (Central European Time)',
       'Europe/Gibraltar' => '(GMT+1:00) Europe/Gibraltar (Central European Time)',
       'Europe/Ljubljana' => '(GMT+1:00) Europe/Ljubljana (Central European Time)',
       'Europe/Luxembourg' => '(GMT+1:00) Europe/Luxembourg (Central European Time)',
       'Europe/Madrid' => '(GMT+1:00) Europe/Madrid (Central European Time)',
       'Europe/Malta' => '(GMT+1:00) Europe/Malta (Central European Time)',
       'Europe/Monaco' => '(GMT+1:00) Europe/Monaco (Central European Time)',
       'Europe/Oslo' => '(GMT+1:00) Europe/Oslo (Central European Time)',
       'Europe/Paris' => '(GMT+1:00) Europe/Paris (Central European Time)',
       'Europe/Podgorica' => '(GMT+1:00) Europe/Podgorica (Central European Time)',
       'Europe/Prague' => '(GMT+1:00) Europe/Prague (Central European Time)',
       'Europe/Rome' => '(GMT+1:00) Europe/Rome (Central European Time)',
       'Europe/San_Marino' => '(GMT+1:00) Europe/San_Marino (Central European Time)',
       'Europe/Sarajevo' => '(GMT+1:00) Europe/Sarajevo (Central European Time)',
       'Europe/Skopje' => '(GMT+1:00) Europe/Skopje (Central European Time)',
       'Europe/Stockholm' => '(GMT+1:00) Europe/Stockholm (Central European Time)',
       'Europe/Tirane' => '(GMT+1:00) Europe/Tirane (Central European Time)',
       'Europe/Vaduz' => '(GMT+1:00) Europe/Vaduz (Central European Time)',
       'Europe/Vatican' => '(GMT+1:00) Europe/Vatican (Central European Time)',
       'Europe/Vienna' => '(GMT+1:00) Europe/Vienna (Central European Time)',
       'Europe/Warsaw' => '(GMT+1:00) Europe/Warsaw (Central European Time)',
       'Europe/Zagreb' => '(GMT+1:00) Europe/Zagreb (Central European Time)',
       'Europe/Zurich' => '(GMT+1:00) Europe/Zurich (Central European Time)',
       'Africa/Blantyre' => '(GMT+2:00) Africa/Blantyre (Central African Time)',
       'Africa/Bujumbura' => '(GMT+2:00) Africa/Bujumbura (Central African Time)',
       'Africa/Cairo' => '(GMT+2:00) Africa/Cairo (Eastern European Time)',
       'Africa/Gaborone' => '(GMT+2:00) Africa/Gaborone (Central African Time)',
       'Africa/Harare' => '(GMT+2:00) Africa/Harare (Central African Time)',
       'Africa/Johannesburg' => '(GMT+2:00) Africa/Johannesburg (South Africa Standard Time)',
       'Africa/Kigali' => '(GMT+2:00) Africa/Kigali (Central African Time)',
       'Africa/Lubumbashi' => '(GMT+2:00) Africa/Lubumbashi (Central African Time)',
       'Africa/Lusaka' => '(GMT+2:00) Africa/Lusaka (Central African Time)',
       'Africa/Maputo' => '(GMT+2:00) Africa/Maputo (Central African Time)',
       'Africa/Maseru' => '(GMT+2:00) Africa/Maseru (South Africa Standard Time)',
       'Africa/Mbabane' => '(GMT+2:00) Africa/Mbabane (South Africa Standard Time)',
       'Africa/Tripoli' => '(GMT+2:00) Africa/Tripoli (Eastern European Time)',
       'Asia/Amman' => '(GMT+2:00) Asia/Amman (Eastern European Time)',
       'Asia/Beirut' => '(GMT+2:00) Asia/Beirut (Eastern European Time)',
       'Asia/Damascus' => '(GMT+2:00) Asia/Damascus (Eastern European Time)',
       'Asia/Gaza' => '(GMT+2:00) Asia/Gaza (Eastern European Time)',
       'Asia/Istanbul' => '(GMT+2:00) Asia/Istanbul (Eastern European Time)',
       'Asia/Jerusalem' => '(GMT+2:00) Asia/Jerusalem (Israel Standard Time)',
       'Asia/Nicosia' => '(GMT+2:00) Asia/Nicosia (Eastern European Time)',
       'Asia/Tel_Aviv' => '(GMT+2:00) Asia/Tel_Aviv (Israel Standard Time)',
       'Europe/Athens' => '(GMT+2:00) Europe/Athens (Eastern European Time)',
       'Europe/Bucharest' => '(GMT+2:00) Europe/Bucharest (Eastern European Time)',
       'Europe/Chisinau' => '(GMT+2:00) Europe/Chisinau (Eastern European Time)',
       'Europe/Helsinki' => '(GMT+2:00) Europe/Helsinki (Eastern European Time)',
       'Europe/Istanbul' => '(GMT+2:00) Europe/Istanbul (Eastern European Time)',
       'Europe/Kaliningrad' => '(GMT+2:00) Europe/Kaliningrad (Eastern European Time)',
       'Europe/Kiev' => '(GMT+2:00) Europe/Kiev (Eastern European Time)',
       'Europe/Mariehamn' => '(GMT+2:00) Europe/Mariehamn (Eastern European Time)',
       'Europe/Minsk' => '(GMT+2:00) Europe/Minsk (Eastern European Time)',
       'Europe/Nicosia' => '(GMT+2:00) Europe/Nicosia (Eastern European Time)',
       'Europe/Riga' => '(GMT+2:00) Europe/Riga (Eastern European Time)',
       'Europe/Simferopol' => '(GMT+2:00) Europe/Simferopol (Eastern European Time)',
       'Europe/Sofia' => '(GMT+2:00) Europe/Sofia (Eastern European Time)',
       'Europe/Tallinn' => '(GMT+2:00) Europe/Tallinn (Eastern European Time)',
       'Europe/Tiraspol' => '(GMT+2:00) Europe/Tiraspol (Eastern European Time)',
       'Europe/Uzhgorod' => '(GMT+2:00) Europe/Uzhgorod (Eastern European Time)',
       'Europe/Vilnius' => '(GMT+2:00) Europe/Vilnius (Eastern European Time)',
       'Europe/Zaporozhye' => '(GMT+2:00) Europe/Zaporozhye (Eastern European Time)',
       'Africa/Addis_Ababa' => '(GMT+3:00) Africa/Addis_Ababa (Eastern African Time)',
       'Africa/Asmara' => '(GMT+3:00) Africa/Asmara (Eastern African Time)',
       'Africa/Asmera' => '(GMT+3:00) Africa/Asmera (Eastern African Time)',
       'Africa/Dar_es_Salaam' => '(GMT+3:00) Africa/Dar_es_Salaam (Eastern African Time)',
       'Africa/Djibouti' => '(GMT+3:00) Africa/Djibouti (Eastern African Time)',
       'Africa/Kampala' => '(GMT+3:00) Africa/Kampala (Eastern African Time)',
       'Africa/Khartoum' => '(GMT+3:00) Africa/Khartoum (Eastern African Time)',
       'Africa/Mogadishu' => '(GMT+3:00) Africa/Mogadishu (Eastern African Time)',
       'Africa/Nairobi' => '(GMT+3:00) Africa/Nairobi (Eastern African Time)',
       'Antarctica/Syowa' => '(GMT+3:00) Antarctica/Syowa (Syowa Time)',
       'Asia/Aden' => '(GMT+3:00) Asia/Aden (Arabia Standard Time)',
       'Asia/Baghdad' => '(GMT+3:00) Asia/Baghdad (Arabia Standard Time)',
       'Asia/Bahrain' => '(GMT+3:00) Asia/Bahrain (Arabia Standard Time)',
       'Asia/Kuwait' => '(GMT+3:00) Asia/Kuwait (Arabia Standard Time)',
       'Asia/Qatar' => '(GMT+3:00) Asia/Qatar (Arabia Standard Time)',
       'Europe/Moscow' => '(GMT+3:00) Europe/Moscow (Moscow Standard Time)',
       'Europe/Volgograd' => '(GMT+3:00) Europe/Volgograd (Volgograd Time)',
       'Indian/Antananarivo' => '(GMT+3:00) Indian/Antananarivo (Eastern African Time)',
       'Indian/Comoro' => '(GMT+3:00) Indian/Comoro (Eastern African Time)',
       'Indian/Mayotte' => '(GMT+3:00) Indian/Mayotte (Eastern African Time)',
       'Asia/Tehran' => '(GMT+3:30) Asia/Tehran (Iran Standard Time)',
       'Asia/Baku' => '(GMT+4:00) Asia/Baku (Azerbaijan Time)',
       'Asia/Dubai' => '(GMT+4:00) Asia/Dubai (Gulf Standard Time)',
       'Asia/Muscat' => '(GMT+4:00) Asia/Muscat (Gulf Standard Time)',
       'Asia/Tbilisi' => '(GMT+4:00) Asia/Tbilisi (Georgia Time)',
       'Asia/Yerevan' => '(GMT+4:00) Asia/Yerevan (Armenia Time)',
       'Europe/Samara' => '(GMT+4:00) Europe/Samara (Samara Time)',
       'Indian/Mahe' => '(GMT+4:00) Indian/Mahe (Seychelles Time)',
       'Indian/Mauritius' => '(GMT+4:00) Indian/Mauritius (Mauritius Time)',
       'Indian/Reunion' => '(GMT+4:00) Indian/Reunion (Reunion Time)',
       'Asia/Kabul' => '(GMT+4:30) Asia/Kabul (Afghanistan Time)',
       'Asia/Aqtau' => '(GMT+5:00) Asia/Aqtau (Aqtau Time)',
       'Asia/Aqtobe' => '(GMT+5:00) Asia/Aqtobe (Aqtobe Time)',
       'Asia/Ashgabat' => '(GMT+5:00) Asia/Ashgabat (Turkmenistan Time)',
       'Asia/Ashkhabad' => '(GMT+5:00) Asia/Ashkhabad (Turkmenistan Time)',
       'Asia/Dushanbe' => '(GMT+5:00) Asia/Dushanbe (Tajikistan Time)',
       'Asia/Karachi' => '(GMT+5:00) Asia/Karachi (Pakistan Time)',
       'Asia/Oral' => '(GMT+5:00) Asia/Oral (Oral Time)',
       'Asia/Samarkand' => '(GMT+5:00) Asia/Samarkand (Uzbekistan Time)',
       'Asia/Tashkent' => '(GMT+5:00) Asia/Tashkent (Uzbekistan Time)',
       'Asia/Yekaterinburg' => '(GMT+5:00) Asia/Yekaterinburg (Yekaterinburg Time)',
       'Indian/Kerguelen' => '(GMT+5:00) Indian/Kerguelen (French Southern & Antarctic Lands Time)',
       'Indian/Maldives' => '(GMT+5:00) Indian/Maldives (Maldives Time)',
       'Asia/Calcutta' => '(GMT+5:30) Asia/Calcutta (India Standard Time)',
       'Asia/Colombo' => '(GMT+5:30) Asia/Colombo (India Standard Time)',
       'Asia/Kolkata' => '(GMT+5:30) Asia/Kolkata (India Standard Time)',
       'Asia/Katmandu' => '(GMT+5:45) Asia/Katmandu (Nepal Time)',
       'Antarctica/Mawson' => '(GMT+6:00) Antarctica/Mawson (Mawson Time)',
       'Antarctica/Vostok' => '(GMT+6:00) Antarctica/Vostok (Vostok Time)',
       'Asia/Almaty' => '(GMT+6:00) Asia/Almaty (Alma-Ata Time)',
       'Asia/Bishkek' => '(GMT+6:00) Asia/Bishkek (Kirgizstan Time)',
       'Asia/Dacca' => '(GMT+6:00) Asia/Dacca (Bangladesh Time)',
       'Asia/Dhaka' => '(GMT+6:00) Asia/Dhaka (Bangladesh Time)',
       'Asia/Novosibirsk' => '(GMT+6:00) Asia/Novosibirsk (Novosibirsk Time)',
       'Asia/Omsk' => '(GMT+6:00) Asia/Omsk (Omsk Time)',
       'Asia/Qyzylorda' => '(GMT+6:00) Asia/Qyzylorda (Qyzylorda Time)',
       'Asia/Thimbu' => '(GMT+6:00) Asia/Thimbu (Bhutan Time)',
       'Asia/Thimphu' => '(GMT+6:00) Asia/Thimphu (Bhutan Time)',
       'Indian/Chagos' => '(GMT+6:00) Indian/Chagos (Indian Ocean Territory Time)',
       'Asia/Rangoon' => '(GMT+6:30) Asia/Rangoon (Myanmar Time)',
       'Indian/Cocos' => '(GMT+6:30) Indian/Cocos (Cocos Islands Time)',
       'Antarctica/Davis' => '(GMT+7:00) Antarctica/Davis (Davis Time)',
       'Asia/Bangkok' => '(GMT+7:00) Asia/Bangkok (Indochina Time)',
       'Asia/Ho_Chi_Minh' => '(GMT+7:00) Asia/Ho_Chi_Minh (Indochina Time)',
       'Asia/Hovd' => '(GMT+7:00) Asia/Hovd (Hovd Time)',
       'Asia/Jakarta' => '(GMT+7:00) Asia/Jakarta (West Indonesia Time)',
       'Asia/Krasnoyarsk' => '(GMT+7:00) Asia/Krasnoyarsk (Krasnoyarsk Time)',
       'Asia/Phnom_Penh' => '(GMT+7:00) Asia/Phnom_Penh (Indochina Time)',
       'Asia/Pontianak' => '(GMT+7:00) Asia/Pontianak (West Indonesia Time)',
       'Asia/Saigon' => '(GMT+7:00) Asia/Saigon (Indochina Time)',
       'Asia/Vientiane' => '(GMT+7:00) Asia/Vientiane (Indochina Time)',
       'Indian/Christmas' => '(GMT+7:00) Indian/Christmas (Christmas Island Time)',
       'Antarctica/Casey' => '(GMT+8:00) Antarctica/Casey (Western Standard Time (Australia))',
       'Asia/Brunei' => '(GMT+8:00) Asia/Brunei (Brunei Time)',
       'Asia/Choibalsan' => '(GMT+8:00) Asia/Choibalsan (Choibalsan Time)',
       'Asia/Chongqing' => '(GMT+8:00) Asia/Chongqing (China Standard Time)',
       'Asia/Chungking' => '(GMT+8:00) Asia/Chungking (China Standard Time)',
       'Asia/Harbin' => '(GMT+8:00) Asia/Harbin (China Standard Time)',
       'Asia/Hong_Kong' => '(GMT+8:00) Asia/Hong_Kong (Hong Kong Time)',
       'Asia/Irkutsk' => '(GMT+8:00) Asia/Irkutsk (Irkutsk Time)',
       'Asia/Kashgar' => '(GMT+8:00) Asia/Kashgar (China Standard Time)',
       'Asia/Kuala_Lumpur' => '(GMT+8:00) Asia/Kuala_Lumpur (Malaysia Time)',
       'Asia/Kuching' => '(GMT+8:00) Asia/Kuching (Malaysia Time)',
       'Asia/Macao' => '(GMT+8:00) Asia/Macao (China Standard Time)',
       'Asia/Macau' => '(GMT+8:00) Asia/Macau (China Standard Time)',
       'Asia/Makassar' => '(GMT+8:00) Asia/Makassar (Central Indonesia Time)',
       'Asia/Manila' => '(GMT+8:00) Asia/Manila (Philippines Time)',
       'Asia/Shanghai' => '(GMT+8:00) Asia/Shanghai (China Standard Time)',
       'Asia/Singapore' => '(GMT+8:00) Asia/Singapore (Singapore Time)',
       'Asia/Taipei' => '(GMT+8:00) Asia/Taipei (China Standard Time)',
       'Asia/Ujung_Pandang' => '(GMT+8:00) Asia/Ujung_Pandang (Central Indonesia Time)',
       'Asia/Ulaanbaatar' => '(GMT+8:00) Asia/Ulaanbaatar (Ulaanbaatar Time)',
       'Asia/Ulan_Bator' => '(GMT+8:00) Asia/Ulan_Bator (Ulaanbaatar Time)',
       'Asia/Urumqi' => '(GMT+8:00) Asia/Urumqi (China Standard Time)',
       'Australia/Perth' => '(GMT+8:00) Australia/Perth (Western Standard Time (Australia))',
       'Australia/West' => '(GMT+8:00) Australia/West (Western Standard Time (Australia))',
       'Australia/Eucla' => '(GMT+8:45) Australia/Eucla (Central Western Standard Time (Australia))',
       'Asia/Dili' => '(GMT+9:00) Asia/Dili (Timor-Leste Time)',
       'Asia/Jayapura' => '(GMT+9:00) Asia/Jayapura (East Indonesia Time)',
       'Asia/Pyongyang' => '(GMT+9:00) Asia/Pyongyang (Korea Standard Time)',
       'Asia/Seoul' => '(GMT+9:00) Asia/Seoul (Korea Standard Time)',
       'Asia/Tokyo' => '(GMT+9:00) Asia/Tokyo (Japan Standard Time)',
       'Asia/Yakutsk' => '(GMT+9:00) Asia/Yakutsk (Yakutsk Time)',
       'Australia/Adelaide' => '(GMT+9:30) Australia/Adelaide (Central Standard Time (South Australia))',
       'Australia/Broken_Hill' => '(GMT+9:30) Australia/Broken_Hill (Central Standard Time (South Australia/New South Wales))',
       'Australia/Darwin' => '(GMT+9:30) Australia/Darwin (Central Standard Time (Northern Territory))',
       'Australia/North' => '(GMT+9:30) Australia/North (Central Standard Time (Northern Territory))',
       'Australia/South' => '(GMT+9:30) Australia/South (Central Standard Time (South Australia))',
       'Australia/Yancowinna' => '(GMT+9:30) Australia/Yancowinna (Central Standard Time (South Australia/New South Wales))',
       'Antarctica/DumontDUrville' => '(GMT+10:00) Antarctica/DumontDUrville (Dumont-d\'Urville Time)',
       'Asia/Sakhalin' => '(GMT+10:00) Asia/Sakhalin (Sakhalin Time)',
       'Asia/Vladivostok' => '(GMT+10:00) Asia/Vladivostok (Vladivostok Time)',
       'Australia/ACT' => '(GMT+10:00) Australia/ACT (Eastern Standard Time (New South Wales))',
       'Australia/Brisbane' => '(GMT+10:00) Australia/Brisbane (Eastern Standard Time (Queensland))',
       'Australia/Canberra' => '(GMT+10:00) Australia/Canberra (Eastern Standard Time (New South Wales))',
       'Australia/Currie' => '(GMT+10:00) Australia/Currie (Eastern Standard Time (New South Wales))',
       'Australia/Hobart' => '(GMT+10:00) Australia/Hobart (Eastern Standard Time (Tasmania))',
       'Australia/Lindeman' => '(GMT+10:00) Australia/Lindeman (Eastern Standard Time (Queensland))',
       'Australia/Melbourne' => '(GMT+10:00) Australia/Melbourne (Eastern Standard Time (Victoria))',
       'Australia/NSW' => '(GMT+10:00) Australia/NSW (Eastern Standard Time (New South Wales))',
       'Australia/Queensland' => '(GMT+10:00) Australia/Queensland (Eastern Standard Time (Queensland))',
       'Australia/Sydney' => '(GMT+10:00) Australia/Sydney (Eastern Standard Time (New South Wales))',
       'Australia/Tasmania' => '(GMT+10:00) Australia/Tasmania (Eastern Standard Time (Tasmania))',
       'Australia/Victoria' => '(GMT+10:00) Australia/Victoria (Eastern Standard Time (Victoria))',
       'Australia/LHI' => '(GMT+10:30) Australia/LHI (Lord Howe Standard Time)',
       'Australia/Lord_Howe' => '(GMT+10:30) Australia/Lord_Howe (Lord Howe Standard Time)',
       'Asia/Magadan' => '(GMT+11:00) Asia/Magadan (Magadan Time)',
       'Antarctica/McMurdo' => '(GMT+12:00) Antarctica/McMurdo (New Zealand Standard Time)',
       'Antarctica/South_Pole' => '(GMT+12:00) Antarctica/South_Pole (New Zealand Standard Time)',
       'Asia/Anadyr' => '(GMT+12:00) Asia/Anadyr (Anadyr Time)',
       'Asia/Kamchatka' => '(GMT+12:00) Asia/Kamchatka (Petropavlovsk-Kamchatski Time)'
    );
}

/**
 * 
 * @return bool If cache_plugin_compatibility option is enabled or disabled
 * @since  2.3.0
 */

if ( ! function_exists( 'directorist_hours_cache_plugin_compatibility' ) ) {

    function directorist_hours_cache_plugin_compatibility() {

        return get_directorist_option('cache_plugin_compatibility', false );

    }
}

if ( ! function_exists( 'directorist_is_business_hour_enabled_for_listing' ) ) {
    /**
     * Check if Directorist Business Hours are enabled for a listing.
     *
     * @param int $listing_id The listing ID.
     * @return bool True if enabled, false otherwise.
     */
    function directorist_is_business_hour_enabled_for_listing( $listing_id ) {
        $enable_directorist_bh_listing = get_post_meta( $listing_id, '_enable_bz_hour_listing', true );
        $bdbh_version                  = get_post_meta( $listing_id, '_bdbh_version', true );
        $disable_business_hours        = get_post_meta( $listing_id, '_disable_bz_hour_listing', true );

        return ! empty( $enable_directorist_bh_listing ) || ( ( ! empty( $bdbh_version ) && 'update' == $bdbh_version ) && empty( $disable_business_hours ) );
    }
}

function directorist_get_wp_default_timezone_identifier( $fallback = 'America/New_York' ) {
    $timezone = get_option('timezone_string');

    if ( !empty($timezone) && in_array($timezone, DateTimeZone::listIdentifiers(), true) ) {
        return $timezone;
    }

    // যদি timezone_string খালি হয়, তাহলে gmt_offset থেকে চেষ্টা করো
    $gmt_offset = (float) get_option('gmt_offset');

    if ( is_null($gmt_offset) ) {
        return $fallback;
    }

    return directorist_convert_gmt_offset_to_timezone( $gmt_offset, $fallback );
}

function directorist_convert_gmt_offset_to_timezone( $gmt_offset, $fallback = 'America/New_York' ) {
    // common mapping (rounded to nearest 15 minutes)
    $common_mappings = [
        '-12.0' => 'Pacific/Kwajalein',
        '-11.0' => 'Pacific/Midway',
        '-10.0' => 'Pacific/Honolulu',
        '-9.0'  => 'America/Anchorage',
        '-8.0'  => 'America/Los_Angeles',
        '-7.0'  => 'America/Denver',
        '-6.0'  => 'America/Chicago',
        '-5.0'  => 'America/New_York',
        '-4.0'  => 'America/Halifax',
        '-3.5'  => 'America/St_Johns',
        '-3.0'  => 'America/Sao_Paulo',
        '-2.0'  => 'Atlantic/South_Georgia',
        '-1.0'  => 'Atlantic/Azores',
        '0.0'   => 'UTC',
        '1.0'   => 'Europe/Paris',
        '2.0'   => 'Europe/Berlin',
        '3.0'   => 'Europe/Moscow',
        '3.5'   => 'Asia/Tehran',
        '4.0'   => 'Asia/Dubai',
        '4.5'   => 'Asia/Kabul',
        '5.0'   => 'Asia/Tashkent',
        '5.5'   => 'Asia/Kolkata',
        '5.75'  => 'Asia/Kathmandu',
        '6.0'   => 'Asia/Dhaka',
        '6.5'   => 'Asia/Rangoon',
        '7.0'   => 'Asia/Bangkok',
        '8.0'   => 'Asia/Shanghai',
        '8.75'  => 'Australia/Eucla',
        '9.0'   => 'Asia/Tokyo',
        '9.5'   => 'Australia/Adelaide',
        '10.0'  => 'Australia/Sydney',
        '10.5'  => 'Australia/Lord_Howe',
        '11.0'  => 'Pacific/Guadalcanal',
        '11.5'  => 'Pacific/Norfolk',
        '12.0'  => 'Pacific/Auckland',
        '12.75' => 'Pacific/Chatham',
        '13.0'  => 'Pacific/Tongatapu',
        '14.0'  => 'Pacific/Kiritimati',
    ];

    $rounded_offset = round( $gmt_offset * 4 ) / 4;
    $rounded_offset_str = (string)$rounded_offset;

    if ( isset( $common_mappings[ $rounded_offset_str ] ) ) {
        return $common_mappings[ $rounded_offset_str ];
    }

    return $fallback;
}
