<?php

if ( ! class_exists( 'Directorist_Gallery_Data_Importer' ) ):
	class Directorist_Gallery_Data_Importer {

		/**
		 * Constructor - Initializes hooks and schedules.
		 */
		public function __construct() {
			// Hook after listings import
			add_action( 'directorist_after_listings_import', [$this, 'enable_gallery_import'] );

			// Register custom schedule
			add_action( 'init', [$this, 'register_gallery_import_schedule'] );

			// Cron task callback
			add_action( 'directorist_gallery_import_cron', [$this, 'import_gallery_images'] );
		}

		/**
		 * Enable the gallery importer.
		 */
		public function enable_gallery_import() {
			update_option( 'directorist_gallery_importer_run', 'yes' );
		}

		/**
		 * Register a custom schedule for the gallery import task.
		 */
		public function register_gallery_import_schedule() {
			// Add custom interval for per-second schedule
			add_filter( 'cron_schedules', function ( $schedules ) {
				$schedules['per_second'] = [
					'interval' => 1, // Every 1 second
					'display'  => __( 'Every Second' ),
				];

				return $schedules;
			} );

			// Schedule the task if not already scheduled
			if ( ! wp_next_scheduled( 'directorist_gallery_import_cron' ) ) {
				wp_schedule_event( time(), 'per_second', 'directorist_gallery_import_cron' );
			}
		}

		/**
		 * The callback for the scheduled gallery import task.
		 */
		public function import_gallery_images() {

			// Check if the importer is enabled
			$is_importer_enabled = get_option( 'directorist_gallery_importer_run', 'no' );

			if ( 'yes' !== $is_importer_enabled ) {
				return; // Skip if not enabled
			}

			// Fetch listings to process
			$listings = $this->get_listings_for_gallery_import();

			if ( empty( $listings ) ) {
				update_option( 'directorist_gallery_importer_run', 'no' ); // Disable importer if no listings
				return;
			}

			// Process up to 10 listings
			foreach ( $listings as $listing ) {

				$gallery_urls = get_post_meta( $listing->ID, '_gallery_img', true );
				update_post_meta( $listing->ID, '_gallery_image_urls', true );

				// Skip invalid or empty data
				if ( empty( $gallery_urls ) ) {
					delete_post_meta( $listing->ID, '_directorist_imported_by_csv' );
					continue;
				}

				// Check if the data is comma-separated or a single URL
				if ( strpos( $gallery_urls, ',' ) !== false ) {
					// Comma-separated data
					$plain_urls_arr = explode( ',', $gallery_urls );
				} else {
					// Single URL
					$plain_urls_arr = [$gallery_urls];
				}

				// Validate the array
				if ( empty( $plain_urls_arr ) || ! is_array( $plain_urls_arr ) ) {
					delete_post_meta( $listing->ID, '_directorist_imported_by_csv' );
					continue;
				}

				// Process each URL
				$attachment_ids = [];
				foreach ( $plain_urls_arr as $url ) {
					$attachment_id = $this->insert_attachment_from_url( $url, $listing->ID );
					if ( $attachment_id ) {
						$attachment_ids[] = $attachment_id;
					}
				}

				if ( ! empty( $attachment_ids ) ) {
					update_post_meta( $listing->ID, '_gallery_img', $attachment_ids );
				}

				delete_post_meta( $listing->ID, '_directorist_imported_by_csv' );
			}
		}

		/**
		 * Fetch listings that need gallery import.
		 *
		 * @return array
		 */
		private function get_listings_for_gallery_import() {
			$args = [
				'post_type'      => 'at_biz_dir',
				'post_status'    => 'any',
				'meta_query'     => [
					[
						'key'     => '_directorist_imported_by_csv',
						'value'   => 'yes',
						'compare' => '=',
					],
				],
				'posts_per_page' => 10, // Limit to 10 listings per run
			];

			return get_posts( $args );
		}

		/**
		 * Insert an attachment from a URL.
		 *
		 * @param string $url     The image URL.
		 * @param int    $post_id The ID of the post to attach the image to.
		 * @return int|false Attachment ID or false on failure.
		 */
		private function insert_attachment_from_url( $url, $post_id ) {
			// Include required WordPress files
			if ( ! function_exists( 'download_url' ) ) {
				require_once ABSPATH . 'wp-admin/includes/file.php';
			}
			if ( ! function_exists( 'media_handle_sideload' ) ) {
				require_once ABSPATH . 'wp-admin/includes/media.php';
			}

			if ( ! function_exists( 'wp_read_image_metadata' ) ) {
				require_once ABSPATH . 'wp-admin/includes/image.php';
			}

			// Download the file temporarily
			$tmp = download_url( $url );
			if ( is_wp_error( $tmp ) ) {
				error_log( 'Error downloading URL: ' . $tmp->get_error_message() );

				return false;
			}

			// Extract filename from URL or set a default name
			$file_name = basename( parse_url( $url, PHP_URL_PATH ) );
			if ( empty( $file_name ) ) {
				$file_name = 'downloaded_image.jpg'; // Default file name if none is detected
			}

			// Ensure file has an image extension
			$path_info = pathinfo( $file_name );
			if ( empty( $path_info['extension'] ) || ! in_array( strtolower( $path_info['extension'] ), ['jpg', 'jpeg', 'png', 'gif', 'webp'] ) ) {
				$file_name .= '.jpg'; // Default to .jpg if no valid extension is found
			}

			// Prepare file array for media upload
			$file = [
				'name'     => $file_name,
				'tmp_name' => $tmp,
			];

			// Handle the sideloaded media
			$attachment_id = media_handle_sideload( $file, $post_id );

			// Check for errors and clean up temporary file
			if ( is_wp_error( $attachment_id ) ) {
				error_log( 'Error inserting attachment: ' . $attachment_id->get_error_message() );
				@unlink( $tmp );

				return false;
			}

			return $attachment_id;
		}
	}

endif;