<?php
/**
 * Plugin Name: Directorist - WooCommerce Pricing Plans
 * Requires Plugins: directorist, woocommerce
 * Plugin URI: https://directorist.com/product/directorist-woocommerce-pricing-plans
 * Description: Monetize your directory by integrating WooCommerce-based pricing plans for enhanced flexibility.
 * Version: 3.6.7
 * Author: wpWax
 * Author URI: https://wpwax.com
 * License: GPLv2 or later
 * WC requires at least: 3.6
 * WC tested up to: 8.3
 * Text Domain: directorist-woocommerce-pricing-plans
 * Domain Path: /languages
 */
// HPOS https://github.com/woocommerce/woocommerce/wiki/High-Performance-Order-Storage-Upgrade-Recipe-Book#supporting-high-performance-order-storage
// prevent direct access to the file
defined( 'ABSPATH' ) || die( 'No direct script access allowed!' );

if ( ! class_exists( 'ATBDP_Pricing_Plans' ) && ! class_exists( 'DWPP_Pricing_Plans' ) ) {
    final class DWPP_Pricing_Plans {
        /**
         * @var DWPP_Pricing_Plans The one true DWPP_Pricing_Plans
         * @since 1.0
         */
        private static $instance;

        private static $plan_id;

        /**
         * Main DWPP_Pricing_Plans Instance.
         *
         * Insures that only one instance of DWPP_Pricing_Plans exists in memory at any one
         * time. Also prevents needing to define globals all over the place.
         *
         * @return object|DWPP_Pricing_Plans The one true DWPP_Pricing_Plans
         * @uses DWPP_Pricing_Plans::setup_constants() Setup the constants needed.
         * @uses DWPP_Pricing_Plans::includes() Include the required files.
         * @see  DWPP_Pricing_Plans()
         * @since 1.0
         * @static
         * @static_var array $instance
         */
        public static function instance() {
            if ( ! isset( self::$instance ) && ! ( self::$instance instanceof DWPP_Pricing_Plans ) ) {
                self::$instance = new DWPP_Pricing_Plans;
                self::$instance->setup_constants();
                self::$instance->includes();
                new DWPP_Init(); // run the initial tasks
                new DWPP_Enqueuer(); // enqueue required styles and scripts
                new DWPP_Admin_Notices(); // print admin notices
                new DWPP_Settings(); // push necessary settings
                new DWPP_Views(); // views of the whole plugin
                new DWPP_WC_Controller(); // add product type and process the data
                new DWPP_Controller(); // add product type and process the data
                new DWPP_Ajax_Handler(); // add product type and process the data
                new DWPP_Post_Type_Manager(); // Post Type
                new Directorist_Direct_Purchase(); // direct purchase
                new Directorist_Subscription(); // wc subscription integration
                new Directorist_Plan_Restrictions(); // wc subscription integration

                add_filter( 'atbdp_reviewed_listing_status_controller_argument', [self::$instance, 'update_edited_listings_status'] );
                
                add_action( 'admin_init', [ self::$instance, 'update_controller' ] );

            }

            return self::$instance;
        }

        public function update_controller() {
            $data = get_user_meta( get_current_user_id(), '_plugins_available_in_subscriptions', true );
            $license_key = ! empty( $data['directorist-woocommerce-pricing-plans'] ) ? $data['directorist-woocommerce-pricing-plans']['license'] : '';
            new EDD_SL_Plugin_Updater( ATBDP_AUTHOR_URL, __FILE__, [
                'version' => DWPP_VERSION, // current version number
                'license' => $license_key, // license key (used get_option above to retrieve from DB)
                'item_id' => ATBDP_WOO_PRICING_PLAN_POST_ID, // id of this plugin
                'author'  => 'wpWax', // author of this plugin
                'url'     => home_url(),
                'beta'    => false, // set to true if you wish customers to receive update notifications of beta releases
            ] );
        }

        private function __construct() {
            /*making it private prevents constructing the object*/
        }

        // update_edited_listings_status
        public static function update_edited_listings_status( $args = [] ) {
            $plan_id = get_post_meta( get_the_ID(), '_fm_plans', true );

            if ( empty( $plan_id ) ) {
                return $args;
            }

            $regular_price = get_post_meta( $plan_id, '_regular_price', true );
            $price         = ( is_numeric( $regular_price ) ) ? (int) $regular_price : 0;

            if ( $price <= 0 ) {
                $id = get_the_ID();

                $directory_type = get_post_meta( $id, '_directory_type', true );
                if ( ! is_numeric( $directory_type ) ) {
                    $term           = get_term_by( 'slug', $directory_type, ATBDP_TYPE );
                    $directory_type = ! empty( $term ) ? $term->term_id : '';

                    update_post_meta( $id, '_directory_type', $directory_type );
                    wp_set_object_terms( $id, (int) $directory_type, ATBDP_TYPE );
                }

                $new_l_status  = get_term_meta( $directory_type, 'new_listing_status', true );
                $edit_l_status = get_term_meta( $directory_type, 'edit_listing_status', true );

                $edited = isset( $_GET['edited'] ) ? esc_attr( $_GET['edited'] ) : '';
                $status = ( $edited ) ? $edit_l_status : $new_l_status;

                $args['post_status'] = $status;
            }

            return $args;
        }

        /**
         * Throw error on object clone.
         *
         * The whole idea of the singleton design pattern is that there is a single
         * object therefore, we don't want the object to be cloned.
         *
         * @return void
         * @since 1.0
         * @access protected
         */
        public function __clone() {
            // Cloning instances of the class is forbidden.
            _doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'directorist-woocommerce-pricing-plans' ), '1.0' );
        }

        /**
         * Disable unserializing of the class.
         *
         * @return void
         * @since 1.0
         * @access protected
         */
        public function __wakeup() {
            // Unserializing instances of the class is forbidden.
            _doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'directorist-woocommerce-pricing-plans' ), '1.0' );
        }

        /**
         * It  loads a template file from the Default template directory.
         * @param string $template Name of the file that should be loaded from the template directory.
         * @param array $args Additional arguments that should be passed to the template file for rendering dynamic  data.
         */
        public function load_template( $template, $args = [] ) {
            dwpp_get_template( $template, $args );
        }

        /**
         * It Includes and requires necessary files.
         *
         * @access private
         * @return void
         * @since 1.0
         */
        private function includes() {
            require_once DWPP_INC_DIR . 'helper-functions.php';
            require_once DWPP_INC_DIR . 'class-woo-admin.php';
            require_once DWPP_INC_DIR . 'validator.php';
            require_once DWPP_INC_DIR . 'init.php';
            require_once DWPP_VIEWS_DIR . 'html-views.php';
            require_once DWPP_CLASSES_DIR . 'class-enqueuer.php';
            require_once DWPP_CLASSES_DIR . 'class-admin-notices.php';
            require_once DWPP_CLASSES_DIR . 'class-settings.php';
            require_once DWPP_CLASSES_DIR . 'class-direct-purchase.php';
            require_once DWPP_CLASSES_DIR . 'class-wc-controller.php';
            require_once DWPP_CLASSES_DIR . 'class-data-controller.php';
            require_once DWPP_CLASSES_DIR . 'class-ajax-handler.php';
            require_once DWPP_CLASSES_DIR . 'class-listing_type_manager.php';
            require_once DWPP_CLASSES_DIR . 'class-subscription.php';
            require_once DWPP_CLASSES_DIR . 'class-restrictions.php';

            // setup the updater
            if ( ! class_exists( 'EDD_SL_Plugin_Updater' ) ) {
                // load our custom updater if it doesn't already exist
                include dirname( __FILE__ ) . '/inc/EDD_SL_Plugin_Updater.php';
            }
        }

        /**
         * Setup plugin constants.
         *
         * @access private
         * @return void
         * @since 1.0
         */
        private function setup_constants() {
            if ( ! defined( 'DWPP_FILE' ) ) {define( 'DWPP_FILE', __FILE__ );}
            require_once plugin_dir_path( __FILE__ ) . '/const-helper.php'; // loads constant from a file so that it can be available on all files.
            require_once plugin_dir_path( __FILE__ ) . '/config.php'; // loads constant from a file so that it can be available on all files.
        }
    }

    if ( ! function_exists( 'directorist_is_plugin_active' ) ) {
        function directorist_is_plugin_active( $plugin ) {
            return in_array( $plugin, (array) get_option( 'active_plugins', [] ), true ) || directorist_is_plugin_active_for_network( $plugin );
        }
    }

    if ( ! function_exists( 'directorist_is_plugin_active_for_network' ) ) {
        function directorist_is_plugin_active_for_network( $plugin ) {
            if ( ! is_multisite() ) {
                return false;
            }

            $plugins = get_site_option( 'active_sitewide_plugins' );
            if ( isset( $plugins[$plugin] ) ) {
                return true;
            }

            return false;
        }
    }

/**
 * The main function for that returns DWPP_Pricing_Plans
 *
 * The main function responsible for returning the one true DWPP_Pricing_Plans
 * Instance to functions everywhere.
 *
 * Use this function like you would a global variable, except without needing
 * to declare the global.
 *
 *
 * @return object|DWPP_Pricing_Plans The one true DWPP_Pricing_Plans Instance.
 * @since 1.0
 */
    function DWPP_Pricing_Plans() {
        return DWPP_Pricing_Plans::instance();
    }

    $is_active_directorist   = directorist_is_plugin_active( 'directorist/directorist-base.php' );
    $is_active_pricing_plans = directorist_is_plugin_active( 'directorist-pricing-plans/directorist-pricing-plans.php' );

    if ( $is_active_directorist && ! $is_active_pricing_plans ) {
        DWPP_Pricing_Plans(); // get the plugin running
    }
/**
 * Register a custom WooCommerce product type.
 *
 * @since     1.0.0
 * @access    public
 */
    function dwpp_register_woocommerce_product_type() {

        /**
         * DWPP_Product_Listings_Package Class
         *
         * @since    1.0.0
         * @access   public
         */
        if ( class_exists( 'WooCommerce' ) ) {
            class DWPP_Product_Listings_Package extends WC_Product {
                protected $product_type;
                public function __construct( $product ) {

                    $this->product_type = 'listing_pricing_plans';
                    parent::__construct( $product );
                }
            }
        }
    }
    add_action( 'plugins_loaded', 'dwpp_register_woocommerce_product_type' );
}