<?php

namespace UltimatePostKit\Modules\AmoxGrid\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Typography;
use UltimatePostKit\Includes\Controls\GroupQuery\Group_Control_Query;
use UltimatePostKit\Traits\Global_Widget_Controls;
use UltimatePostKit\Traits\Global_Widget_Functions;
use WP_Query;

if (!defined('ABSPATH')) {
    exit;
}
// Exit if accessed directly

class Amox_Grid extends Group_Control_Query
{

    use Global_Widget_Controls;
    use Global_Widget_Functions;

    private $_query = null;

    public function get_name()
    {
        return 'upk-amox-grid';
    }

    public function get_title()
    {
        return BDTUPK . esc_html__('Amox Grid', 'ultimate-post-kit');
    }

    public function get_icon()
    {
        return 'upk-widget-icon upk-icon-amox-grid';
    }

    public function get_categories()
    {
        return ['ultimate-post-kit'];
    }

    public function get_keywords()
    {
        return ['post', 'grid', 'blog', 'recent', 'news', 'amox'];
    }

    public function get_style_depends()
    {
        if ($this->upk_is_edit_mode()) {
            return ['upk-all-styles'];
        } else {
            return ['upk-font', 'upk-amox-grid'];
        }
    }

    public function get_script_depends()
    {
        if ($this->upk_is_edit_mode()) {
            return ['upk-all-scripts'];
        } else {
            return ['upk-ajax-loadmore'];
        }
    }

    public function get_custom_help_url()
    {
        return 'https://youtu.be/BeJ77OLErAk';
    }

    public function get_query()
    {
        return $this->_query;
    }

    public function has_widget_inner_wrapper(): bool {
        return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
    }

    protected function register_controls() {
        $this->start_controls_section(
            'section_content_layout',
            [
                'label' => esc_html__('Layout', 'ultimate-post-kit'),
            ]
        );

        $this->add_responsive_control(
            'columns',
            [
                'label' => __('Columns', 'ultimate-post-kit'),
                'type' => Controls_Manager::SELECT,
                'default' => '3',
                'tablet_default' => '2',
                'mobile_default' => '1',
                'options' => [
                    '1' => '1',
                    '2' => '2',
                    '3' => '3',
                    '4' => '4',
                    '5' => '5',
                    '6' => '6',
                ],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid' => 'grid-template-columns: repeat({{SIZE}}, 1fr);',
                ],
            ]
        );

        $this->add_responsive_control(
            'row_gap',
            [
                'label' => esc_html__('Row Gap', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid' => 'grid-row-gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'column_gap',
            [
                'label' => esc_html__('Column Gap', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_responsive_control(
			'image_height',
			[
				'label'     => esc_html__('Image Height', 'ultimate-post-kit'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 100,
						'max' => 800,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-amox-grid .upk-img-wrap' => 'height: {{SIZE}}px;',
				],
			]
		);

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'primary_thumbnail',
                'exclude' => ['custom'],
                'default' => 'medium',
            ]
        );

        $this->add_responsive_control(
            'content_alignment',
            [
                'label' => __('Alignment', 'ultimate-post-kit'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'ultimate-post-kit'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'ultimate-post-kit'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'ultimate-post-kit'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-item .upk-content' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'active_item',
            [
                'label' => __('Active Item', 'ultimate-post-kit') . BDTUPK_PC,
                'type' => Controls_Manager::NUMBER,
                'default' => 2,
                'description' => __('Be more creative with your design by typing in your item number.', 'ultimate-post-kit'),
                'separator' => 'before',
                'classes' => BDTUPK_IS_PC,
            ]
        );

        $this->end_controls_section();

        //New Query Builder Settings
        $this->start_controls_section(
            'section_post_query_builder',
            [
                'label' => __('Query', 'ultimate-post-kit'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'item_limit',
            [
                'label' => esc_html__('Item Limit', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 21,
                    ],
                ],
                'default' => [
                    'size' => 6,
                ],
            ]
        );

        $this->register_query_builder_controls();

        $this->end_controls_section();

        $this->start_controls_section(
            'section_content_additional',
            [
                'label' => esc_html__('Additional Options', 'ultimate-post-kit'),
            ]
        );

        //Global Title Controls
        $this->register_title_controls();

        $this->add_control(
            'show_category',
            [
                'label' => esc_html__('Category', 'ultimate-post-kit'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        //Global Date Controls
        $this->register_date_controls();

        //Global Reading Time Controls
        $this->register_reading_time_controls();

        $this->add_control(
            'meta_separator',
            [
                'label' => esc_html__('Separator', 'ultimate-post-kit'),
                'type' => Controls_Manager::TEXT,
                'default' => '.',
                'label_block' => false,
            ]
        );

        $this->add_control(
            'show_comments',
            [
                'label' => esc_html__('Show Comments', 'ultimate-post-kit'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'show_pagination',
            [
                'label' => esc_html__('Pagination', 'ultimate-post-kit'),
                'type' => Controls_Manager::SWITCHER,
                'separator' => 'before',
            ]
        );

        //Global Ajax Controls
        $this->register_ajax_loadmore_controls();

        $this->add_control(
            'global_link',
            [
                'label' => esc_html__('Item Wrapper Link', 'ultimate-post-kit'),
                'type' => Controls_Manager::SWITCHER,
                'prefix_class' => 'upk-global-link-',
                'description' => esc_html__('Be aware! When Item Wrapper Link activated then title link and read more link will not work', 'ultimate-post-kit'),
            ]
        );

        $this->end_controls_section();

        //Style
        $this->start_controls_section(
            'upk_section_style',
            [
                'label' => esc_html__('Items', 'ultimate-post-kit'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'content_padding',
            [
                'label' => esc_html__('Content Padding', 'ultimate-post-kit'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_item_style');

        $this->start_controls_tab(
            'tab_item_normal',
            [
                'label' => esc_html__('Normal', 'ultimate-post-kit'),
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'content_background',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-item',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'item_border',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-item',
            ]
        );

        $this->add_responsive_control(
            'item_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ultimate-post-kit'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => esc_html__('Padding', 'ultimate-post-kit'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_box_shadow',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-item',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_item_hover',
            [
                'label' => esc_html__('Hover', 'ultimate-post-kit'),
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'content_hover_background',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-item:hover',
            ]
        );

        $this->add_control(
            'item_hover_border_color',
            [
                'label' => esc_html__('Border Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'item_border_border!' => '',
                ],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-item:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_hover_box_shadow',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-item:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_title',
            [
                'label' => esc_html__('Title', 'ultimate-post-kit'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_title' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'title_style',
            [
                'label' => esc_html__('Style', 'ultimate-post-kit'),
                'type' => Controls_Manager::SELECT,
                'default' => 'underline',
                'options' => [
                    '' => esc_html__('Default', 'ultimate-post-kit'),
                    'underline' => esc_html__('Underline', 'ultimate-post-kit'),
                    'middle-underline' => esc_html__('Middle Underline', 'ultimate-post-kit'),
                    'overline' => esc_html__('Overline', 'ultimate-post-kit'),
                    'middle-overline' => esc_html__('Middle Overline', 'ultimate-post-kit'),
                ],
            ]
        );

        $this->add_responsive_control(
            'title_spacing',
            [
                'label' => esc_html__('Spacing', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-title' => 'padding-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'label' => esc_html__('Typography', 'ultimate-post-kit'),
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-title',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'title_text_shadow',
                'label' => __('Text Shadow', 'ultimate-post-kit'),
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-title',
            ]
        );

        $this->start_controls_tabs('tabs_title_style');

        $this->start_controls_tab(
            'tab_title_normal',
            [
                'label' => esc_html__('Normal', 'ultimate-post-kit'),
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_title_hover',
            [
                'label' => esc_html__('Hover', 'ultimate-post-kit'),
            ]
        );

        $this->add_control(
            'title_color_hover',
            [
                'label' => esc_html__('Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-title a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_meta',
            [
                'label' => esc_html__('Meta', 'ultimate-post-kit'),
                'tab' => Controls_Manager::TAB_STYLE,
                'conditions' => [
                    'relation' => 'or',
                    'terms' => [
                        [
                            'name' => 'show_date',
                            'value' => 'yes',
                        ],
                        [
                            'name' => 'show_comments',
                            'value' => 'yes',
                        ],
                    ],
                ],
            ]
        );

        $this->add_control(
            'meta_color',
            [
                'label' => esc_html__('Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-meta' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'meta_space_between',
            [
                'label' => esc_html__('Space Between', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-meta > div:before' => 'margin: 0 {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'meta_typography',
                'label' => esc_html__('Typography', 'ultimate-post-kit'),
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-meta',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_category',
            [
                'label' => esc_html__('Category', 'ultimate-post-kit'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_category' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'category_bottom_spacing',
            [
                'label' => esc_html__('Spacing', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-category' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_category_style');

        $this->start_controls_tab(
            'tab_category_normal',
            [
                'label' => esc_html__('Normal', 'ultimate-post-kit'),
            ]
        );

        $this->add_control(
            'category_color',
            [
                'label' => esc_html__('Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-category a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'category_background',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-category a',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'category_border',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-category a',
            ]
        );

        $this->add_responsive_control(
            'category_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ultimate-post-kit'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-category a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'category_padding',
            [
                'label' => esc_html__('Padding', 'ultimate-post-kit'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-category a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'category_spacing',
            [
                'label' => esc_html__('Space Between', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 2,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-category' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'category_shadow',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-category a',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'category_typography',
                'label' => esc_html__('Typography', 'ultimate-post-kit'),
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-category a',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_category_hover',
            [
                'label' => esc_html__('Hover', 'ultimate-post-kit'),
            ]
        );

        $this->add_control(
            'category_hover_color',
            [
                'label' => esc_html__('Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-category a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'category_hover_background',
                'selector' => '{{WRAPPER}} .upk-amox-grid .upk-category a:hover',
            ]
        );

        $this->add_control(
            'category_hover_border_color',
            [
                'label' => esc_html__('Border Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'category_border_border!' => '',
                ],
                'selectors' => [
                    '{{WRAPPER}} .upk-amox-grid .upk-category a:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        if (_is_upk_pro_activated()) {
            $this->start_controls_tab(
                'tab_category_active',
                [
                    'label' => esc_html__('Active', 'ultimate-post-kit'),
                ]
            );

            $this->add_control(
                'category_active_normal_heading',
                [
                    'label' => esc_html__('NORMAL', 'ultimate-post-kit'),
                    'type' => Controls_Manager::HEADING,
                ]
            );

            $this->end_controls_tab();
        }

        $this->end_controls_tabs();

        $this->end_controls_section();

        //Pagination
        $this->start_controls_section(
            'section_style_pagination',
            [
                'label' => esc_html__('Pagination', 'ultimate-post-kit'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_pagination' => 'yes',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_pagination_style');

        $this->start_controls_tab(
            'tab_pagination_normal',
            [
                'label' => esc_html__('Normal', 'ultimate-post-kit'),
            ]
        );

        $this->add_control(
            'pagination_color',
            [
                'label' => esc_html__('Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} ul.upk-pagination li a, {{WRAPPER}} ul.upk-pagination li span' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'pagination_background',
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} ul.upk-pagination li a',
                'separator' => 'after',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'pagination_border',
                'label' => esc_html__('Border', 'ultimate-post-kit'),
                'selector' => '{{WRAPPER}} ul.upk-pagination li a',
            ]
        );

        $this->add_responsive_control(
            'pagination_offset',
            [
                'label' => esc_html__('Offset', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .upk-pagination' => 'margin-top: {{SIZE}}px;',
                ],
            ]
        );

        $this->add_responsive_control(
            'pagination_space',
            [
                'label' => esc_html__('Spacing', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .upk-pagination' => 'margin-left: {{SIZE}}px;',
                    '{{WRAPPER}} .upk-pagination > *' => 'padding-left: {{SIZE}}px;',
                ],
            ]
        );

        $this->add_responsive_control(
            'pagination_padding',
            [
                'label' => esc_html__('Padding', 'ultimate-post-kit'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} ul.upk-pagination li a' => 'padding: {{TOP}}px {{RIGHT}}px {{BOTTOM}}px {{LEFT}}px;',
                ],
            ]
        );

        $this->add_responsive_control(
            'pagination_radius',
            [
                'label' => esc_html__('Radius', 'ultimate-post-kit'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} ul.upk-pagination li a' => 'border-radius: {{TOP}}px {{RIGHT}}px {{BOTTOM}}px {{LEFT}}px;',
                ],
            ]
        );

        $this->add_responsive_control(
            'pagination_arrow_size',
            [
                'label' => esc_html__('Arrow Size', 'ultimate-post-kit'),
                'type' => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} ul.upk-pagination li a svg' => 'height: {{SIZE}}px; width: auto;',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'pagination_typography',
                'label' => esc_html__('Typography', 'ultimate-post-kit'),
                //'scheme'   => Schemes\Typography::TYPOGRAPHY_4,
                'selector' => '{{WRAPPER}} ul.upk-pagination li a, {{WRAPPER}} ul.upk-pagination li span',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_pagination_hover',
            [
                'label' => esc_html__('Hover', 'ultimate-post-kit'),
            ]
        );

        $this->add_control(
            'pagination_hover_color',
            [
                'label' => esc_html__('Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} ul.upk-pagination li a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'pagination_hover_border_color',
            [
                'label' => esc_html__('Border Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} ul.upk-pagination li a:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'pagination_hover_background',
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} ul.upk-pagination li a:hover',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_pagination_active',
            [
                'label' => esc_html__('Active', 'ultimate-post-kit'),
            ]
        );

        $this->add_control(
            'pagination_active_color',
            [
                'label' => esc_html__('Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} ul.upk-pagination li.upk-active a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'pagination_active_border_color',
            [
                'label' => esc_html__('Border Color', 'ultimate-post-kit'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} ul.upk-pagination li.upk-active a' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'pagination_active_background',
                'selector' => '{{WRAPPER}} ul.upk-pagination li.upk-active a',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        //Global Ajax Loadmore Style Controls
        $this->register_ajax_loadmore_style_controls();
    }

    /**
     * Main query render for this widget
     * @param $posts_per_page number item query limit
     */
    public function query_posts($posts_per_page)
    {
        $default = $this->getGroupControlQueryArgs();
		$args = [];
        if ($posts_per_page) {
            $args['posts_per_page'] = $posts_per_page;
            $args['paged'] = max(1, get_query_var('paged'), get_query_var('page'));
        }
        $args = array_merge($default, $args);
        $this->_query = new WP_Query($args);
    }

    public function render_comments($id = 0)
    {

        if (!$this->get_settings('show_comments')) {
            return;
        }
        ?>

		<div class="upk-comments upk-flex upk-flex-middle">
			<i class="upk-icon-post-comments"></i>
			<span><?php echo get_comments_number($id) ?></span>
		</div>

	<?php
}

    public function render_post_grid_item($post_id, $image_size, $active_item)
    {
        $settings = $this->get_settings_for_display();

        if ('yes' == $settings['global_link']) {

            $this->add_render_attribute('grid-item', 'onclick', "window.open('" . esc_url(get_permalink()) . "', '_self')", true);
        }

        $this->add_render_attribute('grid-item', 'class', 'upk-item ' . $active_item, true);

        ?>
		<div <?php $this->print_render_attribute_string('grid-item');?>>
			<div class="upk-img-wrap">
				<?php $this->render_image(get_post_thumbnail_id($post_id), $image_size);?>
			</div>
			<div class="upk-content">
				<?php $this->render_category();?>
				<?php $this->render_title(substr($this->get_name(), 4));?>

				<?php if ($settings['show_comments'] or $settings['show_date'] or $settings['show_reading_time']): ?>
				<div class="upk-meta upk-flex-inline upk-flex-middle">
					<?php $this->render_date();?>
					<?php if (_is_upk_pro_activated()):
            if ('yes' === $settings['show_reading_time']): ?>
                    <div class="upk-reading-time" data-separator="<?php echo esc_html($settings['meta_separator']); ?>">
                        <?php echo esc_html( ultimate_post_kit_reading_time( get_the_content(), $settings['avg_reading_speed'], $settings['hide_seconds'] ?? 'no', $settings['hide_minutes'] ?? 'no' ) );?>
                    </div>
                    <?php endif;?>
					<?php endif;?>
					<div data-separator="<?php echo esc_html($settings['meta_separator']); ?>">
						<?php $this->render_comments($post_id);?>
					</div>
				</div>
				<?php endif;?>
			</div>
		</div>


	<?php
}

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $this->query_posts($settings['item_limit']['size']);
        $wp_query = $this->get_query();

        if (!$wp_query->found_posts) {
            return;
        }

        $this->add_render_attribute('grid-wrap', 'class', 'upk-amox-grid upk-ajax-grid-wrap');
        $this->add_render_attribute(
            [
                'upk-amox-grid-container' => [
                    'class'         => 'upk-amox-grid-container upk-ajax-grid',
                    'data-loadmore' => [
                        wp_json_encode(
                            array_filter([
                                'loadmore_enable'   => $settings['ajax_loadmore_enable'],
                                'loadmore_btn'      => $settings['ajax_loadmore_btn'],
                                'infinite_scroll'   => $settings['ajax_loadmore_infinite_scroll'],
                            ])
                        ),
                    ],
                ],
            ]
        );        

        if ( $settings['ajax_loadmore_enable'] == 'yes' ) {
            $ajax_settings = [
                'posts_source'                   => isset( $settings['posts_source'] ) ? $settings['posts_source'] : 'post',
                'posts_per_page'                 => isset( $posts_load ) ? $posts_load : 6,
                'ajax_item_load'                 => isset( $settings['ajax_loadmore_items'] ) ? $settings['ajax_loadmore_items'] : 3,
                'posts_selected_ids'             => isset( $settings['posts_selected_ids'] ) ? $settings['posts_selected_ids'] : '',
                'posts_include_by'               => isset( $settings['posts_include_by'] ) ? $settings['posts_include_by'] : [],
                'posts_include_author_ids'       => isset( $settings['posts_include_author_ids'] ) ? $settings['posts_include_author_ids'] : '',
                'posts_include_term_ids'         => isset( $settings['posts_include_term_ids'] ) ? $settings['posts_include_term_ids'] : '',
                'posts_exclude_by'               => isset( $settings['posts_exclude_by'] ) ? $settings['posts_exclude_by'] : [],
                'posts_exclude_ids'              => isset( $settings['posts_exclude_ids'] ) ? $settings['posts_exclude_ids'] : '',
                'posts_exclude_author_ids'       => isset( $settings['posts_exclude_author_ids'] ) ? $settings['posts_exclude_author_ids'] : '',
                'posts_exclude_term_ids'         => isset( $settings['posts_exclude_term_ids'] ) ? $settings['posts_exclude_term_ids'] : '',
                'posts_offset'                   => isset( $settings['posts_offset'] ) ? $settings['posts_offset'] : 0,
                'posts_select_date'              => isset( $settings['posts_select_date'] ) ? $settings['posts_select_date'] : '',
                'posts_date_before'              => isset( $settings['posts_date_before'] ) ? $settings['posts_date_before'] : '',
                'posts_date_after'               => isset( $settings['posts_date_after'] ) ? $settings['posts_date_after'] : '',
                'posts_orderby'                  => isset( $settings['posts_orderby'] ) ? $settings['posts_orderby'] : 'date',
                'posts_order'                    => isset( $settings['posts_order'] ) ? $settings['posts_order'] : 'DESC',
                'posts_ignore_sticky_posts'      => isset( $settings['posts_ignore_sticky_posts'] ) ? $settings['posts_ignore_sticky_posts'] : 'no',
                'posts_only_with_featured_image' => isset( $settings['posts_only_with_featured_image'] ) ? $settings['posts_only_with_featured_image'] : 'no',
                // Grid Settings
                'show_title'                     => isset( $settings['show_title'] ) ? $settings['show_title'] : 'yes',
                'title_tags'                     => isset( $settings['title_tags'] ) ? $settings['title_tags'] : 'h3',
                'show_date'                      => isset( $settings['show_date'] ) ? $settings['show_date'] : 'yes',
                'show_time'                      => isset( $settings['show_time'] ) ? $settings['show_time'] : 'no',
                'show_category'                  => isset( $settings['show_category'] ) ? $settings['show_category'] : 'yes',
                'show_comments'                  => isset( $settings['show_comments'] ) ? $settings['show_comments'] : 'yes',
                'show_reading_time'              => isset( $settings['show_reading_time'] ) ? $settings['show_reading_time'] : 'no',
                'avg_reading_speed'              => isset( $settings['avg_reading_speed'] ) ? $settings['avg_reading_speed'] : 200,
                'hide_seconds'                   => isset( $settings['hide_seconds'] ) ? $settings['hide_seconds'] : 'no',
                'hide_minutes'                   => isset( $settings['hide_minutes'] ) ? $settings['hide_minutes'] : 'no',
                'upk_link_new_tab'               => isset( $settings['upk_link_new_tab'] ) ? $settings['upk_link_new_tab'] : 'no',
                'meta_separator'                 => isset( $settings['meta_separator'] ) ? $settings['meta_separator'] : '.',
                'human_diff_time'                => isset( $settings['human_diff_time'] ) ? $settings['human_diff_time'] : 'no',
                'human_diff_time_short'          => isset( $settings['human_diff_time_short'] ) ? $settings['human_diff_time_short'] : 'no',
                'title_style'                    => isset( $settings['title_style'] ) ? $settings['title_style'] : 'underline',
                'global_link'                    => isset( $settings['global_link'] ) ? $settings['global_link'] : 'no',
            ];
        
            $this->add_render_attribute(
                [
                    'upk-amox-grid-container' => [
                        'data-settings' => [
                            wp_json_encode( $ajax_settings ),
                        ],
                    ],
                ]
            );
        }

        if (isset($settings['upk_in_animation_show']) && ($settings['upk_in_animation_show'] == 'yes')) {
            $this->add_render_attribute('grid-wrap', 'class', 'upk-in-animation');
            if (isset($settings['upk_in_animation_delay']['size'])) {
                $this->add_render_attribute('grid-wrap', 'data-in-animation-delay', $settings['upk_in_animation_delay']['size']);
            }
        }

        ?>
		<div <?php $this->print_render_attribute_string('upk-amox-grid-container');?>>
			<div <?php $this->print_render_attribute_string('grid-wrap');?>>
				<?php
				$i = 0;
				while ($wp_query->have_posts()):
					$wp_query->the_post();
					$thumbnail_size = $settings['primary_thumbnail_size'];

					$i++;
					$active_item = '';
					if (_is_upk_pro_activated()) {
						$active_item = apply_filters('amox_grid_active_item', $this, $i);
					}

					?>
					<?php $this->render_post_grid_item(get_the_ID(), $thumbnail_size, $active_item);?>
				<?php endwhile;?>
			</div>
		</div>

		<?php $this->render_ajax_loadmore(); ?>

		<?php

        if ($settings['show_pagination']) {?>
			<div class="ep-pagination">
				<?php ultimate_post_kit_post_pagination($wp_query, $this->get_id());?>
			</div>
<?php
}
        wp_reset_postdata();
    }
}
