<?php
/**
 * Database update and migration functions.
 * @author  wpWax
 * @since   1.0
 * @version 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Migration_V8 {
	protected static $instance = null;

	private function __construct() {
		add_action( 'init', [$this, 'all_listing_layout_migrate'] );
		add_action( 'init', [$this, 'migrate_single_listing_header'] );
		add_filter( 'directorist_single_listing_header_migration_data', [$this, 'listing_header_migration_data'], 10, 2 );
	}

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function listing_header_migration_data( $new_listing_data = [], $old_listing_data = [] ) {

        $title_options = $old_listing_data['options']['content_settings']['listing_title'];

		foreach ( $new_listing_data as $key => $value ) {

			if ( isset( $value['placeholderKey'] ) && ( 'listing-title-placeholder' === $value['placeholderKey'] ) ) {

				// Define the options to migrate
				$options = [
					'enable_title', 'enable_tagline', 'enable_gender', 'enable_age', 'enable_reviews', 'enable_claim_listing', 'enable_consultation', 'enable_consultation_text'
				];

				// Create an array to hold the updated options
				$updated_title_options = [];
				foreach ( $options as $option ) {
					$updated_title_options[$option] = $title_options[$option] ?? '';
				}

				// Merge existing widget data with new title options
				$new_listing_data[$key]['selectedWidgets'][0]['options']['fields'] = array_merge(
					$new_listing_data[$key]['selectedWidgets'][0]['options']['fields'],
					$updated_title_options
				);
			}
		}

		// Reorganize the data structure as required
		$new_structure = [
			$new_listing_data[0], // actions
			$new_listing_data[2], // title
			$new_listing_data[3], // info
			$new_listing_data[1], // slider
		];
    
		// Mark the migration as completed
		update_option( 'v8_single_listing_header_migration', true );
        
		return $new_structure;
	}

	public static function migrate_single_listing_header() {

		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if migration is already completed
		if ( get_option( 'v8_single_listing_header_migration', false ) && get_option( 'directorist_builder_header_migrated', false ) ) {
			return;
		}

		$backup_data_json  = get_option( 'directorist_builder_backup_data' );
		$backup_data_array = json_decode( $backup_data_json, true );

		if ( empty( $backup_data_array ) ) {
			return;
		}

		// Define the list of options to migrate
		$options = [
            'enable_title', 'enable_tagline', 'enable_gender', 'enable_age', 'enable_reviews', 'enable_claim_listing', 'enable_consultation', 'enable_consultation_text'
        ];

		// Process each directory type in the backup data
		foreach ( $backup_data_array as $directory_type => $data ) {

			$current_listing_header_data = get_term_meta( $directory_type, 'single_listing_header', true );
			
			$title_options = $data['single_listing_header']['options']['content_settings']['listing_title'] ?? [];

			// Build the array of title options from the backup data
			$backup_title_options = [];
			foreach ( $options as $option ) {
				$backup_title_options[$option] = $title_options[$option] ?? '';
			}

			// Update listing header data where placeholderKey matches
			if ( is_array( $current_listing_header_data ) ) {
				foreach ( $current_listing_header_data as $key => $value ) {
					if ( isset( $value['placeholderKey'] ) && ( 'listing-title-placeholder' === $value['placeholderKey'] ) ) {
						if( ! isset( $value['selectedWidgets'][0]['options'] ) ) {
							continue;
						}
						// Merge backup options with the current fields
						$current_fields = isset( $value['selectedWidgets'][0]['options'] ) ? $value['selectedWidgets'][0]['options']['fields'] : $value['selectedWidgets'][0];

						$current_listing_header_data[$key]['selectedWidgets'][0]['options']['fields'] = array_merge( $current_fields, $backup_title_options );
					}
				}
			}

			// Reorder the data structure as required
			$reordered_listing_header_data = [
				$current_listing_header_data[0], // actions
				$current_listing_header_data[2], // title
				$current_listing_header_data[3], // info
				$current_listing_header_data[1], // slider
			];

			// Update the term meta with the new structure
			update_term_meta( $directory_type, 'single_listing_header', $reordered_listing_header_data );
		}

		// Mark the migration as completed
		update_option( 'v8_single_listing_header_migration', true );
	}

	public function all_listing_layout_migrate() {

		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		if ( empty( get_option( 'directorist_builder_header_migrated', false ) ) ) {
			return;
		}

		if ( get_option( 'v8_all_listing_layout_migration', false ) ) {
			return;
		}

		$directory_types = get_terms( [
			'taxonomy'   => ATBDP_DIRECTORY_TYPE,
			'hide_empty' => false,
		] );

		foreach ( $directory_types as $directory_type ) {
			update_term_meta( $directory_type->term_id, 'all_listing_layout', 'left_sidebar' );
		}

		update_option( 'v8_all_listing_layout_migration', true );

	}
}

Migration_V8::instance();